package com.vungle.warren;

import android.content.Context;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

import com.vungle.warren.AdConfig.AdSize;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Placement;
import com.vungle.warren.persistence.FutureResult;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.utility.Executors;
import com.vungle.warren.utility.TimeoutProvider;

import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

import static com.vungle.warren.error.VungleException.AD_UNABLE_TO_PLAY;
import static com.vungle.warren.error.VungleException.ExceptionCode;
import static com.vungle.warren.error.VungleException.INCORRECT_BANNER_API_USAGE;
import static com.vungle.warren.error.VungleException.INVALID_SIZE;
import static com.vungle.warren.error.VungleException.PLACEMENT_NOT_FOUND;
import static com.vungle.warren.error.VungleException.VUNGLE_NOT_INTIALIZED;

/**
 * Banners API for loading banner ads and retrieving VungleBanner view instances
 */
public final class Banners {

    private static final String TAG = Banners.class.getSimpleName();
    /**
     * Check if we can play an advertisement for the given placement. This method checks out file
     * system to see if there are asset files for the given placement and adSize and returns true if we have
     * assets stored which have not expired.
     *
     * @param id     The placement identifier.
     * @param adSize The placement adSize
     * @return true if an advertisement can be played immediately, false otherwise.
     */
    public static boolean canPlayAd(@NonNull final String id, final @NonNull AdSize adSize) {
        if (!AdSize.isBannerAdSize(adSize)) {
            Log.e(TAG, "Invalid Ad Size. Cannot check loaded status of non banner size placements.");
            return false;
        }

        final Context context = Vungle.appContext();

        if (context == null) {
            Log.e(TAG, "Context is null");
            return false;
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
        Executors sdkExecutors = serviceLocator.getService(Executors.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        FutureResult futureResult = new FutureResult<>(sdkExecutors.getApiExecutor()
                .submit(new Callable<Boolean>() {
                    @Override
                    public Boolean call() {

                        if (!Vungle.isInitialized()) {
                            Log.e(TAG, "Vungle is not initialized");
                            return false;
                        }

                        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
                        Repository repository = serviceLocator.getService(Repository.class);
                        Advertisement advertisement = repository
                                .findValidAdvertisementForPlacement(id).get();

                        Placement placement = repository.load(id, Placement.class).get();

                        if (placement == null)
                            return false;

                        AdSize adSizeOfLoadedAd = placement.getAdSize();

                        if (adSize != adSizeOfLoadedAd) {
                            return false;
                        }

                        if (advertisement == null || !advertisement.getAdConfig()
                                .getAdSize().equals(adSize)) {
                            return false;
                        }

                        return Vungle.canPlayAd(advertisement);
                    }
                })
        );

        return Boolean.TRUE.equals(futureResult.get(provider.getTimeout(), TimeUnit.MILLISECONDS));
    }

    /**
     * Retrieve a VungleBanner view that displays a banner ad if an ad is loaded for that placement.
     * DO NOT MODIFY LayoutParams.
     *
     * Usage Example:
     * VungleBanner vungleBanner = Banners.getBanner(.....)
     * parentView.addView(vungleBanner)
     *
     * @param placementId   The placement identifier
     * @param adSize        Ad Size
     * @param playAdCallback    Optional callback for ad state
     * @return VungleBanner View after calling load ad successfully.
     */
    @Nullable
    public static VungleBanner getBanner(@NonNull String placementId,
                                         @NonNull AdSize adSize,
                                         @Nullable PlayAdCallback playAdCallback) {
        Context appContext = Vungle.appContext();
        //AdSize is being requested in both loadAd and getBanner.
        if (appContext == null || !Vungle.isInitialized()) {
            Log.e(TAG, "Vungle is not initialized, returned VungleNativeAd = null");
            onPlaybackError(placementId, playAdCallback, VUNGLE_NOT_INTIALIZED);
            return null;
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(appContext);
        final Repository repository = serviceLocator.getService(Repository.class);

        final VungleSettings settings = ServiceLocator.getInstance(appContext)
                .getService(RuntimeValues.class)
                .settings
                .get();

        if (TextUtils.isEmpty(placementId)) {
            onPlaybackError(placementId, playAdCallback, PLACEMENT_NOT_FOUND);
            return null;
        }

        Placement placement = repository.load(placementId, Placement.class).get();
        if (placement == null) {
            onPlaybackError(placementId, playAdCallback, PLACEMENT_NOT_FOUND);
            return null;
        }

        if (!AdSize.isBannerAdSize(adSize)) {
            onPlaybackError(placementId, playAdCallback, INCORRECT_BANNER_API_USAGE);
            return null;
        }

        if (!canPlayAd(placementId, adSize)) {
            onPlaybackError(placementId, playAdCallback, AD_UNABLE_TO_PLAY);
            return null;
        }

        if (settings != null && settings.getBannerRefreshDisabled()) {
            //disable ad refresh
            return new VungleBanner(
                    appContext,
                    placementId,
                    0,
                    adSize,
                    playAdCallback);
        }

        return new VungleBanner(
                appContext,
                placementId,
                placement.getAdRefreshDuration(),
                adSize,
                playAdCallback
        );
    }

    /**
     * Request a banner ad from Vungle for a specific size.
     *
     * Banners.loadBanner("BANNER-12345", AdConfig.AdSize.BANNER, new LoadAdCallback(){... });
     *
     * @param placementId - Banner Placement Id
     * @param adSize - size of ad to be requested see {@link AdSize}
     * @param callback - optional callback for successful and unsuccessful ad load
     */
    @SuppressWarnings("squid:S2583")
    public static void loadBanner(@NonNull final String placementId,
                                  @NonNull final AdSize adSize,
                                  @Nullable final LoadAdCallback callback) {

        Context appCtx = Vungle.appContext();
        if (appCtx == null || !Vungle.isInitialized()) {
            onLoadError(placementId, callback, VUNGLE_NOT_INTIALIZED);
            return;
        }

        if (adSize == null) {
            onLoadError(placementId, callback, INVALID_SIZE);
            return;
        }

        final AdConfig adConfig = new AdConfig();
        adConfig.setAdSize(adSize);

        if (!AdSize.isBannerAdSize(adConfig.getAdSize())) {
            onLoadError(placementId, callback, INCORRECT_BANNER_API_USAGE);
        }

        Vungle.loadAdInternal(placementId, adConfig, callback);
    }

    private static void onPlaybackError(@NonNull String placementId,
                                        @Nullable PlayAdCallback playAdCallback,
                                        @ExceptionCode int code) {
        if (playAdCallback != null) {
            playAdCallback.onError(placementId, new VungleException(code));
        }
    }

    private static void onLoadError(@NonNull String id,
                                    @Nullable LoadAdCallback callback,
                                    @ExceptionCode int code) {
        if (callback != null) {
            callback.onError(id, new VungleException(code));
        }
    }

}
