package com.vungle.warren;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.AsyncTask;
import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.moat.analytics.mobile.vng.MoatFactory;
import com.moat.analytics.mobile.vng.ReactiveVideoTracker;
import com.moat.analytics.mobile.vng.ReactiveVideoTrackerPlugin;
import com.vungle.warren.analytics.AdAnalytics;
import com.vungle.warren.analytics.JobDelegateAnalytics;
import com.vungle.warren.analytics.MoatTracker;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.Placement;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.ui.CloseDelegate;
import com.vungle.warren.ui.JavascriptBridge;
import com.vungle.warren.ui.OrientationDelegate;
import com.vungle.warren.ui.contract.WebAdContract;
import com.vungle.warren.ui.presenter.LocalAdPresenter;
import com.vungle.warren.ui.presenter.MRAIDAdPresenter;
import com.vungle.warren.ui.state.OptionsState;
import com.vungle.warren.ui.view.FullAdWidget;
import com.vungle.warren.ui.view.LocalAdView;
import com.vungle.warren.ui.view.MRAIDAdView;
import com.vungle.warren.ui.view.VungleWebClient;
import com.vungle.warren.utility.ActivityManager;
import com.vungle.warren.utility.HandlerScheduler;

import java.io.File;
import java.util.concurrent.atomic.AtomicReference;

import static com.vungle.warren.error.VungleException.AD_UNABLE_TO_PLAY;
import static com.vungle.warren.error.VungleException.DB_ERROR;
import static com.vungle.warren.error.VungleException.INVALID_SIZE;
import static com.vungle.warren.error.VungleException.PLACEMENT_NOT_FOUND;
import static com.vungle.warren.error.VungleException.VUNGLE_NOT_INTIALIZED;
import static com.vungle.warren.ui.contract.AdContract.AdView;
import static com.vungle.warren.ui.contract.AdContract.AdvertisementPresenter;

/**
 * Created by zioye on 1/16/2018.
 */
public class AdvertisementPresentationFactory implements PresentationFactory {

    private static final String TAG = AdvertisementPresentationFactory.class.getSimpleName();

    private static final String EXTRA_ADVERTISEMENT = "ADV_FACTORY_ADVERTISEMENT";

    private final JobRunner jobRunner;
    private VungleApiClient apiClient;
    private BaseTask task;
    private Repository repository;
    private VungleStaticApi vungleStaticApi;
    private Advertisement currentAdvertisement;
    private final AdLoader adLoader;
    private final SessionData sessionData;

    AdvertisementPresentationFactory(@NonNull AdLoader adLoader,
                                            @NonNull VungleStaticApi vungleStaticApi,
                                            @NonNull Repository repository,
                                            @NonNull VungleApiClient vungleApiClient,
                                            @NonNull JobRunner jobRunner,
                                            @NonNull RuntimeValues runtimeValues) {
        this.vungleStaticApi = vungleStaticApi;
        this.repository = repository;
        this.apiClient = vungleApiClient;
        this.jobRunner = jobRunner;
        this.adLoader = adLoader;
        this.sessionData = runtimeValues.sessionData.get();
    }

    /**
     * Factory method for creating an {@link AdvertisementPresenter}. Using the placement ID and the
     * persistor and designer objects, it will create the {@link Advertisement} from the data on disk
     * and determine which renderer is most appropriate for this placement.
     *
     * @param context
     * @param fullAdWidget
     * @return An {@link AdvertisementPresenter} appropriate for the type of currentAdvertisement being played for this placement.
     * @throws VungleException In case the placement identifier is invalid, or no presenter for the currentAdvertisement has been designated.
     */
    @Override
    public void getFullScreenPresentation(@NonNull final Context context,
                                          @NonNull final String placementId,
                                          @NonNull final FullAdWidget fullAdWidget,
                                          @Nullable final OptionsState optionsState,
                                          @NonNull final CloseDelegate closeDelegate,
                                          @NonNull final OrientationDelegate orientationDelegate,
                                          @Nullable Bundle savedState,
                                          @NonNull final FullScreenCallback fullscreenCallback) {
        /// Extract placement and currentAdvertisement information, validate current SDK state.

        cancelTask();
        task = new FullScreenPresentationTask(
                context,
                adLoader,
                placementId,
                repository,
                vungleStaticApi,
                jobRunner,
                apiClient,
                sessionData,
                fullAdWidget,
                optionsState,
                orientationDelegate,
                closeDelegate,
                fullscreenCallback,
                onModelLoadListener,
                savedState
        );

        task.execute();
    }

    private void cancelTask() {
        if (task != null) {
            task.cancel(true);
            task.clear();
        }
    }

    /**
     * Factory method for creating an {@link AdvertisementPresenter}. Using the placement ID and the
     * persistor and designer objects, it will create the {@link Advertisement} from the data on disk
     * and determine which renderer is most appropriate for this placement.
     *
     * @return An {@link AdvertisementPresenter} appropriate for the type of currentAdvertisement being played for this placement.
     * @throws VungleException In case the placement identifier is invalid, or no presenter for the currentAdvertisement has been designated.
     */
    @Override
    public void getNativeViewPresentation(@NonNull final String placementId,
                                          @Nullable final AdConfig adConfig,
                                          @NonNull final CloseDelegate closeDelegate,
                                          @NonNull final ViewCallback viewCallback) {

        cancelTask();
        task = new NativeViewPresentationTask(
                placementId,
                adConfig,
                adLoader,
                repository,
                vungleStaticApi,
                jobRunner,
                viewCallback,
                null,
                sessionData,
                onModelLoadListener
        );

        task.execute();
    }

    @Override
    public void saveState(Bundle bundle) {
        bundle.putString(EXTRA_ADVERTISEMENT, currentAdvertisement == null ? null : currentAdvertisement.getId());
    }

    @Override
    public void destroy() {
        cancelTask();
    }

    private static class PresentationResultHolder {
        private String appId;
        private AdView adView;
        private AdvertisementPresenter advertisementPresenter;
        private VungleException exception;
        private VungleWebClient webClient;
        private MoatTracker tracker;

        PresentationResultHolder(VungleException exception) {
            this.exception = exception;
        }

        PresentationResultHolder(AdView adView,
                                 AdvertisementPresenter advertisementPresenter,
                                 VungleWebClient webClient,
                                 MoatTracker tracker,
                                 String appId) {
            this.adView = adView;
            this.advertisementPresenter = advertisementPresenter;
            this.webClient = webClient;
            this.tracker = tracker;
            this.appId = appId;
        }
    }

    private static class FullScreenPresentationTask extends BaseTask {

        private final AdLoader adLoader;

        @SuppressLint("StaticFieldLeak")
        private FullAdWidget fullAdWidget;
        @SuppressLint("StaticFieldLeak")
        private Context context;

        private final String placementId;
        private final OptionsState optionsState;
        private final FullScreenCallback fullscreenCallback;
        private final Bundle savedState;
        private final JobRunner jobRunner;
        private final VungleApiClient apiClient;
        private final CloseDelegate closeDelegate;
        private final OrientationDelegate orientationDelegate;
        private final SessionData sessionData;
        private Advertisement advertisement;

        FullScreenPresentationTask(Context context,
                                   AdLoader adLoader,
                                   String placementId,
                                   Repository repository,
                                   VungleStaticApi vungleStaticApi,
                                   JobRunner jobRunner,
                                   VungleApiClient apiClient,
                                   SessionData sessionData,
                                   FullAdWidget fullAdWidget,
                                   OptionsState optionsState,
                                   OrientationDelegate orientationDelegate,
                                   CloseDelegate closeDelegate,
                                   FullScreenCallback fullscreenCallback,
                                   OnModelLoadListener onModelLoadListener,
                                   Bundle savedState) {
            super(repository, vungleStaticApi, onModelLoadListener);
            this.placementId = placementId;
            this.fullAdWidget = fullAdWidget;
            this.optionsState = optionsState;
            this.context = context;
            this.fullscreenCallback = fullscreenCallback;
            this.savedState = savedState;
            this.jobRunner = jobRunner;
            this.apiClient = apiClient;
            this.orientationDelegate = orientationDelegate;
            this.closeDelegate = closeDelegate;
            this.adLoader = adLoader;
            this.sessionData = sessionData;
        }

        @Override
        void clear() {
            super.clear();
            context = null;
            fullAdWidget = null;
        }

        @Override
        protected PresentationResultHolder doInBackground(Void... voids) {
            Pair<Advertisement, Placement> data;
            try {
                data = loadPresentationData(placementId, savedState);
            } catch (VungleException e) {
                return new PresentationResultHolder(e);
            }

            advertisement = data.first;
            Placement placement = data.second;

            if (!adLoader.canRenderAd(advertisement)) {
                Log.e(TAG, "Advertisement is null or assets are missing");
                return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }

            AdAnalytics adAnalytics = new JobDelegateAnalytics(jobRunner);

            String appId = null;
            Cookie appIdCookie = repository.load(Cookie.APP_ID, Cookie.class).get();
            if (appIdCookie != null && !TextUtils.isEmpty(appIdCookie.getString("appId"))) {
                appId = appIdCookie.getString("appId");
            }

            VungleWebClient webClient = new VungleWebClient(advertisement, placement);
            File assetDir = repository.getAdvertisementAssetDirectory(advertisement.getId()).get();
            if (assetDir == null || !assetDir.isDirectory()) {
                Log.e(TAG, "Advertisement assets dir is missing");
                return new PresentationResultHolder(new VungleException(DB_ERROR));
            }

            switch (advertisement.getAdType()) {
                case Advertisement.TYPE_VUNGLE_LOCAL:
                    MoatTracker tracker = MoatTracker.connect(fullAdWidget.videoView, apiClient.getMoatEnabled());

                    LocalAdPresenter localPresenter
                            = new LocalAdPresenter(
                            advertisement,
                            placement,
                            repository,
                            new HandlerScheduler(),
                            adAnalytics,
                            tracker,
                            webClient,
                            optionsState,
                            assetDir,
                            sessionData,
                            ActivityManager.getInstance());

                    LocalAdView localView = new LocalAdView(context, fullAdWidget, orientationDelegate, closeDelegate);

                    return new PresentationResultHolder(localView, localPresenter, webClient, tracker, appId);

                case Advertisement.TYPE_VUNGLE_MRAID:
                    MRAIDAdPresenter mraidPresenter
                            = new MRAIDAdPresenter(
                            advertisement,
                            placement,
                            repository,
                            new HandlerScheduler(),
                            adAnalytics,
                            webClient,
                            optionsState,
                            assetDir,
                            sessionData,
                            ActivityManager.getInstance());

                    MRAIDAdView mraidView = new MRAIDAdView(context, fullAdWidget, orientationDelegate, closeDelegate);

                    return new PresentationResultHolder(mraidView, mraidPresenter, webClient, null, null);

                default:
                    return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }
        }

        @Override
        protected void onPostExecute(PresentationResultHolder result) {
            super.onPostExecute(result);

            if (!isCancelled() && fullscreenCallback != null) {
                if (result.exception != null) {
                    Log.e(TAG, "Exception on creating presenter", result.exception);
                    fullscreenCallback.onResult(new Pair<AdView, AdvertisementPresenter>(null, null), result.exception);
                    return;
                }

                fullAdWidget.linkWebView(result.webClient, new JavascriptBridge(result.advertisementPresenter));

                if (result.tracker != null) {
                    ReactiveVideoTracker reactiveVideoTracker = MoatFactory.create().createCustomTracker(new ReactiveVideoTrackerPlugin(MoatTracker.VUNGLE_ID));
                    result.tracker.configure(placementId, advertisement, result.appId, reactiveVideoTracker);
                }

                fullscreenCallback.onResult(new Pair(result.adView, result.advertisementPresenter), result.exception);
            }
        }
    }

    private static class NativeViewPresentationTask extends BaseTask {

        private final String placementId;
        private final AdConfig adConfig;
        private final ViewCallback viewCallback;
        private final Bundle savedState;
        private final JobRunner jobRunner;
        private final AdLoader adLoader;
        private final SessionData sessionData;

        NativeViewPresentationTask(String placementId,
                                   AdConfig adConfig,
                                   AdLoader adLoader,
                                   Repository repository,
                                   VungleStaticApi vungleStaticApi,
                                   JobRunner jobRunner,
                                   ViewCallback viewCallback,
                                   Bundle savedState,
                                   SessionData sessionData,
                                   OnModelLoadListener onModelLoadListener) {
            super(repository, vungleStaticApi, onModelLoadListener);
            this.placementId = placementId;
            this.adConfig = adConfig;
            this.viewCallback = viewCallback;
            this.savedState = savedState;
            this.jobRunner = jobRunner;
            this.adLoader = adLoader;
            this.sessionData = sessionData;
        }

        @Override
        protected PresentationResultHolder doInBackground(Void... voids) {
            Pair<Advertisement, Placement> data;
            try {
                data = loadPresentationData(placementId, savedState);
            } catch (VungleException e) {
                return new PresentationResultHolder(e);
            }

            Advertisement advertisement = data.first;

            if (advertisement.getAdType() != Advertisement.TYPE_VUNGLE_MRAID) {
                return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }

            Placement placement = data.second;

            if (!adLoader.canPlayAd(advertisement)) {
                Log.e(TAG, "Advertisement is null or assets are missing");
                if (placement.isAutoCached()) {
                    adLoader.loadEndless(placement, 0);
                }
                return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }

            AdAnalytics adAnalytics = new JobDelegateAnalytics(jobRunner);

            VungleWebClient webClient = new VungleWebClient(advertisement, placement);
            File assetDir = repository.getAdvertisementAssetDirectory(advertisement.getId()).get();
            if (assetDir == null || !assetDir.isDirectory()) {
                Log.e(TAG, "Advertisement assets dir is missing");
                return new PresentationResultHolder(new VungleException(DB_ERROR));
            }

            if (advertisement.getAdType() != Advertisement.TYPE_VUNGLE_MRAID) {
                Log.e(TAG, "Invalid Ad Type for Native Ad.");
                return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }

            if ("mrec".equals(advertisement.getTemplateType()) &&
                    adConfig.getAdSize() != AdConfig.AdSize.VUNGLE_MREC ||
                    "flexfeed".equals(advertisement.getTemplateType()) &&
                            adConfig.getAdSize() != AdConfig.AdSize.VUNGLE_DEFAULT) {
                Log.e(TAG, "Corresponding AdConfig#setAdSize must be passed for the type/size of native ad");
                return new PresentationResultHolder(new VungleException(INVALID_SIZE));
            }

            advertisement.configure(adConfig);
            try {
                repository.save(advertisement);
            } catch (DatabaseHelper.DBException e) {
                return new PresentationResultHolder(new VungleException(DB_ERROR));
            }

            MRAIDAdPresenter presenter = new MRAIDAdPresenter(
                    advertisement,
                    placement,
                    repository,
                    new HandlerScheduler(),
                    adAnalytics,
                    webClient,
                    null,
                    assetDir,
                    sessionData,
                    ActivityManager.getInstance()
            );

            return new PresentationResultHolder(null, presenter, webClient, null, null);
        }

        @Override
        protected void onPostExecute(PresentationResultHolder result) {
            super.onPostExecute(result);
            if (!isCancelled() && viewCallback != null) {
                viewCallback.onResult(new Pair<>((WebAdContract.WebAdPresenter) result.advertisementPresenter, result.webClient), result.exception);
            }
        }
    }

    private abstract static class BaseTask extends AsyncTask<Void, Void, PresentationResultHolder> {

        protected final Repository repository;
        protected final VungleStaticApi vungleStaticApi;
        private OnModelLoadListener onDataLoadedListener;
        private AtomicReference<Advertisement> adRef = new AtomicReference<>();
        private AtomicReference<Placement> plRef = new AtomicReference<>();

        BaseTask(Repository repository, VungleStaticApi vungleStaticApi, OnModelLoadListener onModelLoadListener) {
            this.repository = repository;
            this.vungleStaticApi = vungleStaticApi;
            this.onDataLoadedListener = onModelLoadListener;
        }

        Pair<Advertisement, Placement> loadPresentationData(String placementId, Bundle savedInstanceState) throws VungleException {
            if (!vungleStaticApi.isInitialized()) {
                throw new VungleException(VUNGLE_NOT_INTIALIZED);
            }

            if (TextUtils.isEmpty(placementId)) {
                throw new VungleException(AD_UNABLE_TO_PLAY);
            }

            Placement placement = repository.load(placementId, Placement.class).get();
            if (placement == null) {
                Log.e(TAG, "No Placement for ID");
                throw new VungleException(PLACEMENT_NOT_FOUND);
            }

            plRef.set(placement);

            Advertisement advertisement = null;
            if (savedInstanceState == null) {
                advertisement = repository.findValidAdvertisementForPlacement(placementId).get();
            } else {
                // Restore cached adv, it should be deleted from disk already
                String adId = savedInstanceState.getString(EXTRA_ADVERTISEMENT);
                if (!TextUtils.isEmpty(adId)) {
                    advertisement = repository.load(adId, Advertisement.class).get();
                }
            }

            if (advertisement == null) {
                throw new VungleException(AD_UNABLE_TO_PLAY);
            }

            adRef.set(advertisement);

            File assetDir = repository.getAdvertisementAssetDirectory(advertisement.getId()).get();
            if (assetDir == null || !assetDir.isDirectory()) {
                Log.e(TAG, "Advertisement assets dir is missing");
                throw new VungleException(DB_ERROR);
            }

            return new Pair<>(advertisement, placement);
        }

        @Override
        protected void onPostExecute(PresentationResultHolder presentationResultHolder) {
            super.onPostExecute(presentationResultHolder);
            if (onDataLoadedListener != null) {
                onDataLoadedListener.onLoad(adRef.get(), plRef.get());
            }
        }

        void clear() {
            onDataLoadedListener = null;
        }

        interface OnModelLoadListener {
            void onLoad(Advertisement advertisement, Placement placement);
        }
    }

    private BaseTask.OnModelLoadListener onModelLoadListener = new BaseTask.OnModelLoadListener() {

        @Override
        public void onLoad(Advertisement ad, Placement pl) {
            currentAdvertisement = ad;
        }
    };
}
