package com.vungle.warren;

import android.text.TextUtils;
import android.util.Log;
import android.webkit.URLUtil;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.vungle.warren.AdConfig.AdSize;
import com.vungle.warren.downloader.AssetDownloadListener;
import com.vungle.warren.downloader.AssetDownloadListener.DownloadError.ErrorReason;
import com.vungle.warren.downloader.DownloadRequest;
import com.vungle.warren.downloader.Downloader;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.AdAsset;
import com.vungle.warren.model.AdAsset.FileType;
import com.vungle.warren.model.AdAsset.Status;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.JsonUtil;
import com.vungle.warren.model.Placement;
import com.vungle.warren.network.Call;
import com.vungle.warren.network.Callback;
import com.vungle.warren.network.Response;
import com.vungle.warren.persistence.CacheManager;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.tasks.DownloadJob;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.ui.HackMraid;
import com.vungle.warren.utility.Executors;
import com.vungle.warren.utility.FileUtility;
import com.vungle.warren.utility.UnzipUtility;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import static com.vungle.warren.error.VungleException.AD_FAILED_TO_DOWNLOAD;
import static com.vungle.warren.error.VungleException.ASSET_DOWNLOAD_ERROR;
import static com.vungle.warren.error.VungleException.ASSET_DOWNLOAD_RECOVERABLE;
import static com.vungle.warren.error.VungleException.DB_ERROR;
import static com.vungle.warren.error.VungleException.INVALID_SIZE;
import static com.vungle.warren.error.VungleException.NETWORK_ERROR;
import static com.vungle.warren.error.VungleException.NO_SERVE;
import static com.vungle.warren.error.VungleException.NO_SPACE_TO_DOWNLOAD_ASSETS;
import static com.vungle.warren.error.VungleException.NO_SPACE_TO_LOAD_AD;
import static com.vungle.warren.error.VungleException.NO_SPACE_TO_LOAD_AD_AUTO_CACHED;
import static com.vungle.warren.error.VungleException.OPERATION_CANCELED;
import static com.vungle.warren.error.VungleException.PLACEMENT_NOT_FOUND;
import static com.vungle.warren.error.VungleException.SERVER_ERROR;
import static com.vungle.warren.error.VungleException.SERVER_RETRY_ERROR;
import static com.vungle.warren.error.VungleException.SERVER_TEMPORARY_UNAVAILABLE;
import static com.vungle.warren.error.VungleException.UNKNOWN_ERROR;
import static com.vungle.warren.error.VungleException.VUNGLE_NOT_INTIALIZED;
import static com.vungle.warren.model.AdAsset.Status.DOWNLOAD_SUCCESS;
import static com.vungle.warren.model.AdAsset.Status.PROCESSED;
import static com.vungle.warren.model.Advertisement.ERROR;
import static com.vungle.warren.model.Advertisement.KEY_POSTROLL;
import static com.vungle.warren.model.Advertisement.KEY_TEMPLATE;
import static com.vungle.warren.model.Advertisement.NEW;
import static com.vungle.warren.model.Advertisement.READY;
import static com.vungle.warren.model.Advertisement.TYPE_VUNGLE_MRAID;

/**
 * The AdLoader is responsible for loading Ad and it's assets.
 */
public class AdLoader implements OperationSequence.Callback {

    private static final String TAG = AdLoader.class.getCanonicalName();
    public static final long EXPONENTIAL_RATE = 2;
    public static final long RETRY_DELAY = 2L * 1000L;
    public static final int RETRY_COUNT = 5;

    /**
     * Simple callback interface for downloading.
     */
    interface DownloadCallback {
        /**
         * Callback handler, triggered when the download has completed.
         */
        void onDownloadCompleted(@NonNull String placementId, @NonNull String advertisementId);

        /**
         * Callback handler, triggered when an error has occurred while downloading advertisement
         * assets.
         *
         * @param exception VungleException describing the error.
         */
        void onDownloadFailed(@NonNull VungleException exception, @Nullable String placementId, @Nullable String advertisementId);

        void onReady(@NonNull String id, @NonNull Placement placement, @NonNull Advertisement advertisement);
    }

    @IntDef(value = {ReschedulePolicy.EXPONENTIAL, ReschedulePolicy.EXPONENTIAL_ENDLESS_AD})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ReschedulePolicy {
        int EXPONENTIAL = 0;
        /**
         * Try to download assets {@link AdLoader#RETRY_COUNT} times.
         */
        int EXPONENTIAL_ENDLESS_AD = 1;
    }

    @IntDef(value = {Priority.HIGHEST, Priority.HIGH, Priority.LOWEST})
    public @interface Priority {
        int HIGHEST = 0;//for external API call
        int HIGH = 1;//for Placements with priority
        int LOWEST = Integer.MAX_VALUE;//for Placements without priority
    }

    public static class Operation {
        final String id;
        @NonNull
        final AdSize size;
        long delay;
        long retryDelay;
        int retry;
        int retryLimit;
        @ReschedulePolicy
        int policy;
        @NonNull
        final Set<LoadAdCallback> loadAdCallbacks = new CopyOnWriteArraySet<>();
        @NonNull
        final AtomicBoolean loading;
        boolean logError;
        @Priority
        int priority;
        List<DownloadRequest> requests = new CopyOnWriteArrayList<>();

        Operation(String id,
                  @NonNull AdSize size,
                  long delay,
                  long retryDelay,
                  int retryLimit,
                  @ReschedulePolicy int policy,
                  int retry,
                  boolean logError,
                  @Priority int priority,
                  @Nullable LoadAdCallback... loadAdCallbacks) {
            this.id = id;
            this.delay = delay;
            this.retryDelay = retryDelay;
            this.retryLimit = retryLimit;
            this.policy = policy;
            this.retry = retry;
            this.loading = new AtomicBoolean();
            this.size = size;
            this.logError = logError;
            this.priority = priority;
            if (loadAdCallbacks != null) {
                this.loadAdCallbacks.addAll(Arrays.asList(loadAdCallbacks));
            }
        }

        Operation delay(long delay) {
            return new Operation(id, size, delay, retryDelay, retryLimit, policy, retry, logError, priority, loadAdCallbacks.toArray(new LoadAdCallback[0]));
        }

        Operation retryDelay(long retryDelay) {
            return new Operation(id, size, delay, retryDelay, retryLimit, policy, retry, logError, priority, loadAdCallbacks.toArray(new LoadAdCallback[0]));
        }

        Operation retry(int retry) {
            return new Operation(id, size, delay, retryDelay, retryLimit, policy, retry, logError, priority, loadAdCallbacks.toArray(new LoadAdCallback[0]));
        }

        /**
         * Merges this operation with other.
         */
        void merge(Operation other) {
            //keep current: adSize, id, loading
            delay = Math.min(delay, other.delay);
            retryDelay = Math.min(retryDelay, other.retryDelay);
            retryLimit = Math.min(retryLimit, other.retryLimit);
            //EXPONENTIAL_ENDLESS_AD will not deliver callback
            policy = (other.policy == ReschedulePolicy.EXPONENTIAL ? other.policy : policy);
            retry = Math.min(retry, other.retry);
            logError |= other.logError;
            //keep highest priority
            priority = Math.min(priority, other.priority);
            loadAdCallbacks.addAll(other.loadAdCallbacks);
        }

        @NonNull
        @Override
        public String toString() {
            return "id=" + id +
                    " size=" + size.toString() +
                    " priority=" + priority +
                    " policy=" + policy +
                    " retry=" + retry + "/" + retryLimit +
                    " delay=" + delay + "->" + retryDelay +
                    " log=" + logError;
        }
    }

    /**
     * A state-tracking field which contains information about ongoing load operations. There can
     * only ever be a single load operation per active placement.
     */
    private final Map<String, Operation> loadOperations = new ConcurrentHashMap<>();
    private final Map<String, Operation> pendingOperations = new ConcurrentHashMap<>();
    private final OperationSequence sequence;

    //fixed values
    @NonNull
    private final Repository repository;
    @NonNull
    private final Executors sdkExecutors;
    @NonNull
    private final VungleApiClient vungleApiClient;
    @NonNull
    private final CacheManager cacheManager;
    @NonNull
    private final Downloader downloader;
    @NonNull
    private final RuntimeValues runtimeValues;
    @Nullable
    private JobRunner jobRunner;
    @NonNull
    private final VungleStaticApi vungleApi;
    @NonNull
    private final VisionController visionController;

    public AdLoader(
            @NonNull Executors sdkExecutors,
            @NonNull Repository repository,
            @NonNull VungleApiClient vungleApiClient,
            @NonNull CacheManager cacheManager,
            @NonNull Downloader downloader,
            @NonNull RuntimeValues runtimeValues,
            @NonNull VungleStaticApi vungleApi,
            @NonNull VisionController visionController,
            @NonNull OperationSequence sequence
    ) {
        this.sdkExecutors = sdkExecutors;
        this.repository = repository;
        this.vungleApiClient = vungleApiClient;
        this.cacheManager = cacheManager;
        this.downloader = downloader;
        this.runtimeValues = runtimeValues;
        this.vungleApi = vungleApi;
        this.visionController = visionController;
        this.sequence = sequence;
        sequence.init(this, loadOperations);
    }

    public synchronized void init(@NonNull JobRunner jobRunner) {
        this.jobRunner = jobRunner;
        downloader.init();
    }

    private boolean canReDownload(Advertisement advertisement) {
        if (advertisement == null || (advertisement.getState() != NEW && advertisement.getState() != READY)) {
            return false;
        }

        List<AdAsset> adAssets = repository.loadAllAdAssets(advertisement.getId()).get();
        if (adAssets == null || adAssets.size() == 0) {
            return false;
        }

        for (AdAsset asset : adAssets) {
            //Discard ad if unzipped asset is missing
            if (asset.fileType == FileType.ZIP_ASSET) {
                File file = new File(asset.localPath);
                if (!fileIsValid(file, asset))
                    return false;

            } else if (TextUtils.isEmpty(asset.serverPath)) {
                return false;
            }
        }

        return true;
    }

    @WorkerThread
    public boolean canPlayAd(final Advertisement advertisement) {
        if (advertisement == null || advertisement.getState() != READY) {
            return false;
        }

        return hasAssetsFor(advertisement.getId());
    }

    @WorkerThread
    public boolean canRenderAd(Advertisement advertisement) {
        if (advertisement == null)
            return false;


        if (advertisement.getState() != Advertisement.READY && advertisement.getState() != Advertisement.VIEWING)
            return false;

        return hasAssetsFor(advertisement.getId());
    }

    @WorkerThread
    public synchronized void clear() {
        Set<String> ids = new HashSet<>();
        ids.addAll(loadOperations.keySet());
        ids.addAll(pendingOperations.keySet());
        for (String id : ids) {
            onError(loadOperations.remove(id), OPERATION_CANCELED);
            onError(pendingOperations.remove(id), OPERATION_CANCELED);
        }
        for (Operation op : sequence.removeAll()) {
            onError(op, OPERATION_CANCELED);
        }
    }

    public synchronized boolean isLoading(String id) {
        if (sequence.contains(id)) {
            return true;
        }
        Operation op = loadOperations.get(id);
        return op != null && op.loading.get();
    }

    private void setLoading(String id, boolean loading) {
        Operation op = loadOperations.get(id);
        if (op != null) {
            op.loading.set(loading);
        }
    }

    /**
     * should be called from {@link DownloadJob} only
     */
    public synchronized void loadPendingInternal(final String id) {
        Operation op = pendingOperations.remove(id);
        if (op == null)
            return;

        load(op.delay(0));
    }

    public synchronized void load(@NonNull Operation op) {
        if (jobRunner == null) {
            onError(op, VUNGLE_NOT_INTIALIZED);
            return;
        }

        @Nullable Operation pending = pendingOperations.remove(op.id);
        if (pending != null) {
            op.merge(pending);
        }

        if (op.delay <= 0) {
            sequence.offer(op);
        } else {
            pendingOperations.put(op.id, op);
            jobRunner.execute(DownloadJob.makeJobInfo(op.id).setDelay(op.delay).setUpdateCurrent(true));
        }
    }

    /**
     * Can be called from different thread depending on
     * {@link OperationSequence#offer(Operation)}, {@link OperationSequence#reportFinished(String)} calls.
     * @param op Operation to load next
     */
    @Override
    public void onLoadNext(Operation op) {
        loadOperations.put(op.id, op);
        loadAd(op, new DownloadCallbackWrapper(sdkExecutors.getBackgroundExecutor(), new DownloadAdCallback()));
    }

    @Override
    public void onChangePriority(Operation op) {
        for (DownloadRequest request : op.requests) {
            request.setPriority(getAssetPriority(op.priority));
            downloader.updatePriority(request);
        }
    }

    private void onError(@Nullable Operation op, @VungleException.ExceptionCode int code) {
        if (op != null) {
            for (LoadAdCallback loadAdCallback : op.loadAdCallbacks) {
                loadAdCallback.onError(op.id, new VungleException(code));
            }
        }
    }

    private VungleException reposeCodeToVungleException(int code) {
        if (recoverableServerCode(code)) {
            return new VungleException(SERVER_TEMPORARY_UNAVAILABLE);
        } else {
            return new VungleException(SERVER_ERROR);
        }
    }

    private boolean recoverableServerCode(int code) {
        return code == 408 || (500 <= code && code < 600);
    }

    private VungleException retrofitToVungleException(Throwable throwable) {
        if (throwable instanceof UnknownHostException) {
            return new VungleException(AD_FAILED_TO_DOWNLOAD);
        } else if (throwable instanceof IOException) {
            return new VungleException(NETWORK_ERROR);
        } else {
            return new VungleException(AD_FAILED_TO_DOWNLOAD);
        }
    }

    /**
     * Request the Vungle SDK to load the assets for an advertisement by the placement identifier.
     * This will cause us to request an ad from the Ad Server, and if the bid is filled, the assets
     * will be downloaded. It is possible for the bid to get no responses and therefore no assets
     * will be loaded. In this case, the SDK will automatically retryCount at a later time, specified by
     * the Vungle Server. The callback will be notified that the assets are pending download.
     *
     * @param op operation
     * @param listener The optional callback which will be notified when the assets are loaded, or
     *                 if they are deferred. Errors will also be sent through this callback.
     */
    private void loadAd(@NonNull final Operation op, @NonNull final DownloadCallbackWrapper listener) {
        final long adRequestStartTimeStamp = System.currentTimeMillis();

        sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
            @Override
            public void run() {
                if (!vungleApi.isInitialized()) {
                    listener.onDownloadFailed(new VungleException(VUNGLE_NOT_INTIALIZED), op.id, null);
                    return;
                }

                Placement placement = repository.load(op.id, Placement.class).get();
                if (placement == null) {
                    listener.onDownloadFailed(new VungleException(PLACEMENT_NOT_FOUND), op.id, null);
                    return;
                }

                if (isSizeInvalid(placement, op.size)) {
                    listener.onDownloadFailed(new VungleException(INVALID_SIZE), op.id, null);
                    return;
                }

                final Advertisement advertisement = repository.findValidAdvertisementForPlacement(placement.getId()).get();
                if (placement.getPlacementAdType() == Placement.TYPE_VUNGLE_BANNER && advertisement != null && advertisement.getAdConfig().getAdSize() != op.size) {
                    try {
                        repository.deleteAdvertisement(advertisement.getId());
                    } catch (DatabaseHelper.DBException e) {
                        listener.onDownloadFailed(new VungleException(DB_ERROR), op.id, null);
                        return;
                    }
                }

                /// If the assets are already loaded and have not expired, do not download again.
                if (advertisement != null && canPlayAd(advertisement)) {
                    sequence.reportFinished(op.id);
                    listener.onReady(op.id, placement, advertisement);
                } else if (canReDownload(advertisement)) {
                    Log.d(TAG, "Found valid adv but not ready - downloading content");

                    final VungleSettings settings = runtimeValues.settings.get();
                    if (settings == null || cacheManager.getBytesAvailable() < settings.getMinimumSpaceForAd()) {
                        if (advertisement.getState() != ERROR) {
                            try {
                                repository.saveAndApplyState(advertisement, op.id, ERROR);
                            } catch (DatabaseHelper.DBException e) {
                                listener.onDownloadFailed(new VungleException(DB_ERROR), op.id, null);
                                return;
                            }
                        }
                        listener.onDownloadFailed(new VungleException(NO_SPACE_TO_DOWNLOAD_ASSETS), op.id, null);
                        return;
                    }

                    /// Update the instance state
                    setLoading(op.id, true);

                    if (advertisement.getState() != NEW) {
                        try {
                            repository.saveAndApplyState(advertisement, op.id, NEW);
                        } catch (DatabaseHelper.DBException e) {
                            listener.onDownloadFailed(new VungleException(DB_ERROR), op.id, null);
                            return;
                        }
                    }
                    advertisement.setAdRequestStartTime(adRequestStartTimeStamp);
                    advertisement.setAssetDownloadStartTime(System.currentTimeMillis());
                    downloadAdAssets(op, advertisement, listener);
                } else if (placement.getWakeupTime() > System.currentTimeMillis()) {
                    listener.onDownloadFailed(new VungleException(NO_SERVE), op.id, null);
                    Log.w(TAG, "Placement " + placement.getId() + " is  snoozed");

                    //rescheduling download to time when placement is active
                    if (placement.isAutoCached()) {
                        Log.d(TAG, "Placement " + placement.getId() + " is sleeping rescheduling it ");
                        loadEndless(placement, op.size, placement.getWakeupTime() - System.currentTimeMillis());
                    }
                } else {
                    Log.i(TAG, "didn't find cached adv for " + op.id + " downloading ");

                    if (advertisement != null) {
                        try {
                            repository.saveAndApplyState(advertisement, op.id, ERROR);
                        } catch (DatabaseHelper.DBException e) {
                            listener.onDownloadFailed(new VungleException(DB_ERROR), op.id, null);
                            return;
                        }
                    }

                    /// Download the ad
                    final VungleSettings settings = runtimeValues.settings.get();
                    if (settings != null && cacheManager.getBytesAvailable() < settings.getMinimumSpaceForAd()) {
                        listener.onDownloadFailed(new VungleException(placement.isAutoCached()
                                ? NO_SPACE_TO_LOAD_AD_AUTO_CACHED : NO_SPACE_TO_LOAD_AD), op.id, null);
                        return;
                    }

                    //Clear placement data as no valid adv found?
                    Log.d(TAG, "No adv for placement " + placement.getId() + " getting new data ");

                    /// Update the instance state
                    setLoading(op.id, true);
                    fetchAdMetadata(op, placement, listener);
                }
            }
        });
    }

    private boolean isSizeInvalid(Placement placement, AdSize size) {
        // if...else... checking two condition
        // 1 : If placement is BANNER TYPE but requested ad size is non banner than return INVALID_SIZE error
        // 2 : Else If placement is NON_BANNER TYPE but requested ad size is banner than return INVALID_SIZE error
        return ((placement.getPlacementAdType() == Placement.TYPE_VUNGLE_BANNER && !AdSize.isBannerAdSize(size))
                || (placement.getPlacementAdType() == Placement.TYPE_DEFAULT && AdSize.isBannerAdSize(size)));
    }

    /**
     * Multi-step method that downloads the required assets for the given placement identifier. It begins by
     * fetching an advertisement for the given placement, which the ad server provides along with all the
     * metadata and pointers to the downloadable assets. Once this information is ready, we download the
     * individual assets. Once the assets have finished downloading, we inform the callback of the success
     * of the operation.
     *
     * @param op Operation.
     * @param placement        The placement should be loaded
     * @param downloadCallback The callback that will notified of the success or failure of the download operation.
     */
    private void fetchAdMetadata(@NonNull final Operation op, @NonNull Placement placement, @NonNull final DownloadCallback downloadCallback) {
        final HeaderBiddingCallback bidTokenCallBack = runtimeValues.headerBiddingCallback.get();

        final long requestStartTime = System.currentTimeMillis();

        vungleApiClient.requestAd(
                op.id, AdSize.isBannerAdSize(op.size) ? op.size.getName() : "",
                placement.isHeaderBidding(),
                visionController.isEnabled() ? visionController.getPayload() : null
        ).enqueue(new Callback<JsonObject>() {
            @Override
            public void onFailure(Call<JsonObject> call, Throwable e) {
                downloadCallback.onDownloadFailed(retrofitToVungleException(e), op.id, null);
            }

            @Override
            public void onResponse(Call<JsonObject> call, final Response<JsonObject> response) {
                //Lets do it in separate thread, since there are few IO calls involved.
                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        /// The Ad Server has given us metadata for the advertisement, we now download the assets

                        /// Load the placement information from disk. If it cannot be found, an error has
                        /// occurred. Most likely this would be due to a cache clear or db dump.
                        Placement placement = repository.load(op.id, Placement.class).get();
                        if (placement == null) {
                            Log.e(TAG, "Placement metadata not found for requested advertisement.");
                            downloadCallback.onDownloadFailed(new VungleException(UNKNOWN_ERROR), op.id, null);
                            return;
                        }

                        if (!response.isSuccessful()) {
                            long retryAfterHeaderValue = vungleApiClient.getRetryAfterHeaderValue(response);
                            if (retryAfterHeaderValue > 0 && placement.isAutoCached()) {     //Rescheduling only if Auto cached Ad
                                loadEndless(placement, op.size, retryAfterHeaderValue);
                                downloadCallback.onDownloadFailed(new VungleException(SERVER_RETRY_ERROR), op.id, null);
                                return;
                            }

                            /// We can't do anything if the request failed.
                            Log.e(TAG, "Failed to retrieve advertisement information");
                            downloadCallback.onDownloadFailed(reposeCodeToVungleException(response.code()), op.id, null);
                            return;
                        }

                        JsonObject jsonObject = response.body();
                        Log.d(TAG, "Ads Response: " + jsonObject);
                        if (jsonObject != null && jsonObject.has("ads") && !jsonObject.get("ads").isJsonNull()) {
                            JsonArray ads = jsonObject.getAsJsonArray("ads");

                            if (ads == null || ads.size() == 0) {
                                // If there is no ad available, we need to inform the caller about the no serve response.
                                downloadCallback.onDownloadFailed(new VungleException(NO_SERVE), op.id, null);
                                return;
                            }

                            JsonObject ad = ads.get(0).getAsJsonObject();
                            try {
                                final Advertisement advertisement = new Advertisement(ad);

                                if (visionController.isEnabled()) {
                                    JsonObject markup = ad.getAsJsonObject("ad_markup");
                                    if (JsonUtil.hasNonNull(markup, VisionController.DATA_SCIENCE_CACHE)) {
                                        visionController.setDataScienceCache(markup.get(VisionController.DATA_SCIENCE_CACHE).getAsString());
                                    } else {
                                        visionController.setDataScienceCache(null);
                                    }
                                }

                                /*
                                 *  Validation for Ad having same Ad Id. (Not the real scenario, only happens when using mock response)
                                 *  This code failing some test cases if there is auto-cached Ad in the config response,
                                 *  because generally for test case we are using mock response which has same ID.
                                 *  So loadAd() is called two times one for autocache ad and one for loadAd called manually,
                                 * in this case for second loadAd() response is giving operation_canceled error callback.
                                 */
                                Advertisement advertisementInDB = repository.load(advertisement.getId(), Advertisement.class).get();
                                if (advertisementInDB != null) {
                                    int state = advertisementInDB.getState();
                                    if (state == Advertisement.NEW || state == Advertisement.READY || state == Advertisement.VIEWING) {
                                        Log.d(TAG, "Operation Cancelled");
                                        downloadCallback.onDownloadFailed(new VungleException(OPERATION_CANCELED), op.id, null);
                                        return;
                                    }
                                }

                                if (placement.isHeaderBidding() && bidTokenCallBack != null) {
                                    bidTokenCallBack.onBidTokenAvailable(op.id, advertisement.getBidToken());
                                }

                                //clear data if exists
                                repository.deleteAdvertisement(advertisement.getId());

                                Set<Map.Entry<String, String>> entries = advertisement.getDownloadableUrls().entrySet();
                                final File destinationDir = getDestinationDir(advertisement);
                                if (destinationDir == null || !destinationDir.isDirectory()) {
                                    downloadCallback.onDownloadFailed(new VungleException(DB_ERROR), op.id, advertisement.getId());
                                    return;
                                }

                                for (Map.Entry<String, String> entry : entries) {
                                    if (URLUtil.isHttpsUrl(entry.getValue()) || URLUtil.isHttpUrl(entry.getValue())) {
                                        saveAsset(advertisement, destinationDir,
                                                entry.getKey(), entry.getValue());
                                    } else {
                                        downloadCallback.onDownloadFailed(new VungleException(AD_FAILED_TO_DOWNLOAD), op.id, advertisement.getId());
                                        return;
                                    }
                                }

                                if (placement.getPlacementAdType() == Placement.TYPE_VUNGLE_BANNER &&
                                        (advertisement.getAdType() != TYPE_VUNGLE_MRAID || !"banner".equals(advertisement.getTemplateType()))) {
                                    downloadCallback.onDownloadFailed(new VungleException(NO_SERVE), op.id, advertisement.getId());
                                    return;
                                }

                                advertisement.getAdConfig().setAdSize(op.size);
                                advertisement.setAdRequestStartTime(requestStartTime);
                                advertisement.setAssetDownloadStartTime(System.currentTimeMillis());
                                repository.saveAndApplyState(advertisement, op.id, NEW);
                                downloadAdAssets(op, advertisement, downloadCallback);
                            } catch (IllegalArgumentException badAd) {
                                /// Failed to parse the advertisement, which means there was no ad served.
                                /// Check for a sleep code and schedule a download job.
                                JsonObject admarkup = ad.getAsJsonObject("ad_markup");
                                if (admarkup.has("sleep")) {
                                    int sleep = admarkup.get("sleep").getAsInt();

                                    /// Set this sleep time on the placement. We should not attempt to load
                                    /// more ads for this placement ID until the time has elapsed.
                                    placement.snooze(sleep);
                                    try {
                                        repository.save(placement);
                                    } catch (DatabaseHelper.DBException ignored) {
                                        downloadCallback.onDownloadFailed(new VungleException(DB_ERROR), op.id, null);
                                        return;
                                    }
                                    if (placement.isAutoCached()) {
                                        /// If the placement is auto-cached, schedule a download as soon as
                                        /// it wakes up. Note that since sleep is in seconds, we need to
                                        /// convert to milliseconds.
                                        loadEndless(placement, op.size, sleep * 1000L);
                                    }
                                }

                                downloadCallback.onDownloadFailed(new VungleException(NO_SERVE), op.id, null);
                            } catch (DatabaseHelper.DBException ignored) {
                                downloadCallback.onDownloadFailed(new VungleException(DB_ERROR), op.id, null);
                            }
                        } else {
                            downloadCallback.onDownloadFailed(new VungleException(NO_SERVE), op.id, null);
                        }
                    }
                });
            }

        });
    }

    @Nullable
    File getDestinationDir(Advertisement advertisement) {
        return repository.getAdvertisementAssetDirectory(advertisement.getId()).get();
    }

    void saveAsset(Advertisement advertisement, File destinationDir, String key, String url) throws DatabaseHelper.DBException {
        String path = destinationDir.getPath() + File.separator + key;

        @FileType int type = path.endsWith(KEY_POSTROLL) || path.endsWith(KEY_TEMPLATE)
                ? FileType.ZIP
                : FileType.ASSET;

        AdAsset adAsset = new AdAsset(advertisement.getId(), url, path);
        adAsset.status = Status.NEW;
        adAsset.fileType = type;
        repository.save(adAsset);
    }

    /**
     * @param op               Operation
     * @param advertisement    Ad
     * @param downloadCallback callback
     */
    private void downloadAdAssets(final Operation op, final Advertisement advertisement, DownloadCallback downloadCallback) {
        sequence.reportFinished(op.id);
        op.requests.clear();

        //validating URL , if one or more URLs is empty or not valid, SDK cannot determine how important that asset
        // is or what will be its side-effect. Its best to fail and drop the entire Ad-delivery.
        for (Map.Entry<String, String> entry : advertisement.getDownloadableUrls().entrySet()) {
            if (TextUtils.isEmpty(entry.getKey()) || TextUtils.isEmpty(entry.getValue()) ||
                    !URLUtil.isValidUrl(entry.getValue())) {
                downloadCallback.onDownloadFailed(new VungleException(AD_FAILED_TO_DOWNLOAD), op.id, null);
                Log.e(TAG, "Aborting, Failed to download Ad assets for: " + advertisement.getId());
                return;
            }
        }

        final DownloadCallback callback = new DownloadCallbackWrapper(sdkExecutors.getUIExecutor(), downloadCallback);

        try {
            repository.save(advertisement);
        } catch (DatabaseHelper.DBException e) {
            downloadCallback.onDownloadFailed(new VungleException(DB_ERROR), op.id, advertisement.getId());
            return;
        }

        /// Kick off the downloads in sequence. The downloader implementation will
        /// decide how to handle concurrent downloads, but this operation will
        /// notify the subscriber when all assets have been downloaded.

        List<AdAsset> assets = repository.loadAllAdAssets(advertisement.getId()).get();
        if (assets == null) {
            callback.onDownloadFailed(new VungleException(DB_ERROR), op.id, advertisement.getId());
            return;
        }

        for (AdAsset asset : assets) {
            if (asset.status == Status.DOWNLOAD_SUCCESS) {
                if (fileIsValid(new File(asset.localPath), asset)) {
                    continue;
                }

                if (asset.fileType == FileType.ZIP_ASSET) {
                    callback.onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), op.id, advertisement.getId());
                    return;
                }
            }

            if (asset.status == Status.PROCESSED && asset.fileType == FileType.ZIP) {
                continue;
            }

            if (TextUtils.isEmpty(asset.serverPath)) {
                callback.onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), op.id, advertisement.getId());
                return;
            }

            DownloadRequest downloadRequest = getDownloadRequest(asset, op.priority);

            if (asset.status == Status.DOWNLOAD_RUNNING) {
                downloader.cancelAndAwait(downloadRequest, 1000);
                downloadRequest = getDownloadRequest(asset, op.priority);
            }

            Log.d(TAG, "Starting download for " + asset);
            asset.status = Status.DOWNLOAD_RUNNING;
            try {
                repository.save(asset);
            } catch (DatabaseHelper.DBException ignored) {
                callback.onDownloadFailed(new VungleException(DB_ERROR), op.id, advertisement.getId());
                return;
            }
            op.requests.add(downloadRequest);
        }

        //All assets were downloaded already
        if (op.requests.size() == 0) {
            onAssetDownloadFinished(op.id, callback, advertisement, Collections.EMPTY_LIST);
            return;
        }

        AssetDownloadListener downloadListener = getAssetDownloadListener(advertisement, op, callback);
        for (DownloadRequest downloadRequest : op.requests) {
            downloader.download(downloadRequest, downloadListener);
        }
    }

    private DownloadRequest getDownloadRequest(AdAsset asset, @Priority int priority) {
        return new DownloadRequest(
                Downloader.NetworkType.ANY,
                getAssetPriority(priority),
                asset.serverPath, asset.localPath,
                false,
                asset.identifier);
    }

    private @DownloadRequest.Priority
    int getAssetPriority(@Priority int priority) {
        return Math.max(DownloadRequest.Priority.CRITICAL + 1, priority);
    }

    @NonNull
    private AssetDownloadListener getAssetDownloadListener(
            final Advertisement advertisement,
            final Operation op,
            final DownloadCallback callback) {

        return new AssetDownloadListener() {

            AtomicLong downloadCount = new AtomicLong(op.requests.size());
            List<DownloadError> errors = Collections.synchronizedList(new ArrayList<DownloadError>());

            @Override
            public void onError(@NonNull final DownloadError downloadError,
                                final @Nullable DownloadRequest downloadRequest) {

                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        Log.e(TAG, "Download Failed");
                        if (downloadRequest != null) {
                            String id = downloadRequest.cookieString;
                            AdAsset asset = TextUtils.isEmpty(id) ? null
                                    : repository.load(id, AdAsset.class).get();

                            if (asset != null) {
                                errors.add(downloadError);
                                asset.status = Status.DOWNLOAD_FAILED;
                                try {
                                    repository.save(asset);
                                } catch (DatabaseHelper.DBException e) {
                                    errors.add(new DownloadError(
                                            -1,
                                            new VungleException(DB_ERROR),
                                            ErrorReason.INTERNAL_ERROR
                                    ));
                                }
                            } else {
                                errors.add(new DownloadError(
                                        -1,
                                        new IOException("Downloaded file not found!"),
                                        ErrorReason.REQUEST_ERROR
                                ));
                            }
                        } else {
                            errors.add(new DownloadError(-1,
                                    new RuntimeException("error in request"), ErrorReason.INTERNAL_ERROR));
                        }

                        if (downloadCount.decrementAndGet() <= 0) {
                            onAssetDownloadFinished(op.id, callback, advertisement, errors);
                        }
                    }
                });
            }

            @Override
            public void onProgress(@NonNull Progress progress,
                                   @NonNull DownloadRequest downloadRequest) {

            }

            @Override
            public void onSuccess(@NonNull final File downloadedFile,
                                  @NonNull final DownloadRequest downloadRequest) {
                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        if (!downloadedFile.exists()) {
                            onError(new DownloadError(
                                            -1,
                                            new IOException("Downloaded file not found!"),
                                            ErrorReason.FILE_NOT_FOUND_ERROR
                                    ),
                                    downloadRequest);   //AdAsset table will be updated in onError callback
                            return;
                        }

                        String id = downloadRequest.cookieString;
                        AdAsset adAsset = id == null ? null : repository.load(id, AdAsset.class).get();
                        if (adAsset == null) {
                            onError(new DownloadError(
                                            -1,
                                            new IOException("Downloaded file not found!"),
                                            ErrorReason.REQUEST_ERROR
                                    ),
                                    downloadRequest);   //AdAsset table will be updated in onError callback
                            return;
                        }

                        adAsset.fileType = isZip(downloadedFile) ? FileType.ZIP : FileType.ASSET;
                        adAsset.fileSize = downloadedFile.length();
                        adAsset.status = Status.DOWNLOAD_SUCCESS;
                        try {
                            repository.save(adAsset);
                        } catch (DatabaseHelper.DBException e) {
                            onError(new DownloadError(-1, new VungleException(DB_ERROR), ErrorReason.INTERNAL_ERROR), downloadRequest);
                            return;
                        }

                        if (downloadCount.decrementAndGet() <= 0) {
                            /// In the case of an MRAID ad, we also have to update the
                            /// cacheable_replacements map to point to the local files.N
                            /// Otherwise, the html will load them from the CDN and could
                            /// cause the user to experience lag.

                            onAssetDownloadFinished(op.id, callback, advertisement, errors);
                        }
                    }
                });
            }
        };
    }

    private boolean isZip(File downloadedFile) {
        return downloadedFile.getName().equals(KEY_POSTROLL) || downloadedFile.getName().equals(KEY_TEMPLATE);
    }

    private void onAssetDownloadFinished(@NonNull final String placementId,
                                         @NonNull final DownloadCallback callback,
                                         @NonNull final Advertisement advertisement,
                                         @NonNull List<AssetDownloadListener.DownloadError> errors) {
        if (errors.isEmpty()) {
            List<AdAsset> assets = repository.loadAllAdAssets(advertisement.getId()).get();

            if (assets == null || assets.size() == 0) {
                callback.onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), placementId, advertisement.getId());
                return;
            }

            for (AdAsset asset : assets) {
                if (asset.status == Status.DOWNLOAD_SUCCESS) {
                    File f = new File(asset.localPath);

                    if (!fileIsValid(f, asset)) {
                        callback.onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), placementId, advertisement.getId());
                        return;
                    }

                    if (asset.fileType == FileType.ZIP) {
                        try {
                            unzipFile(advertisement, asset, f, assets);
                        } catch (IOException e) {
                            // Remove from cache as zip is malformed
                            downloader.dropCache(asset.serverPath);
                            callback.onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), placementId, advertisement.getId());
                            return;
                        } catch (DatabaseHelper.DBException e) {
                            callback.onDownloadFailed(new VungleException(DB_ERROR), placementId, advertisement.getId());
                            return;
                        }
                    }
                } else if (asset.fileType == FileType.ZIP && asset.status != PROCESSED) {
                    callback.onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), placementId, advertisement.getId());
                    return;
                }
            }

            if (advertisement.getAdType() == TYPE_VUNGLE_MRAID) {
                File destinationDir = getDestinationDir(advertisement);
                if (destinationDir == null || !destinationDir.isDirectory()) {
                    callback.onDownloadFailed(new VungleException(DB_ERROR), placementId, advertisement.getId());
                    return;
                }

                Log.d(TAG, "saving MRAID for " + advertisement.getId());
                advertisement.setMraidAssetDir(destinationDir);
                try {
                    repository.save(advertisement);
                } catch (DatabaseHelper.DBException e) {
                    callback.onDownloadFailed(new VungleException(DB_ERROR), placementId, advertisement.getId());
                    return;
                }
            }

            callback.onDownloadCompleted(placementId, advertisement.getId());
        } else {
            VungleException endError = null;

            for (AssetDownloadListener.DownloadError downloadError : errors) {
                VungleException error;

                if (VungleException.getExceptionCode(downloadError.cause) == DB_ERROR) {
                    endError = new VungleException(DB_ERROR);
                    break;
                }

                if (recoverableServerCode(downloadError.serverCode) && downloadError.reason == ErrorReason.REQUEST_ERROR) {
                    error = new VungleException(ASSET_DOWNLOAD_RECOVERABLE);
                } else if (downloadError.reason == ErrorReason.CONNECTION_ERROR) {
                    error = new VungleException(ASSET_DOWNLOAD_RECOVERABLE);
                } else {
                    error = new VungleException(ASSET_DOWNLOAD_ERROR);
                }

                endError = error;

                if (endError.getExceptionCode() == ASSET_DOWNLOAD_ERROR)
                    break;
            }

            final VungleException exception = endError;
            callback.onDownloadFailed(exception, placementId, advertisement.getId());
        }
    }

    private class DownloadAdCallback implements DownloadCallback {

        @Override
        public void onDownloadCompleted(@NonNull final String placementId, @NonNull final String advertisementId) {
            synchronized (AdLoader.this) {
                Log.d(TAG, "download completed " + placementId);

                final Placement placement = repository.load(placementId, Placement.class).get();
                if (placement == null) {
                    onDownloadFailed(new VungleException(PLACEMENT_NOT_FOUND), placementId, advertisementId);
                    return;
                }

                final Advertisement advertisement = TextUtils.isEmpty(advertisementId)
                        ? null
                        : repository.load(advertisementId, Advertisement.class).get();
                if (advertisement == null) {
                    onDownloadFailed(new VungleException(AD_FAILED_TO_DOWNLOAD), placementId, advertisementId);
                    return;
                }

                advertisement.setFinishedDownloadingTime(System.currentTimeMillis());

                try {
                    repository.saveAndApplyState(advertisement, placementId, READY);
                } catch (DatabaseHelper.DBException e) {
                    onDownloadFailed(new VungleException(DB_ERROR), placementId, advertisementId);
                    return;
                }
                onReady(placementId, placement, advertisement);
            }
        }

        @Override
        public void onReady(@NonNull final String id, @NonNull final Placement placement, @NonNull final Advertisement advertisement) {
            synchronized (AdLoader.this) {
                setLoading(id, false);

                //for ads that are already cached and SDK receives loadAd it should also fire callback with bid token
                HeaderBiddingCallback headerBiddingCallback = runtimeValues.headerBiddingCallback.get();
                if (placement.isHeaderBidding() && headerBiddingCallback != null) {
                    headerBiddingCallback.adAvailableForBidToken(id, advertisement.getBidToken());
                }

                Log.i(TAG, "found already cached valid adv, calling onAdLoad " + id + " callback ");
                /// We have assets for this placement. Verify that the advertisement link is still valid.
                final InitCallback initCallback = runtimeValues.initCallback.get();
                if (placement.isAutoCached() && initCallback != null) {
                    initCallback.onAutoCacheAdAvailable(id);
                }

                Operation operation = loadOperations.remove(id);

                if (operation != null) {
                    placement.setAdSize(operation.size);

                    try {
                        repository.save(placement);
                    } catch (DatabaseHelper.DBException e) {
                        onDownloadFailed(new VungleException(DB_ERROR), id, advertisement.getId());
                    }

                    for (LoadAdCallback loadAdCallback : operation.loadAdCallbacks) {
                        loadAdCallback.onAdLoad(id);
                    }
                }
            }
        }

        @Override
        public void onDownloadFailed(@NonNull VungleException exception, final String placementId, final String advertisementId) {
            synchronized (AdLoader.this) {
                final Operation op = loadOperations.remove(placementId);
                sequence.reportFinished(placementId);

                Placement placement = repository.load(placementId, Placement.class).get();
                Advertisement advertisement = advertisementId == null
                        ? null
                        : repository.load(advertisementId, Advertisement.class).get();

                if (placement == null) {
                    if (advertisement != null) {
                        try {
                            repository.saveAndApplyState(advertisement, placementId, ERROR);
                        } catch (DatabaseHelper.DBException ignored) {
                            exception = new VungleException(DB_ERROR);
                        }
                    }

                    if (op != null) {
                        for (LoadAdCallback loadAdCallback : op.loadAdCallbacks) {
                            loadAdCallback.onError(placementId, exception);
                        }
                    }
                    setLoading(placementId, false);
                    return;
                }

                boolean canRetry = false;
                boolean stopInfinite = false;
                @Advertisement.State int state = ERROR;

                switch (exception.getExceptionCode()) {
                    case SERVER_TEMPORARY_UNAVAILABLE:
                    case NETWORK_ERROR:
                        canRetry = true;
                        break;
                    case ASSET_DOWNLOAD_RECOVERABLE:
                        if (advertisement != null) {
                            canRetry = true;
                            state = NEW;
                        }
                        break;
                    case OPERATION_CANCELED:
                    case NO_SERVE:
                    case SERVER_RETRY_ERROR:
                        stopInfinite = true;
                        break;
                    default:
                        break;
                }

                if (op == null || op.logError) {
                    Log.e(TAG, "Failed to load Ad/Assets for " + placementId + ". Cause : ", exception);
                }

                setLoading(placementId, false);

                if (op != null) {
                    try {
                        if (op.policy == ReschedulePolicy.EXPONENTIAL) {
                            if (op.retry < op.retryLimit && canRetry) {
                                if (advertisement != null) {
                                    repository.saveAndApplyState(advertisement, placementId, state);
                                }
                                load(op.delay(op.retryDelay).retryDelay(op.retryDelay * EXPONENTIAL_RATE).retry(op.retry + 1));
                                return;
                            }
                        } else if (op.policy == ReschedulePolicy.EXPONENTIAL_ENDLESS_AD && !stopInfinite) {
                            int retry = op.retry;
                            if (retry < op.retryLimit && canRetry) {
                                retry++;
                            } else {
                                retry = 0;//forever request /ads until stop
                                state = ERROR;
                            }
                            if (advertisement != null) {
                                repository.saveAndApplyState(advertisement, placementId, state);
                            }
                            load(op.delay(op.retryDelay).retryDelay(op.retryDelay * EXPONENTIAL_RATE).retry(retry));
                            return;
                        }

                        if (advertisement != null) {
                            repository.saveAndApplyState(advertisement, placementId, ERROR);
                        }
                    } catch (DatabaseHelper.DBException e) {
                        exception = new VungleException(DB_ERROR);
                    }

                    for (LoadAdCallback loadAdCallback : op.loadAdCallbacks) {
                        loadAdCallback.onError(placementId, exception);
                    }
                }
            }
        }
    }

    //helper functions
    public void load(String id, LoadAdCallback listener) {
        load(new Operation(
                id,
                AdSize.VUNGLE_DEFAULT,
                0,
                RETRY_DELAY,
                RETRY_COUNT,
                ReschedulePolicy.EXPONENTIAL,
                0,
                true,
                Priority.HIGHEST,
                listener
        ));
    }

    //helper functions
    public void load(String id, AdConfig adConfig, LoadAdCallback listener) {
        load(new Operation(
                id,
                adConfig.getAdSize(),
                0,
                RETRY_DELAY,
                RETRY_COUNT,
                ReschedulePolicy.EXPONENTIAL,
                0,
                true,
                Priority.HIGHEST,
                listener
        ));
    }

    public void loadEndless(@NonNull final Placement placement, @NonNull final AdSize size, final long delay) {
        if (isSizeInvalid(placement, size))
            return;

        load(new Operation(
                placement.getId(),
                size,
                delay,
                RETRY_DELAY,
                RETRY_COUNT,
                ReschedulePolicy.EXPONENTIAL_ENDLESS_AD,
                0,
                false,
                placement.getAutoCachePriority()
        ));
    }

    public void loadEndless(@NonNull final Placement placement, final long delay) {
        loadEndless(placement, placement.getAdSize(), delay);
    }

    private void unzipFile(Advertisement advertisement,
                           AdAsset zipAsset,
                           @NonNull final File downloadedFile,
                           List<AdAsset> allAssets) throws IOException, DatabaseHelper.DBException {
        final List<String> existingPaths = new ArrayList<>();
        for (AdAsset asset : allAssets) {
            if (asset.fileType == FileType.ASSET) {
                existingPaths.add(asset.localPath);
            }
        }

        File destinationDir = getDestinationDir(advertisement);
        if (destinationDir == null || !destinationDir.isDirectory()) {
            throw new IOException("Unable to access Destination Directory");
        }

        List<File> extractedFiles = UnzipUtility.unzip(downloadedFile.getPath(), destinationDir.getPath(), new UnzipUtility.Filter() {
            @Override
            public boolean matches(String extractPath) {

                File toExtract = new File(extractPath);

                for (String existing : existingPaths) {
                    File existingFile = new File(existing);

                    if (existingFile.equals(toExtract))
                        return false;

                    if (toExtract.getPath().startsWith(existingFile.getPath() + File.separator))
                        return false;
                }

                return true;
            }
        });

        if (downloadedFile.getName().equals(KEY_TEMPLATE)) {
            //  Updating mraid.js
            // Find the mraid.js file and append the MRAID code. Why this is not done by the server is
            // confusing to me but I assume there must be some historical context to it.
            File mraidJS = new File(destinationDir.getPath() + File.separator + "mraid.js");
            if (mraidJS.exists()) {        //mraid.js exits
                PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(mraidJS, true)));
                HackMraid.apply(out);
                out.close();
            }
        }

        for (File file : extractedFiles) {
            AdAsset extractedAsset = new AdAsset(advertisement.getId(), null, file.getPath());
            extractedAsset.fileSize = file.length();
            extractedAsset.fileType = FileType.ZIP_ASSET;
            extractedAsset.parentId = zipAsset.identifier;
            extractedAsset.status = Status.DOWNLOAD_SUCCESS;
            repository.save(extractedAsset);
        }

        Log.d(TAG, "Uzipped " + destinationDir);
        FileUtility.printDirectoryTree(destinationDir);

        zipAsset.status = Status.PROCESSED;
        repository.save(zipAsset, new Repository.SaveCallback() {
            @Override
            public void onSaved() {
                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            FileUtility.delete(downloadedFile);
                        } catch (IOException e) {
                            //ignored
                            Log.e(TAG, "Error on deleting zip assets archive", e);
                        }
                    }
                });
            }

            @Override
            public void onError(Exception ignored) {
            }
        });
    }

    boolean hasAssetsFor(String adId) throws IllegalStateException {
        List<AdAsset> adAssets = repository.loadAllAdAssets(adId).get();

        if (adAssets == null || adAssets.size() == 0) {
            return false;
        }

        boolean isAllAssetAvailable = true;

        for (AdAsset adAsset : adAssets) {

            if (adAsset.fileType == FileType.ZIP) {
                if (adAsset.status == PROCESSED) {
                    continue;
                }

                isAllAssetAvailable = false;
                break;
            }

            if (adAsset.status != DOWNLOAD_SUCCESS) {
                isAllAssetAvailable = false;
                break;
            }

            File file = new File(adAsset.localPath);
            if (!fileIsValid(file, adAsset)) {
                isAllAssetAvailable = false;
                break;
            }

        }

        return isAllAssetAvailable;
    }

    private boolean fileIsValid(File file, AdAsset adAsset) {
        return file.exists() && file.length() == adAsset.fileSize;
    }

    @VisibleForTesting
    Collection<Operation> getPendingOperations() {
        return pendingOperations.values();
    }

    @VisibleForTesting
    Collection<Operation> getRunningOperations() {
        return loadOperations.values();
    }

    public void dropCache(String advertisementId) {
        List<AdAsset> adAssets = repository.loadAllAdAssets(advertisementId).get();
        if (adAssets == null) {
            Log.w(TAG, "No assets found in ad cache to cleanup");
            return;
        }
        for (AdAsset asset : adAssets) {
            downloader.dropCache(asset.serverPath);
        }
    }
}
