package com.vungle.warren;

import android.util.Log;

import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.Report;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.tasks.SendReportsJob;
import com.vungle.warren.ui.contract.AdContract;

import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import static com.vungle.warren.error.VungleException.AD_UNABLE_TO_PLAY;
import static com.vungle.warren.error.VungleException.ALREADY_PLAYING_ANOTHER_AD;
import static com.vungle.warren.error.VungleException.DB_ERROR;
import static com.vungle.warren.error.VungleException.OPERATION_CANCELED;
import static com.vungle.warren.error.VungleException.PLACEMENT_NOT_FOUND;
import static com.vungle.warren.error.VungleException.RENDER_ERROR;
import static com.vungle.warren.model.Advertisement.VIEWING;

public class AdEventListener implements AdContract.AdvertisementPresenter.EventListener {

    private static final String TAG = AdEventListener.class.getCanonicalName();
    private final Repository repository;
    private final AdLoader adLoader;
    private final JobRunner jobRunner;
    private final VisionController visionController;

    private final Map<String, Boolean> playOperations;
    private final PlayAdCallback playAdCallback;
    private final String placementId;

    private boolean successfulView;
    /**
     * The percentage of the ad that was viewed. We initialize to -1 in order to differentiate
     * between the ad never starting.
     */
    private int percentViewed = -1;

    private boolean adRewarded;
    private Placement placement;
    private Advertisement advertisement;

    AdEventListener(@NonNull String placementId,
                    @NonNull Map<String, Boolean> playOperations,
                    @Nullable PlayAdCallback playAdCallback,
                    @NonNull Repository repository,
                    @NonNull AdLoader adLoader,
                    @NonNull JobRunner jobRunner,
                    @NonNull VisionController visionController,
                    @Nullable Placement placement,
                    @Nullable Advertisement advertisement) {
        this.placementId = placementId;
        this.playOperations = playOperations;
        this.playAdCallback = playAdCallback;
        this.repository = repository;
        this.adLoader = adLoader;
        this.jobRunner = jobRunner;
        this.visionController = visionController;
        this.placement = placement;
        this.advertisement = advertisement;
        playOperations.put(placementId, true);
    }

    public void onNext(@NonNull String s, String value, String id) {
        if (advertisement == null) {
            advertisement = repository.findValidAdvertisementForPlacement(placementId).get();
            if (advertisement == null) {
                Log.e(TAG, "No Advertisement for ID");
                onFinished();
                if (playAdCallback != null) {
                    playAdCallback.onError(placementId, new VungleException(AD_UNABLE_TO_PLAY));
                }
                return;
            }
        }
        if (placement == null) {
            placement = repository.load(placementId, Placement.class).get();
            if (placement == null) {
                Log.e(TAG, "No Placement for ID");
                onFinished();
                if (playAdCallback != null) {
                    playAdCallback.onError(placementId, new VungleException(PLACEMENT_NOT_FOUND));
                }
                return;
            }
        }

        try {
            if (s.equals("start")) {
                repository.saveAndApplyState(advertisement, id, VIEWING);

                if (playAdCallback != null) {
                    playAdCallback.onAdStart(id);
                }

                percentViewed = 0;
                placement = repository.load(placementId, Placement.class).get();
                if (placement != null && placement.isAutoCached()) {
                    /// If the placement is auto-cached, download a new advertisement for it soon.
                    adLoader.loadEndless(placement, 0);
                }
                if (visionController.isEnabled()) {
                    visionController.reportData(
                            advertisement.getCreativeId(),
                            advertisement.getCampaignId(),
                            advertisement.getAdvertiserAppId());
                }
            } else if (s.equals("end")) {
                Log.d("Vungle", "Cleaning up metadata and assets for placement " + id + " and advertisement " + advertisement.getId());
                /// Cleanup the asset folder for the advertisement
                repository.saveAndApplyState(advertisement, id, Advertisement.DONE);
                repository.updateAndSaveReportState(id, advertisement.getAppID(), Report.NEW, Report.READY);

                /// Schedule a job to send the report soon
                jobRunner.execute(SendReportsJob.makeJobInfo(false));

                onFinished();

                /// Inform the listener about the advertisement ending
                if (playAdCallback != null) {
                    playAdCallback.onAdEnd(id,
                            (successfulView || percentViewed >= 80),
                            (value != null && value.equals("isCTAClicked")));
                    playAdCallback.onAdEnd(id);
                }

            } else if (placement.isIncentivized() && s.equals("successfulView")) {
                successfulView = true;
                if (!adRewarded) {
                    adRewarded = true;
                    if (playAdCallback != null) {
                        playAdCallback.onAdRewarded(id);
                    }
                }
            } else if (placement.isIncentivized() && s.startsWith("percentViewed")) {
                /// Parse out the value passed in as a String.
                String[] tokens = s.split(":");
                if (tokens.length == 2) {
                    String percentString = tokens[1];
                    percentViewed = Integer.parseInt(percentString);
                }
                if (!adRewarded && percentViewed >= 80) {
                    adRewarded = true;
                    if (playAdCallback != null) {
                        playAdCallback.onAdRewarded(id);
                    }
                }
            } else if ("open".equals(s) && playAdCallback != null) {
                if ("adClick".equals(value)) {
                    playAdCallback.onAdClick(id);
                } else if ("adLeftApplication".equals(value)) {
                    playAdCallback.onAdLeftApplication(id);
                }
            }
        } catch (DatabaseHelper.DBException ignored) {
            onError(new VungleException(DB_ERROR), id);
        }
    }

    @Override
    public void onError(VungleException exception, String placementId) {
        if (advertisement == null) {
            advertisement = repository.findValidAdvertisementForPlacement(placementId).get();
        }

        if (advertisement != null && exception.getExceptionCode() == RENDER_ERROR) {
            adLoader.dropCache(advertisement.getId());
            return;
        }

        if (advertisement != null
                && exception.getExceptionCode() != ALREADY_PLAYING_ANOTHER_AD//todo missing test for the code
                && exception.getExceptionCode() != OPERATION_CANCELED) {
            try {
                repository.saveAndApplyState(advertisement, placementId, Advertisement.ERROR);
            } catch (DatabaseHelper.DBException e) {
                exception = new VungleException(DB_ERROR);
            }
        }

        onFinished();

        if (playAdCallback != null) {
            playAdCallback.onError(placementId, exception);
        }
    }

    protected void onFinished() {
        playOperations.put(placementId, false);
    }

}

