package com.vungle.warren;

import androidx.annotation.IntDef;
import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.RestrictTo;

import com.google.gson.annotations.SerializedName;

import static com.vungle.warren.AdConfig.AdSize.VUNGLE_DEFAULT;

/**
 * Settings and Configuration Class for playAd Requests.
 * To prevent any issues, generate a new AdConfig object per playAd() request.
 */
public class AdConfig {

    public static final String FLAG_DIRECT_DOWNLOAD = "direct_download";

    /**
     * Bitmask which stores the configuration settings for this advertisement.
     */
    @SerializedName("settings")
    private int settings;

    @SerializedName("flexviewCloseSec")
    private int flexviewCloseSec;

    @SerializedName("ordinal")
    private int ordinal;

    @IntDef(flag = true, value = {MUTED, IMMEDIATE_BACK, IMMERSIVE, TRANSITION_ANIMATE})
    public @interface Settings {
    }

    /**
     * Start this advertisement muted.
     */
    public static final int MUTED = 1;

    /**
     * Allow the user to close the ad immediately after starting it. Usually there is a delay before
     * the X or skip is shown. The back button is the same as the X.
     */
    public static final int IMMEDIATE_BACK = 1 << 1;

    /**
     * Whether or not to use immersive mode. Defaults to true.
     */
    public static final int IMMERSIVE = 1 << 2;

    /**
     * Apply a fade-in/fade-out animation to the advertisement activity
     */
    public static final int TRANSITION_ANIMATE = 1 << 3;

    /**
     * When set, the advertisement will support apk direct download feature.
     */
    public static final int APK_DIRECT_DOWNLOAD = 1 << 5;

    /**
     * When set the SDK will check if it has an Ad that can be rendered in the specific size
     */
    @SerializedName("adSize")
    private AdSize adSize = VUNGLE_DEFAULT;

    /**
     * Enum for Ad Sizes.
     */
    @Keep
    public enum AdSize {

        /** should be used only along with {@link Vungle#getNativeAd(String, AdConfig, PlayAdCallback)} for IAB standard MREC size **/
        VUNGLE_MREC("mrec",300, 250),
        /** should be used only along with {@link Vungle#getNativeAd(String, AdConfig, PlayAdCallback)} for generic FlexFeed **/
        VUNGLE_DEFAULT("default",-1,-1),
        BANNER("banner",320,50),
        BANNER_SHORT("banner_short",300,50),
        BANNER_LEADERBOARD("banner_leaderboard",728, 90);

        private final int width, height;
        private final String name;

        AdSize(String name, int width, int height) {
            this.width = width;
            this.height = height;
            this.name = name;
        }

        public String getName() {
            return this.name;
        }

        public int getWidth() {
            return width;
        }

        public int getHeight() {
            return height;
        }

        public static AdSize fromName(String name) {
            for (AdSize e : AdSize.values()) {
                if (e.name.equals(name)) {
                    return e;
                }
            }
            return VUNGLE_DEFAULT;
        }

        public static boolean isDefaultAdSize(@NonNull AdSize adSize) {
            return adSize == VUNGLE_DEFAULT || adSize == VUNGLE_MREC;
        }

        public static boolean isBannerAdSize(@NonNull AdSize adSize) {
            return adSize == BANNER || adSize == BANNER_LEADERBOARD || adSize == BANNER_SHORT;
        }
    }

    /**
     * When set the SDK will check if it has an Ad that can be rendered in the specific rotation
     */
    @SerializedName("adOrientation")
    private @Orientation int adOrientation = AUTO_ROTATE;

    @IntDef({PORTRAIT, LANDSCAPE, AUTO_ROTATE, MATCH_VIDEO})
    public @interface Orientation {}

    public static final int PORTRAIT = 0;
    public static final int LANDSCAPE = 1;
    public static final int AUTO_ROTATE = 2;
    public static final int MATCH_VIDEO = 3;

    /**
     * returns the settings Bitmask which stores the configuration for this advertisement
     *
     * @return int
     */
    @RestrictTo(RestrictTo.Scope.LIBRARY)
    public int getSettings() {
        return settings;
    }

    /**
     * <p>Sets whether the video should start with its sound matching the
     * settings of your enclosing application <i>(the default</i> or muted.</p>
     * <p>
     * <p>Note that the user can still manually mute or unmute the sound during
     * playback.</p>
     *
     * @param isMuted <code>true</code> if the video should start with its
     *                audio settings matching those of your enclosing application;
     *                <code>false</code> if it should start muted.
     */
    public void setMuted(boolean isMuted) {
        if (isMuted) {
            settings = settings | MUTED;
        } else {
            settings = settings & ~MUTED;
        }
    }

    /**
     * <p>Sets whether the Android back button will be immediately enabled during the
     * video ad, or it will be inactive until the on screen close button appears
     * <i>(the default)</i>.</p>
     * <p>
     * <p>Once enabled, the Android back button allows the user to skip the video
     * ad and proceed to the post-roll if one exists; if the ad does not have a
     * post-roll the ad simply ends.</p>
     *
     * @param isBackButtonImmediatelyEnabled <code>true</code> if back button should
     *                                       be enabled before ad close button appears; otherwise, <code>false</code>
     */
    public void setBackButtonImmediatelyEnabled(boolean isBackButtonImmediatelyEnabled) {
        if (isBackButtonImmediatelyEnabled) {
            settings = settings | IMMEDIATE_BACK;
        } else {
            settings = settings & ~IMMEDIATE_BACK;
        }
    }

    /**
     * Sets whether
     * <a href="https://developer.android.com/training/system-ui/immersive.html" target="_blank">immersive mode</a>
     * will be enabled for KitKat (API level 19) and later devices.
     *
     * @param isImmersiveMode <code>true</code> if immersive mode will be
     *                        enabled for KitKat+ devices; otherwise, <code>false</code>.
     */
    public void setImmersiveMode(boolean isImmersiveMode) {
        if (isImmersiveMode) {
            settings = settings | IMMERSIVE;
        } else {
            settings = settings & ~IMMERSIVE;
        }
    }

    /**
     * <p>Sets whether the video transition animation should be enabled or disabled.
     * The default is enabled (a fade-in/fade-out animation).</p>
     *
     * @param isTransitionAnimationEnabled <code>true</code> if the
     *                                     video transition animation should be enabled;
     *                                     <code>false</code> if it should be disabled.
     */
    public void setTransitionAnimationEnabled(boolean isTransitionAnimationEnabled) {
        if (isTransitionAnimationEnabled) {
            settings = settings | TRANSITION_ANIMATE;
        } else {
            settings = settings & ~TRANSITION_ANIMATE;
        }
    }

    /**
     * @return The number of advertisements set by the developer that have played before this ad
     */
    public int getOrdinal() {
        return ordinal;
    }

    /**
     * Inlcuding this advertisement, the number of advertisements that have been played
     * this session. This is not tracked internally because apps can have multiple ad
     * providers.
     *
     * @param ordinal Count in integers
     */
    public void setOrdinal(int ordinal) {
        this.ordinal = ordinal;
    }

    /**
     * Gets how long a flexview ad is visible for before it closes automatically. Default value of 0 (No time limit).
     *
     * @return Value in seconds
     */
    public int getFlexViewCloseTime() {
        return flexviewCloseSec;
    }

    /**
     * Sets how long a flexview ad is visible for before it closes automatically. Default value of 0 (No time limit).
     *
     * @param seconds Value in seconds
     */
    public void setFlexViewCloseTime(int seconds) {
        flexviewCloseSec = seconds;
    }

    /**
     * Returns AdSize
     *
     * @return AdSize
     */
    public AdSize getAdSize() {
        return adSize == null ? VUNGLE_DEFAULT : adSize;
    }

    /**
     * Sets the AdSize, currently only supported for FlexFeed and MREC ad units
     * should be called immediately before getNativeAd
     *
     * @param AdSize for the Ad
     */
    public void setAdSize(AdSize adSize) {
        this.adSize = adSize;
    }

    /**
     * Returns AdOrientation
     *
     * @return AdOrientation
     */
    public @Orientation int getAdOrientation() {
        return adOrientation;
    }

    /**
     * Sets the AdOrientation
     *
     * @param AdOrientation for the Ad
     */
    public void setAdOrientation(@Orientation int adOrientation) {
        this.adOrientation = adOrientation;
    }
}
