package com.vungle.warren;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.res.Configuration;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;
import android.view.Window;
import android.view.WindowManager;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;

import com.vungle.warren.error.VungleException;
import com.vungle.warren.ui.CloseDelegate;
import com.vungle.warren.ui.OrientationDelegate;
import com.vungle.warren.ui.contract.AdContract;
import com.vungle.warren.ui.contract.AdContract.AdvertisementBus;
import com.vungle.warren.ui.state.BundleOptionsState;
import com.vungle.warren.ui.state.OptionsState;
import com.vungle.warren.ui.view.FullAdWidget;

import java.util.concurrent.atomic.AtomicBoolean;

import static com.vungle.warren.error.VungleException.AD_UNABLE_TO_PLAY;
import static com.vungle.warren.error.VungleException.ALREADY_PLAYING_ANOTHER_AD;

/**
 * The Activity that the Vungle SDK uses to render advertisement content. This single activity can
 * start vungle_local and vungle_mraid advertisement types.
 */
public abstract class AdActivity extends Activity {

    public static final String PLACEMENT_EXTRA = "placement";

    private static final String TAG = "VungleActivity";
    public static final String PRESENTER_STATE = "presenter_state";

    /**
     * Eventbus to provide the presenter. Subscribers of this event bus use the lifecycle events
     * to update their own state.
     */
    private static AdContract.AdvertisementPresenter.EventListener bus;

    /**
     * The advertisement presenter does the heavy lifting for playing an advertisement. It has the
     * business logic to determine where to find the advertisement assets, manages creating the
     * report, and tracking state of the advertisement metadata. It needs to be attached to the view
     * in order to manage the view state.
     */
    @Nullable
    private AdContract.AdvertisementPresenter presenter;

    /**
     * Receiver used to handle events from outside the Activity, such as when the publisher calls
     * <code>close()</code> on an advertisement.
     */
    private BroadcastReceiver broadcastReceiver;

    /**
     * Placement identifier for which ad is going to be played.
     */
    private String placementId;
    private PresentationFactory presenterFactory;
    private OptionsState state;

    private AtomicBoolean pendingStart = new AtomicBoolean(false);
    private boolean started = false;
    private boolean resumed = false;

    /**
     * Provide an event listener for any event that happens in any instance of VungleActivity.
     *
     * @param listener Listener that will receive callbacks whenever events are reported.
     */
    public static void setEventListener(AdContract.AdvertisementPresenter.EventListener listener) {
        bus = listener;
    }

    @VisibleForTesting
    protected static AdContract.AdvertisementPresenter.EventListener getEventListener() {
        return bus;
    }

    @SuppressLint("SetJavaScriptEnabled")
    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        requestWindowFeature(Window.FEATURE_NO_TITLE);

        getWindow().setFlags(
                WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED,
                WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED);

        FullAdWidget fullAdWidget;
        placementId = getIntent().getStringExtra(PLACEMENT_EXTRA);

        ServiceLocator serviceLocator = ServiceLocator.getInstance(this);
        VungleStaticApi vungleStaticApi = serviceLocator.getService(VungleStaticApi.class);

        /// Initialize
        if (!vungleStaticApi.isInitialized() || bus == null || TextUtils.isEmpty(placementId)) {
            // Recovered from crash, deep sleep or else - this should not happen in normal life
            // SDK is not in operable state, BUS should we reworked and should not be set as static listener
            // instead we might use LocalBroadcastReceiver and let Vungle class to listen it
            // and dispatch event to appropriate listeners
            finish();
            return;
        }

        try {
            fullAdWidget = new FullAdWidget(this, getWindow());
        } catch (InstantiationException e) {
            deliverError(AD_UNABLE_TO_PLAY, placementId);
            finish();

            if (BuildConfig.DEBUG) {
                throw new RuntimeException(e);
            }
            return;
        }

        presenterFactory = serviceLocator.getService(PresentationFactory.class);

        state = savedInstanceState == null
                ? null
                : (OptionsState) savedInstanceState.getParcelable(PRESENTER_STATE);

        presenterFactory.getFullScreenPresentation(
                this,
                placementId,
                fullAdWidget,
                state,
                new CloseDelegate() {
                    @Override
                    public void close() {
                        finish();
                    }
                },
                new OrientationDelegate() {
                    @Override
                    public void setOrientation(int orientation) {
                        setRequestedOrientation(orientation);
                    }
                },
                savedInstanceState,
                fullscreenCallback
        );

        setContentView(fullAdWidget, fullAdWidget.getLayoutParams());

        connectBroadcastReceiver();
    }

    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);
        String oldPlacement = getIntent().getStringExtra(PLACEMENT_EXTRA);
        String newPlacement = intent.getStringExtra(PLACEMENT_EXTRA);
        if (oldPlacement != null && newPlacement != null && !oldPlacement.equals(newPlacement)) {
            Log.d(TAG, "Tried to play another placement " + newPlacement + " while playing " + oldPlacement);
            deliverError(ALREADY_PLAYING_ANOTHER_AD, newPlacement);
        }
    }

    private void deliverError(int code, String placementId) {
        if (bus != null) {
            bus.onError(new VungleException(code), placementId);
        }
    }

    private void connectBroadcastReceiver() {
        broadcastReceiver = new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                String command = intent.getStringExtra(AdvertisementBus.COMMAND);
                switch (command) {
                    case AdvertisementBus.CLOSE_FLEX: {
                        String placementID = intent.getStringExtra(AdvertisementBus.PLACEMENT);
                        if (presenter != null) {
                            presenter.handleExit(placementID);
                        }
                        break;
                    }
                    case AdvertisementBus.STOP_ALL: {
                        finish();
                        break;
                    }
                    default:
                        throw new IllegalArgumentException("No such command " + command);
                }
            }
        };

        LocalBroadcastManager.getInstance(getApplicationContext()).registerReceiver(broadcastReceiver, new IntentFilter(AdvertisementBus.ACTION));
    }

    @Override
    public void onWindowFocusChanged(boolean hasFocus) {
        super.onWindowFocusChanged(hasFocus);
        if (hasFocus) {
            start();
        } else {
            stop();
        }
    }

    private void start() {
        if (presenter == null) {
            pendingStart.set(true);
        } else if (!started && resumed && hasWindowFocus()) {
            presenter.start();
            started = true;
        }
    }

    private void stop() {
        if (presenter != null && started) {
            int flag = (isChangingConfigurations() ? AdContract.AdStopReason.IS_CHANGING_CONFIGURATION : 0)
                    | (isFinishing() ? AdContract.AdStopReason.IS_AD_FINISHING : 0);
            presenter.stop(flag);
            started = false;
        }
        pendingStart.set(false);
    }

    @Override
    protected void onResume() {
        super.onResume();
        resumed = true;
        start();
    }

    @Override
    protected void onPause() {
        super.onPause();
        resumed = false;
        stop();
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);

        // Checks the orientation of the screen
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            Log.d(TAG, "landscape");
        } else if (newConfig.orientation == Configuration.ORIENTATION_PORTRAIT) {
            Log.d(TAG, "portrait");
        }
        if (presenter != null) {
            presenter.onViewConfigurationChanged();
        }
    }

    @SuppressLint("ResourceType")
    @Override
    public void onBackPressed() {
        /// Only allow the advertisement to finish when back is pressed if the user has configured it.
        if (presenter != null) {
            presenter.handleExit(null);
        }
    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        Log.d(TAG, "onSaveInstanceState");

        BundleOptionsState optionsState = new BundleOptionsState();
        if (presenter != null) {
            presenter.generateSaveState(optionsState);
            outState.putParcelable(PRESENTER_STATE, optionsState);
        }

        if (presenterFactory != null) {
            presenterFactory.saveState(outState);
        }

        super.onSaveInstanceState(outState);
    }

    @Override
    protected void onRestoreInstanceState(Bundle savedInstanceState) {
        super.onRestoreInstanceState(savedInstanceState);
        Log.d(TAG, "onRestoreInstanceState(" + savedInstanceState + ")");

        if (savedInstanceState != null && presenter != null) {
            presenter.restoreFromSave((OptionsState) savedInstanceState.getParcelable(PRESENTER_STATE));
        }
    }

    @Override
    public void setRequestedOrientation(int requestedOrientation) {
        if (canRotate()) {
            super.setRequestedOrientation(requestedOrientation);
        }
    }

    protected abstract boolean canRotate();

    @Override
    protected void onDestroy() {
        LocalBroadcastManager.getInstance(getApplicationContext()).unregisterReceiver(broadcastReceiver);

        if (presenter != null) {
            presenter.detach(isChangingConfigurations());
        } else if (presenterFactory != null) {
            presenterFactory.destroy();
            presenterFactory = null;
            if (bus != null) {
                bus.onError(new VungleException(VungleException.OPERATION_CANCELED), placementId);
            }
        }

        super.onDestroy();
    }

    private AdvertisementPresentationFactory.FullScreenCallback fullscreenCallback = new AdvertisementPresentationFactory.FullScreenCallback() {
        @Override
        public void onResult(@Nullable Pair<AdContract.AdView, AdContract.AdvertisementPresenter> result, @Nullable VungleException error) {
            //advertisement no longer exists kill activity
            if (result == null || error != null) {
                presenterFactory = null;//needs to be set to null to not generate OPERATION_CANCELED error from finish() below
                deliverError(AD_UNABLE_TO_PLAY, placementId);
                finish();
                return;
            }

            presenter = result.second;
            presenter.setEventListener(bus);

            AdContract.AdView adView = result.first;

            presenter.attach(adView, state);

            if (pendingStart.getAndSet(false)) {
                start();
            }
        }
    };
}
