package com.voxeet.sdk.views;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.os.Handler;
import android.support.annotation.MainThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.widget.FrameLayout;

import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.MediaStreamType;
import com.voxeet.sdk.media.EglBaseRefreshEvent;
import com.voxeet.sdk.utils.Opt;
import com.voxeet.video.R;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.EglBase;
import org.webrtc.RendererCommon;

import java.util.concurrent.CopyOnWriteArrayList;

/**
 * A view that helps with integration of attaching and detaching `MediaStream` into applications.
 */
public class VideoView extends FrameLayout implements RendererCommon.RendererEvents {
    private static final int SCALE_FIT = 0;
    private static final int SCALE_FILL = 1;
    private static final int SCALE_BALANCED = 2;

    private final String TAG = VideoView.class.getSimpleName();

    private boolean mIsAttached = false;

    private CopyOnWriteArrayList<RendererCommon.RendererEvents> mEventsListeners;
    /**
     * The Voxeet renderer.
     */
    //protected SurfaceViewRenderer mRenderer;
    protected VoxeetRenderer mRenderer;

    private String mPeerId;

    private MediaStream mMediaStream;

    private boolean shouldMirror = false;
    private boolean showFlip = false;
    private int mScaleType = SCALE_FILL;
    private Handler mHandler;

    // this view holds the reference to the renderer AND the flip image
    private View mInternalVideoView;
    private View mFlip;
    private RoundedFrameLayout mCornerRadiusView;
    private boolean mIsCircle;
    private float mCornerRadius;
    private boolean enableRefreshEglBase = false;

    private boolean layoutOnAttached = false;
    private boolean waitForFirstFrame = true;

    private float translateX;
    private float translateY;

    /**
     * Instantiates a new Video view.
     *
     * @param context the context
     */
    public VideoView(Context context) {
        super(context);

        init();
    }

    /**
     * Instantiates a new Video view.
     *
     * @param context the context
     * @param attrs   the attrs
     */
    public VideoView(Context context, AttributeSet attrs) {
        super(context, attrs);

        updateAttrs(attrs);

        init();
    }

    @Override
    protected void onAttachedToWindow() {
        try {
            EventBus.getDefault().register(this);
        } catch (Exception e) {
            e.printStackTrace();
        }

        layoutOnAttached = false;

        if (null != mPeerId && null != mMediaStream) {
            attach(mPeerId, mMediaStream);
        }

        EglBase.Context context = Opt.of(ViewFactory.get()).then(ViewFactory.MediaProvider::getEglContext).orNull();
        if (null != context) initView(context);

        super.onAttachedToWindow();
    }

    @Override
    protected void onDetachedFromWindow() {

        if (mPeerId != null) {
            String peerId = mPeerId;
            MediaStream stream = mMediaStream;
            unAttach();
            mMediaStream = stream;
            mPeerId = peerId;

            mRenderer.setFirstFrameRendered(false);
            mRenderer.setVisibility(View.GONE);
            waitForFirstFrame = true;
        }

        releaseView();

        EventBus.getDefault().unregister(this);
        super.onDetachedFromWindow();
    }

    private void init() {
        mHandler = new Handler();
        mEventsListeners = new CopyOnWriteArrayList<>();
    }


    private void updateAttrs(AttributeSet attrs) {
        TypedArray attributes = getContext().obtainStyledAttributes(attrs, R.styleable.VideoView);
        shouldMirror = attributes.getBoolean(R.styleable.VideoView_mirrored, false);
        showFlip = attributes.getBoolean(R.styleable.VideoView_showFlip, false);

        mScaleType = attributes.getInteger(R.styleable.VideoView_scaleType, SCALE_FILL);

        mIsCircle = attributes.getBoolean(R.styleable.VideoView_circle, false);
        mCornerRadius = attributes.getDimension(R.styleable.VideoView_cornerRadius, 0);

        attributes.recycle();
    }

    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        super.onLayout(changed, left, top, right, bottom);

        createRendererIfNeeded();
        setSurfaceViewRenderer();
    }

    /**
     * Changes the view visibility. It updates the flip feature at the same time.
     *
     * @param visibility the new visibility
     */
    @Override
    public void setVisibility(int visibility) {
        super.setVisibility(visibility);

        updateFlip();
    }

    /**
     * Adds a new `RendererEvents` listener to the current `VideoView`.
     * <p>
     * _Warning: catches are not made when listeners are called._
     *
     * @param listener a non null listener
     */
    public void addListener(@NonNull RendererCommon.RendererEvents listener) {
        synchronized (mEventsListeners) {
            if (!mEventsListeners.contains(listener)) {
                mEventsListeners.add(listener);
            }
        }
    }

    /**
     * Removes a specific listener from the list of internal `RendererEvents` listeners.
     *
     * @param listener a non null listener to remove
     */
    public void removeListener(@NonNull RendererCommon.RendererEvents listener) {
        synchronized (mEventsListeners) {
            if (mEventsListeners.contains(listener)) {
                mEventsListeners.remove(listener);
            }
        }
    }

    /**
     * Clears the list of listeners from this view. Use it while cleaning it.
     */
    public void clearListeners() {
        synchronized (mEventsListeners) {
            mEventsListeners.clear();
        }
    }

    /**
     * Changes the video scale to fit it. It also conserves an aspect ratio.
     */
    @MainThread
    public void setVideoFit() {
        if (SCALE_FIT != mScaleType) resetDelta();
        mScaleType = SCALE_FIT;

        setSurfaceViewRenderer();
        requestLayout();
    }

    /**
     * Changes the video scale to fill it. It also conserves an aspect ratio.
     */
    @MainThread
    public void setVideoFill() {
        if (SCALE_FILL != mScaleType) resetDelta();
        mScaleType = SCALE_FILL;
        setSurfaceViewRenderer();
        requestLayout();
    }

    /**
     * Changes the video scale to balance it. It also conserves an aspect ratio.
     */
    @MainThread
    public void setVideoBalanced() {
        if (SCALE_BALANCED != mScaleType) resetDelta();
        mScaleType = SCALE_BALANCED;
        setSurfaceViewRenderer();
        requestLayout();
    }

    /**
     * Sets the `flip` to show the flippable icon in the `VideoView`.
     *
     * @param flip Flag indicating if the view will show the flip icon.
     */
    @MainThread
    public void setFlip(boolean flip) {
        showFlip = flip;

        updateFlip();
    }

    /**
     * Mirrors a self video (or not).
     *
     * @param mirror Flag indicating if the view will be mirrored or displayd normally.
     */
    @MainThread
    public void setMirror(boolean mirror) {
        shouldMirror = mirror;
        if (null != mRenderer) {
            mRenderer.setMirror(mirror);
        }
    }

    private void updateFlip() {
        if (null != mFlip && null != mRenderer) {
            if (showFlip && isAttached() && mRenderer.isFirstFrameRendered()) {
                mFlip.setVisibility(View.VISIBLE);
            } else {
                mFlip.setVisibility(View.GONE);
            }
        }
    }

    /**
     * Sets surface view renderer.
     */
    public void setSurfaceViewRenderer() {
        RendererCommon.ScalingType type = getScalingType();

        if (null != mRenderer) {
            boolean update = shouldMirror != mRenderer.isMirror();
            update |= !type.equals(mRenderer.getScalingType());

            if (update) {
                this.mRenderer.setEnableHardwareScaler(true);
                this.mRenderer.setScalingType(type);
                this.mRenderer.forceRecalculateWidthHeight();

                this.mRenderer.setMirror(shouldMirror);
            }
        }
    }

    /**
     * Gets the current scaling type of this view.
     *
     * @return a default or set value.
     */
    @NonNull
    public RendererCommon.ScalingType getScalingType() {
        switch (mScaleType) {
            case SCALE_BALANCED:
                return RendererCommon.ScalingType.SCALE_ASPECT_BALANCED;
            case SCALE_FILL:
                return RendererCommon.ScalingType.SCALE_ASPECT_FILL;
            default:
                return RendererCommon.ScalingType.SCALE_ASPECT_FIT;
        }
    }

    private void setNextScalingType() {
        switch (mScaleType) {
            case SCALE_BALANCED:
                setVideoFill();
                break;
            case SCALE_FILL:
                setVideoFit();
                break;
            default:
                setVideoFill();
                break;
        }
    }

    /**
     * Releases the renderer.
     */
    public void release() {
        if (null != mRenderer) {
            unAttach();
            mRenderer.release();
            try {
                removeView(mRenderer);
                mRenderer = null;
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Reinitializes the view.
     * <p>
     * A false value, warns that something is incorrect. The renderer initialization can be called without a proper release call.
     *
     * @return if the view was reinitialized.
     */
    public boolean reinit() {
        try {
            boolean hasMedia = Opt.of(ViewFactory.get()).then(ViewFactory.MediaProvider::hasMedia).or(false);
            EglBase.Context context = Opt.of(ViewFactory.get()).then(ViewFactory.MediaProvider::getEglContext).orNull();
            if (null != mRenderer && hasMedia) {
                this.mRenderer.init(context, this);
            }
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    /**
     * the renderer.
     *
     * @return the renderer
     */
    public VoxeetRenderer getRenderer() {
        return mRenderer;
    }

    /**
     * Informs about attachment.
     *
     * @return the boolean.
     */
    public boolean isAttached() {
        return null != mPeerId;
    }

    /**
     * Attaches the stream associated with the `peerId` to the `videoView`.
     * If the previous stream is attached, this method automatically detaches it.
     *
     * @param peerId      the peer id
     * @param mediaStream the media stream
     */
    public void attach(@NonNull String peerId, @Nullable MediaStream mediaStream) {
        boolean wasVisible = null != mRenderer && mRenderer.getVisibility() == View.VISIBLE;
        boolean is_new_participant = !peerId.equals(mPeerId);

        if (isAttached())
            detach(is_new_participant);

        if (!isAttached() && null != mediaStream && (mediaStream.videoTracks().size() > 0 || com.voxeet.android.media.MediaStreamType.ScreenShare.equals(mediaStream.getType()))) {
            mPeerId = peerId;

            mMediaStream = mediaStream;

            createRendererIfNeeded();

            if (null != mRenderer) {
                if (!wasVisible) {
                    mRenderer.setFirstFrameRendered(false);
                    waitForFirstFrame = true;
                } else {
                    waitForFirstFrame = false;
                    mRenderer.setFirstFrameRendered(true);
                }
                boolean result = ViewFactory.get().attachMediaStream(mediaStream, mRenderer);

                setVisibility(View.VISIBLE);

                if (waitForFirstFrame) {
                    mRenderer.setVisibility(View.INVISIBLE);
                } else {
                    mRenderer.setVisibility(View.VISIBLE);
                }
                forceLayout();
                mRenderer.forceLayout();
                requestLayout();
                mRenderer.requestLayout();
            }

            updateFlip();
        }
    }

    /**
     * Detaches the stream from the `VideoView`.
     */
    public void unAttach() {
        detach(false);
    }

    private void detach(boolean invalidate) {
        ViewFactory.get().unAttachMediaStream(mRenderer);
        mPeerId = null;
        mMediaStream = null;

        if (null != mRenderer && invalidate) {
            mRenderer.setFirstFrameRendered(false);
            mRenderer.setVisibility(View.GONE);
            waitForFirstFrame = true;
            updateFlip();
        }
    }

    /**
     * Gets the peer ID of the currently attached conference participant.
     *
     * @return the peer ID.
     */
    public String getPeerId() {
        return mPeerId;
    }

    /**
     * Gets the current screen sharing status.
     *
     * @return true or false.
     */
    @Deprecated
    public boolean isScreenShare() {
        return mMediaStream != null && MediaStreamType.ScreenShare.equals(mMediaStream.getType());
    }

    /**
     * Informs about the current `MediaStream` attached to the `VideoView`.
     *
     * @return a nullable instance of `MediaStream`.
     */
    @Nullable
    public MediaStreamType current() {
        return null != mMediaStream ? mMediaStream.getType() : null;
    }

    /**
     * Gets the current type of a screen sharing. Informs if a video is shared or not.
     *
     * @return true or false.
     */
    public boolean hasVideo() {
        return mMediaStream != null && mMediaStream.videoTracks().size() > 0;
    }

    @Override
    public void onFirstFrameRendered() {
        //make sure we are sending event on the main looper
        mHandler.post(new Runnable() {
            @Override
            public void run() {
                waitForFirstFrame = false;
                if (/*waitForFirstFrame && */null != mRenderer) {
                    mRenderer.setVisibility(View.VISIBLE);
                }
                setSurfaceViewRenderer();

                updateFlip();

                synchronized (mEventsListeners) {
                    for (RendererCommon.RendererEvents events_listener : mEventsListeners) {
                        events_listener.onFirstFrameRendered();
                    }
                }
            }
        });
    }

    @Override
    public void onFrameResolutionChanged(int videoWidth, int videoHeight, int rotation) {
        synchronized (mEventsListeners) {
            for (RendererCommon.RendererEvents events_listener : mEventsListeners) {
                events_listener.onFrameResolutionChanged(videoWidth, videoHeight, rotation);
            }
        }
    }

    private void createRendererIfNeeded() {
        boolean hasMedia = Opt.of(ViewFactory.get()).then(ViewFactory.MediaProvider::hasMedia).or(false);
        EglBase.Context context = Opt.of(ViewFactory.get()).then(ViewFactory.MediaProvider::getEglContext).orNull();
        if (null == mRenderer && hasMedia) {
            //don't setup if no context
            if (null != context) {
                initView(context);
            }
        } else if (null != mRenderer) {
            LayoutParams param = new LayoutParams(
                    LayoutParams.WRAP_CONTENT,
                    LayoutParams.WRAP_CONTENT,
                    Gravity.CENTER);

            //post new layout params just to prevent issues
            mHandler.post(new Runnable() {
                @Override
                public void run() {
                    if (null != mRenderer && !layoutOnAttached) {
                        layoutOnAttached = true;
                        mRenderer.setLayoutParams(param);
                    }
                }
            });
        }
    }

    private void removeRender() {
        if (mRenderer != null) {
            try {
                mRenderer.release();
            } catch (Exception e) {

            }

            //removeView(mRenderer);
            //mRenderer = null;
        }
    }

    /**
     * Gets the current bitmap to create thumbnails for a video view.
     * <p>
     * _Warning: this method consumes time and resources._
     *
     * @return the bitmap or a null value a renderer is not attached.
     */
    @Nullable
    public Bitmap getBitmap() {
        if (null == mRenderer) return null;
        return mRenderer.getBitmap();
    }

    /**
     * Gets the current bitmap for this video view to create thumbnails.
     * <p>
     * _Warning: this method consumes time and resources._
     *
     * @param bitmap a bitmap to draw onto
     * @return the bitmap or a null value if a renderer is not attached.
     */
    @Nullable
    public Bitmap getBitmap(@NonNull Bitmap bitmap) {
        if (null == mRenderer) return null;
        return mRenderer.getBitmap(bitmap);
    }

    /**
     * Get the current bitmap for the video view to take thumbnails.
     * <p>
     * _Warning: this method consumes time and ressources._
     *
     * @param width  the width to make the bitmap fit into
     * @param height the height to make the bitmap fit into
     * @return the bitmap or a null value if a renderer is not attached.
     */
    @Nullable
    public Bitmap getBitmap(int width, int height) {
        if (null == mRenderer) return null;
        return mRenderer.getBitmap(width, height);
    }

    /**
     * Changes the `VideoView` to a circle one. Use it in Avatar view for the best experience.
     *
     * @param isCircle the circle state to change into
     * @return the current instance.
     */
    @NonNull
    public VideoView setIsCircle(boolean isCircle) {
        mIsCircle = isCircle;
        if (null != mCornerRadiusView) mCornerRadiusView.setIsCircle(isCircle);
        return this;
    }

    /**
     * Changes the current `VideoView` corner radius. Use it in an overlay or a card for the best expirience.
     *
     * @param cornerRadius the new corner radius to change into
     * @return the current instance.
     */
    @NonNull
    public VideoView setCornerRadius(float cornerRadius) {
        mCornerRadius = cornerRadius;
        if (null != mCornerRadiusView) mCornerRadiusView.setCornerRadius(mCornerRadius);
        return this;
    }

    /**
     * Activates or deactivates self healing `EglBase` context.
     *
     * @param state the new state of the safe healing management
     */
    public void activateSafeRefreshEglBase(boolean state) {
        this.enableRefreshEglBase = state;
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(EglBaseRefreshEvent event) {
        if (!enableRefreshEglBase || null == ViewFactory.get() || null == mRenderer) {
            return;
        }

        if (Opt.of(ViewFactory.get()).then(ViewFactory.MediaProvider::hasMedia).or(false)) {
            //release the surface
            removeRender();
            String peerId = null;
            MediaStream stream = mMediaStream;

            //is was attached, save and release
            if (isAttached()) {
                peerId = getPeerId();
                stream = mMediaStream;
                unAttach();
            }

            //if was attached, reattach
            if (!TextUtils.isEmpty(peerId) && null != stream) {
                attach(peerId, stream);
            }
        }
    }

    private float originalX;
    private float originalY;
    private float tmpDx = 0;
    private float tmpDy = 0;

    public void resetDelta() {
        if (null == mRenderer) return;

        translateX = translateY = 0;
        tmpDx = tmpDy = 0;

        mRenderer.setDeltaX(0);
        mRenderer.setDeltaY(0);
    }

    public void commitDelta() {
        if (null == mRenderer) return;

        translateX += tmpDx;
        translateY += tmpDy;
        tmpDx = tmpDy = 0;
    }

    public void startTouch(@NonNull MotionEvent event) {
        if (null == mRenderer) return;

        originalX = event.getX();
        originalY = event.getY();
    }

    public void applyDelta(@NonNull MotionEvent event) {
        if (null == mRenderer) return;

        float screenResolutionDx = -(event.getX() - originalX);
        float screenResolutionDy = (event.getY() - originalY);
        tmpDx = screenResolutionDx;
        tmpDy = screenResolutionDy;


        float scaleX = mRenderer.getRendererScaleX();
        float scaleY = mRenderer.getRendererScaleY();

        float frameResolutionXScaledToScreen = 0;
        float frameResolutionYScaledToScreen = 0;

        if (scaleX > scaleY) {
            frameResolutionXScaledToScreen = mRenderer.getRotatedFrameWidth() * getHeight() / mRenderer.getRotatedFrameHeight();
            frameResolutionYScaledToScreen = mRenderer.getRotatedFrameHeight();// * getHeight() / mRenderer.getRotatedFrameHeight();
        } else {
            frameResolutionXScaledToScreen = mRenderer.getRotatedFrameWidth();// * getWidth() / mRenderer.getRotatedFrameWidth();
            frameResolutionYScaledToScreen = mRenderer.getRotatedFrameHeight() * getWidth() / mRenderer.getRotatedFrameWidth();
        }

        float translationX = translateX + tmpDx;
        float translationY = translateY + tmpDy;

        float halfX = (frameResolutionXScaledToScreen - getWidth() * 3.f / 2) / 2;
        float halfY = (frameResolutionYScaledToScreen - getHeight() * 3.f / 2) / 2;

        if (translationX < -halfX) {
            translationX = -halfX;
            tmpDx = translationX - translateX;
        }
        if (translationX > halfX) {
            translationX = halfX;
            tmpDx = translationX - translateX;
        }

        if (translationY < -halfY) {
            translationY = -halfY;
            tmpDy = translationY - translateY;
        }
        if (translationY > halfY) {
            translationY = halfY;
            tmpDy = translationY - translateY;
        }

        if (null != mRenderer) {
            mRenderer.setDeltaX(translationX * 1.f / getWidth());
            mRenderer.setDeltaY(translationY * 1.f / getHeight());
        }
    }


    private boolean isInit = false;

    private void initView(EglBase.Context context) {
        if (isInit) {
            return;
        }

        isInit = true;

        if (null == mRenderer) {
            mInternalVideoView = LayoutInflater.from(getContext())
                    .inflate(R.layout.voxeet_internal_videoview, this, false);
            addView(mInternalVideoView);

            mCornerRadiusView = mInternalVideoView.findViewById(R.id.voxeet_videoview_cornerradius);
            mRenderer = new VoxeetRenderer(getContext());
            mFlip = mInternalVideoView.findViewById(R.id.voxeet_videoview_flip);

            mCornerRadiusView.addView(mRenderer);

            mCornerRadiusView.setIsCircle(mIsCircle);
            mCornerRadiusView.setCornerRadius(mCornerRadius);
        }

        updateFlip();
        mRenderer.init(context, this);

        mRenderer.setDeltaX(translateX);
        mRenderer.setDeltaY(translateY);
        mRenderer.setScalingType(getScalingType());
    }

    private void releaseView() {
        isInit = false;

        if (null != mRenderer) mRenderer.release();
    }
}