package com.voxeet.sdk.views;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Point;
import android.graphics.SurfaceTexture;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.Display;
import android.view.TextureView;
import android.view.WindowManager;

import org.webrtc.EglBase;
import org.webrtc.GlRectDrawer;
import org.webrtc.Logging;
import org.webrtc.RenderFrameStateListener;
import org.webrtc.RendererCommon;
import org.webrtc.RendererState;
import org.webrtc.SafeRenderFrameEglRenderer;
import org.webrtc.ThreadUtils;
import org.webrtc.VideoFrame;
import org.webrtc.VideoSink;

import java.util.concurrent.CountDownLatch;

/**
 * Implements org.webrtc.VideoRenderer.Callbacks by displaying a video stream on a `SurfaceView`. The `renderFrame()` is asynchronous to avoid blocking the calling thread. This class is thread-safe and handles access from potentially four different threads:
 * - Interaction from the main app in init, release, setMirror, and setScalingtype.
 * - Interaction from C++ rtc::VideoSinkInterface in renderFrame.
 * - Interaction from the Activity lifecycle in surfaceCreated, surfaceChanged, and surfaceDestroyed.
 * - Interaction with the layout framework in onMeasure and onSizeChanged.
 */
public class VoxeetRenderer extends TextureView
        implements TextureView.SurfaceTextureListener, VideoSink, RenderFrameStateListener {
    private static final String TAG = "VoxeetRenderer";

    private Point size = new Point();

    // Cached resource name.
    private final String resourceName;
    private final RendererCommon.VideoLayoutMeasure videoLayoutMeasure =
            new RendererCommon.VideoLayoutMeasure();
    private final SafeRenderFrameEglRenderer eglRenderer;
    private Handler mHandler;
    private boolean pendingLayout = false;

    // Callback for reporting renderer events. Read-only after initilization so no lock required.
    private RendererCommon.RendererEvents rendererEvents;

    private final Object layoutLock = new Object();
    private boolean isRenderingPaused = false;
    private boolean isFirstFrameRendered;
    private int rotatedFrameWidth;
    private int rotatedFrameHeight;
    private int frameRotation;

    // Accessed only on the main thread.
    private boolean enableFixedSize;
    private int surfaceWidth;
    private int surfaceHeight;
    private boolean isEglRendererInitialized;

    private RendererCommon.ScalingType setScalingType;
    private int delayedPostTimeout = 1000;
    private float translateX = 0;
    private float translateY = 0;
    private boolean attached;

    /**
     * Standard View constructor. Call `init()` first to order to render something.
     */
    public VoxeetRenderer(Context context) {
        super(context);
        isEglRendererInitialized = false;
        this.resourceName = getResourceName();
        eglRenderer = new SafeRenderFrameEglRenderer(VoxeetRenderer.this::getSurfaceTexture, resourceName, this);
        setSurfaceTextureListener(this);

        mHandler = new Handler(Looper.getMainLooper());
    }

    /**
     * Standard View constructor. Call `init()` first to order to render something.
     */
    public VoxeetRenderer(Context context, AttributeSet attrs) {
        super(context, attrs);
        isEglRendererInitialized = false;
        this.resourceName = getResourceName();
        eglRenderer = new SafeRenderFrameEglRenderer(VoxeetRenderer.this::getSurfaceTexture, resourceName, this);
        setSurfaceTextureListener(this);

        mHandler = new Handler(Looper.getMainLooper());
    }

    /**
     * Initialize this class sharing resources with `sharedContext`. It is allowed to call `init()` to reinitialize the renderer after previous `init()` or `release()` cycle.
     */
    public void init(EglBase.Context sharedContext, RendererCommon.RendererEvents rendererEvents) {
        init(sharedContext, rendererEvents, EglBase.CONFIG_RGBA, new GlRectDrawer());
    }

    /**
     * Initialize this class sharing resources with `sharedContext`.
     * The custom `drawer` is used for drawing frames on the EGLSurface. This class is responsible for calling `release()` on a `drawer`. It is allowed to call `init()` to reinitialize the renderer after previous `init()` or `release()` cycle.
     */
    public void init(final EglBase.Context sharedContext,
                     RendererCommon.RendererEvents rendererEvents, final int[] configAttributes,
                     RendererCommon.GlDrawer drawer) {
        synchronized (eglRenderer) {
            if (isEglRendererInitialized) return;

            ThreadUtils.checkIsOnMainThread();
            this.rendererEvents = rendererEvents;
            synchronized (layoutLock) {
                isFirstFrameRendered = false;
                rotatedFrameWidth = 0;
                rotatedFrameHeight = 0;
                frameRotation = 0;
            }
            eglRenderer.init(sharedContext, configAttributes, drawer);
            isEglRendererInitialized = true;
        }
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        setSurfaceTextureListener(this);

        if (null != eglRenderer) eglRenderer.setAttached(true);

        WindowManager wm = (WindowManager) getContext().getSystemService(Context.WINDOW_SERVICE);
        if (wm != null) {
            Display display = wm.getDefaultDisplay();

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                display.getRealSize(size);
            } else {
                display.getSize(size);
            }
        }
    }

    @Override
    protected void onDetachedFromWindow() {
        if (null != eglRenderer) eglRenderer.setAttached(false);

        super.onDetachedFromWindow();
    }

    /**
     * Blocks until any pending frame is returned and all GL resources released (even if interruptions occur). In case of interruption during `release()`, the interruption flag is set. This function should be called before destroying the Activity and when the EGLContext is still valid. If you don't call this function, the GL resources may leak.
     */
    public void release() {
        synchronized (eglRenderer) {
            if (!isEglRendererInitialized) return;

            isEglRendererInitialized = false;
            setSurfaceTextureListener(null);
            eglRenderer.releaseEglSurface(() -> {
                //nothing to do
            });
            eglRenderer.release();

            mHandler = null;
        }
    }

    /**
     * Registers a callback that will be invoked when a new video frame is received.
     *
     * @param listener    The callback to be invoked. The callback will be invoked on the render thread.
     *                    It should be lightweight and must not call removeFrameListener.
     * @param scale       The scale of the Bitmap passed to the callback, or 0 if no Bitmap is
     *                    required.
     * @param drawerParam Custom drawer to use for this frame listener.
     */
    public void addFrameListener(
            SafeRenderFrameEglRenderer.FrameListener listener, float scale, RendererCommon.GlDrawer drawerParam) {
        eglRenderer.addFrameListener(listener, scale, drawerParam);
    }

    /**
     * Registers a callback that will be invoked when a new video frame is received. This version uses the drawer of the EglRenderer that passed an initialization.
     *
     * @param listener The callback to be invoked. The callback will be invoked on the render thread.
     *                 It should be lightweight and must not call removeFrameListener.
     * @param scale    The scale of the Bitmap passed to the callback, or 0 if no Bitmap is
     *                 required.
     */
    public void addFrameListener(SafeRenderFrameEglRenderer.FrameListener listener, float scale) {
        eglRenderer.addFrameListener(listener, scale);
    }

    /**
     * Unregisters a callback that will be invoked when a new video frame is received.
     *
     * @param listener The specific instance of the listener to remove.
     */
    public void removeFrameListener(SafeRenderFrameEglRenderer.FrameListener listener) {
        eglRenderer.removeFrameListener(listener);
    }

    /**
     * Enables a fixed size for the surface. It provides better performance but on some devices may cause bugs. By default this option turned off.
     */
    public void setEnableHardwareScaler(boolean enabled) {
        ThreadUtils.checkIsOnMainThread();
        enableFixedSize = enabled;
        updateSurfaceSize(false);
    }

    /**
     * Turns on and off the mirroring of video streams.
     */
    public void setMirror(final boolean mirror) {
        eglRenderer.setMirror(mirror);
    }

    /**
     * Checks the mirror state of the renderer.
     *
     * @return the indicator of the mirror activation.
     */
    public boolean isMirror() {
        return eglRenderer.isMirror();
    }

    /**
     * Checks for the first rendered frame. It is useful to prevent blink when a renderer renders a video, stops, disappears and starts a new one showing last previous frame(s).
     *
     * @return the first frame indicator.
     */
    public boolean isFirstFrameRendered() {
        return isFirstFrameRendered;
    }

    /**
     * Sets the video scaling in the allowed layout area.
     */
    public void setScalingType(RendererCommon.ScalingType scalingType) {
        ThreadUtils.checkIsOnMainThread();
        surfaceHeight = 0;
        surfaceWidth = 0;

        setScalingType = scalingType;
        setScalingType(scalingType, scalingType);
    }

    /**
     * Sets a scaling type in two orientations: match and mismatch.
     *
     * @param scalingTypeMatchOrientation    The requested ScalingType in match orientation
     * @param scalingTypeMismatchOrientation The requested ScalingType in mismatch orientation
     */
    public void setScalingType(RendererCommon.ScalingType scalingTypeMatchOrientation,
                               RendererCommon.ScalingType scalingTypeMismatchOrientation) {
        setNextDelayedPostTimeout(0);
        ThreadUtils.checkIsOnMainThread();
        videoLayoutMeasure.setScalingType(scalingTypeMatchOrientation, scalingTypeMismatchOrientation);

        eglRenderer.setScalingType(scalingTypeMatchOrientation);
        updateSurfaceSize();
    }

    /**
     * Informs about the current scaling type. This will be improved in the future to inform about the complex multiple match/mismatch use case.
     *
     * @return The ScalingType set. Default null.
     */
    @Nullable
    public RendererCommon.ScalingType getScalingType() {
        return setScalingType;
    }

    /**
     * Limits render frame rate.
     *
     * @param fps Limit render framerate to this value, or use Float.POSITIVE_INFINITY to disable fps
     *            reduction.
     */
    public void setFpsReduction(float fps) {
        synchronized (layoutLock) {
            isRenderingPaused = fps == 0f;
        }
        eglRenderer.setFpsReduction(fps);
    }

    /**
     * Disables the `FpsReduction`. It is useful to prevent environment testing or to achieve high quality.
     */
    public void disableFpsReduction() {
        synchronized (layoutLock) {
            isRenderingPaused = false;
        }
        eglRenderer.disableFpsReduction();
    }

    /**
     * Pauses the rendering process.
     */
    public void pauseVideo() {
        synchronized (layoutLock) {
            isRenderingPaused = true;
        }
        eglRenderer.pauseVideo();
    }

    @Override
    public void onFrame(VideoFrame frame) {
        try {
            updateFrameDimensionsAndReportEvents(frame);
            eglRenderer.onFrame(frame);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    // View layout interface.
    @Override
    protected void onMeasure(int widthSpec, int heightSpec) {
        ThreadUtils.checkIsOnMainThread();
        final Point size;
        synchronized (layoutLock) {
            size = videoLayoutMeasure.measure(widthSpec, heightSpec, rotatedFrameWidth, rotatedFrameHeight);
        }

        if (size.y > getScreenHeight()) size.y = getScreenHeight();
        if (size.x > getScreenWidth()) size.x = getScreenWidth();

        setMeasuredDimension(size.x, size.y);

        pendingLayout = false;
        posting = false;
    }

    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        ThreadUtils.checkIsOnMainThread();

        eglRenderer.setLayoutAspectRatio((right - left) / (float) (bottom - top));
        updateSurfaceSize(false);
    }

    private void updateSurfaceSize() {
        updateSurfaceSize(true);
    }

    public void forceRecalculateWidthHeight() {
        surfaceWidth = surfaceHeight = 0;
    }

    private void updateSurfaceSize(boolean sendLayout) {
        ThreadUtils.checkIsOnMainThread();
        synchronized (layoutLock) {

            if (enableFixedSize && rotatedFrameWidth != 0 && rotatedFrameHeight != 0 && getWidth() != 0
                    && getHeight() != 0) {
                final float layoutAspectRatio = getWidth() / (float) getHeight();
                final float frameAspectRatio = rotatedFrameWidth / (float) rotatedFrameHeight;
                final int drawnFrameWidth;
                final int drawnFrameHeight;
                if (frameAspectRatio > layoutAspectRatio) {
                    drawnFrameWidth = (int) (rotatedFrameHeight * layoutAspectRatio);
                    drawnFrameHeight = rotatedFrameHeight;
                } else {
                    drawnFrameWidth = rotatedFrameWidth;
                    drawnFrameHeight = (int) (rotatedFrameWidth / layoutAspectRatio);
                }
                // Aspect ratio of the drawn frame and the view is the same.
                final int width = Math.min(getWidth(), drawnFrameWidth);
                final int height = Math.min(getHeight(), drawnFrameHeight);
                if (width != surfaceWidth || height != surfaceHeight) {
                    surfaceWidth = width;
                    surfaceHeight = height;
                    if (sendLayout && surfaceWidth != 0 && surfaceHeight != 0)
                        requestLayoutIfNotPending();
                }
            } else {
                surfaceWidth = surfaceHeight = 0;
            }
        }

    }

    @Override
    public void onSurfaceTextureAvailable(SurfaceTexture surface, int width, int height) {
        ThreadUtils.checkIsOnMainThread();
        eglRenderer.createEglSurface(surface);
        surfaceWidth = surfaceHeight = 0;

        updateSurfaceSize();
    }

    @Override
    public void onSurfaceTextureSizeChanged(SurfaceTexture surface, int width, int height) {
        ThreadUtils.checkIsOnMainThread();
    }

    @Override
    public boolean onSurfaceTextureDestroyed(SurfaceTexture surface) {
        ThreadUtils.checkIsOnMainThread();
        final CountDownLatch completionLatch = new CountDownLatch(1);
        eglRenderer.releaseEglSurface(completionLatch::countDown);
        ThreadUtils.awaitUninterruptibly(completionLatch);

        return true;
    }

    @Override
    public void onSurfaceTextureUpdated(SurfaceTexture surface) {
        ThreadUtils.checkIsOnMainThread();
        surfaceWidth = surfaceHeight = 0;
        updateSurfaceSize(false);
    }

    private String getResourceName() {
        try {
            return getResources().getResourceEntryName(getId()) + ": ";
        } catch (Resources.NotFoundException e) {
            return "";
        }
    }

    /**
     * Posts a task to clear the `SurfaceView` to a transparent uniform color.
     */
    public void clearImage() {
        eglRenderer.clearImage();
    }

    private Runnable update = new Runnable() {
        @Override
        public void run() {
            updateSurfaceSize();
            requestLayoutIfNotPending();
        }
    };

    // Update frame dimensions and report any changes to |rendererEvents|.
    private void updateFrameDimensionsAndReportEvents(VideoFrame frame) {
        synchronized (layoutLock) {
            if (isRenderingPaused) {
                return;
            }
            if (!isFirstFrameRendered) {
                isFirstFrameRendered = true;
                logD("Reporting first rendered frame.");
                if (rendererEvents != null) {
                    rendererEvents.onFirstFrameRendered();
                }
            }
            if (rotatedFrameWidth != frame.getRotatedWidth()
                    || rotatedFrameHeight != frame.getRotatedHeight()
                    || frameRotation != frame.getRotation()) {
                logD("Reporting frame resolution changed to " + frame.getBuffer().getWidth() + "x"
                        + frame.getBuffer().getHeight() + " with rotation " + frame.getRotation());
                if (rendererEvents != null) {
                    rendererEvents.onFrameResolutionChanged(
                            frame.getBuffer().getWidth(), frame.getBuffer().getHeight(), frame.getRotation());
                }
                rotatedFrameWidth = frame.getRotatedWidth();
                rotatedFrameHeight = frame.getRotatedHeight();
                frameRotation = frame.getRotation();
                post(update);
            }
        }
    }

    private void logD(String string) {
        Logging.d(TAG, resourceName + string);
    }

    private boolean posting = false;
    private boolean lock = false;

    private final Runnable post = new Runnable() {
        @Override
        public void run() {
            lock = false;
            if (null != mHandler) mHandler.removeCallbacks(this);
            requestLayout();
        }
    };

    private void setNextDelayedPostTimeout(int delayedPostTimeout) {
        this.delayedPostTimeout = delayedPostTimeout;
    }

    private void requestLayoutIfNotPending() {

        if (posting || (null == mHandler)) return;
        if (lock) return;

        lock = true;
        posting = true;
        int delay = delayedPostTimeout;
        delayedPostTimeout = 1000;
        mHandler.postDelayed(post, delay);
    }

    public int getScreenWidth() {
        return size.x;
    }

    public int getScreenHeight() {
        return size.y;
    }

    /**
     * Forces a view to fire the first frame again. It is forced by the parent view.
     *
     * @param isFirstFrameRendered the new state to force inject.
     */
    public void setFirstFrameRendered(boolean isFirstFrameRendered) {
        this.isFirstFrameRendered = isFirstFrameRendered;
    }

    public void setDeltaX(float dx) {
        translateX = dx;
        applyRawTranslation();
    }

    public void setDeltaY(float dy) {
        translateY = dy;
        applyRawTranslation();
    }

    public float getTranslationY() {
        return translateY;
    }

    public float getTranslationX() {
        return translateX;
    }

    public float getRendererScaleX() {
        return rotatedFrameWidth * 1.f / getWidth();
    }

    public float getRendererScaleY() {
        return rotatedFrameHeight * 1.f / getHeight();
    }

    public float getRotatedFrameWidth() {
        return rotatedFrameWidth;
    }

    public float getRotatedFrameHeight() {
        return rotatedFrameHeight;
    }

    private void applyRawTranslation() {
        float scaleX = getRendererScaleX();
        float scaleY = getRendererScaleY();

        if (scaleY < scaleX) {
            //un/-stretch to fit height of the screen
            scaleX = scaleY;
            scaleY = 1;
        } else if (scaleY > scaleX) {
            //un/stretch to fit the width of the screen
            scaleX = 1;
            scaleY = scaleX;
        }

        float dx = scaleX * translateX;
        float dy = scaleY * translateY;

        eglRenderer.setTranslation(dx, dy);
    }

    @Override
    public void onRendererState(@NonNull RendererState currentState) {
        switch (currentState) {
            case FAULTY:
                if (null != eglRenderer) {

                }
        }
    }
}
