package com.voxeet.sdk.push.center.management;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.sdk.push.utils.Annotate;

/**
 * The VersionFilter model represents a blacklist system management represented by the [VersionFilter](/documentation/sdk/reference/android/models/versionfilter#versionfilter) constructor. It is based on:
 * - The [manufacturer's ID](/documentation/sdk/reference/android/models/versionfilter#manufacturer) that will be filtered. A null value indicates that all manufacturers are impacted.
 * - The [minimal Android API Version](/documentation/sdk/reference/android/models/versionfilter#minimumapiversion) on which the filter will be applied.
 * The filters assume that a blacklist starts with a specific Android version (breaking change) and will be applied to all next versions.
 * The default ones set a black screen for a full-screen notification on Android 10.
 * The [mustFilter](/documentation/sdk/reference/android/models/versionfilter#mustfilter) method informs if the filter is applied.
 */
@Annotate
public class VersionFilter {

    /**
     * Define the case where all manufacturer should be managed
     */
    public final static String ALL = null;

    /**
     * The manufacturer's identifier used for filtering by the `VersionFilter` instance.
     */
    @Nullable
    public String manufacturer;

    /**
     * Keeps the information about minimal Android API Version used for filtering by the `VersionFilter` instance.
     */
    public int minimumAPIVersion;

    /**
     * Constructs the given filter.
     *
     * @param manufacturer manufacturer or a null value
     * @param minimumAPIVersion minimal API version
     */
    public VersionFilter(@Nullable String manufacturer, int minimumAPIVersion) {
        this.manufacturer = manufacturer;
        this.minimumAPIVersion = minimumAPIVersion;
    }

    /**
     * A method called by the SDK to know if a filter is applied. Therefore, it blacklists the current mode to which it is registered to.
     *
     *  This method can be overridden to focus on checking lower versions or exact versions.
     *
     * @param manufacturer manufacturer of the currently used device
     * @param systemAPIVersion the version of the currently used operating system
     * @return the true value informs about blacklisted, an unmet value informs that a filter was not applied.
     */
    public boolean mustFilter(@NonNull String manufacturer, int systemAPIVersion) {
        if(minimumAPIVersion > systemAPIVersion) return false;
        if(null == this.manufacturer) return true;
        return manufacturer.equalsIgnoreCase(this.manufacturer);
    }
}
