package com.voxeet.sdk.push.center;

import android.content.Context;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.voxeet.sdk.push.center.invitation.IIncomingInvitationListener;
import com.voxeet.sdk.push.center.invitation.InvitationBundle;
import com.voxeet.sdk.push.center.management.Constants;
import com.voxeet.sdk.push.center.management.EnforcedNotificationMode;
import com.voxeet.sdk.push.center.management.NotificationMode;
import com.voxeet.sdk.push.center.management.PushConferenceDestroyed;
import com.voxeet.sdk.push.center.management.VersionFilter;
import com.voxeet.sdk.push.center.subscription.event.ConferenceEndedNotificationEvent;
import com.voxeet.sdk.push.center.subscription.event.InvitationReceivedNotificationEvent;
import com.voxeet.sdk.push.utils.Annotate;
import com.voxeet.sdk.utils.Opt;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

@Annotate
public class NotificationCenter {

    private static final String TAG = NotificationCenter.class.getSimpleName();
    public static NotificationCenter instance = new NotificationCenter();
    private Map<NotificationMode, Holder> invitationProviderForModes = new HashMap<>();


    @NonNull
    private EnforcedNotificationMode enforcedNotificationMode = EnforcedNotificationMode.FULLSCREEN_INCOMING_CALL;

    @Nullable
    private Context context;

    @NonNull
    private ReceiveSpecificEvents events = new ReceiveSpecificEvents();

    {
        invitationProviderForModes.put(NotificationMode.NONE, new Holder());
    }

    private NotificationCenter() {

    }

    public void init(@NonNull EventBus eventBus, @NonNull Context context) {
        this.context = context;

        if (!eventBus.isRegistered(eventBus)) eventBus.register(events);
    }

    /**
     * Changes the mode of notification management into the [enforcedNotificationMode](/documentation/sdk/reference/android/models/enforcednotificationmode).
     *
     * @param enforcedNotificationMode new mode
     * @return the current instance.
     */
    public NotificationCenter setEnforcedNotificationMode(@NonNull EnforcedNotificationMode enforcedNotificationMode) {
        this.enforcedNotificationMode = enforcedNotificationMode;
        return this;
    }

    /**
     * Registers the [IIncomingInvitationListener](/documentation/sdk/reference/android/models/iincominginvitationlistener) model dedicated to the [NotificationMode](/documentation/sdk/reference/android/models/notificationmode). The registered [IIncomingInvitationListener](/documentation/sdk/reference/android/models/iincominginvitationlistener) replaces the current model if the [NotificationMode](/documentation/sdk/reference/android/models/notificationmode) has already one attached listener.
     *
     * @param notificationMode   valid NotificationMode that will receive the provider
     * @param invitationProvider instance to register
     * @return the current instance.
     */
    public NotificationCenter register(@NonNull NotificationMode notificationMode, @Nullable IIncomingInvitationListener invitationProvider) {
        Holder holder = invitationProviderForModes.get(notificationMode);
        if (holder == null) {
            holder = new Holder();
            invitationProviderForModes.put(notificationMode, holder);
        }
        holder.invitationHolder = invitationProvider;
        return this;
    }

    /**
     * Registers the [VersionFilter](/documentation/sdk/reference/android/models/versionfilter) model for the specified [NotificationMode](/documentation/sdk/reference/android/models/notificationmode).
     *
     * @param notificationMode valid NotificationMode that will receive the possible filter
     * @param filter           filter to apply
     * @return the current instance.
     */
    public NotificationCenter register(@NonNull NotificationMode notificationMode, @NonNull VersionFilter filter) {
        Holder holder = invitationProviderForModes.get(notificationMode);
        if (holder == null) {
            holder = new Holder();
            invitationProviderForModes.put(notificationMode, holder);
        }

        if (!holder.filters.contains(filter)) holder.filters.add(filter);
        return this;
    }

    /**
     * Method called by the SDK when a push notification concerning an invitation is received by the WebSocket or a plugin.
     * <p>
     * It is possible to use this method when the SDK works on a system that is not Firebase compatible, or developers need to use a different library. If external push notifications are managed elsewhere, then developers are only responsible for transforming the Bundle or data to a valid [InvitationBundle](/documentation/sdk/reference/android/models/invitationbundle) model.
     *
     * @param context          valid context to use
     * @param invitationBundle InvitationBundle instance
     */
    public void onInvitationReceived(@NonNull Context context,
                                     @NonNull InvitationBundle invitationBundle) {
        onInvitationReceived(context, invitationBundle.asMap());
    }

    /**
     * Method called by the SDK when a push notification concerning a cancelled invitation is received by the WebSocket or a plugin.
     * <p>
     * It is possible to use this method when the SDK works on a system that is not Firebase compatible, or developers need to use a different library. If external push notifications are managed elsewhere, then developers are only responsible for transforming the Bundle or data to a valid [InvitationBundle](/documentation/sdk/reference/android/models/invitationbundle) model.
     *
     * @param context valid context to use
     * @return the corresponding conference id to be used on higher levels
     */
    @NonNull
    public String onInvitationCanceledReceived(@NonNull Context context,
                                               @NonNull String conferenceId) {
        IIncomingInvitationListener listener = getIncomingInvitation();
        if (null != listener) {
            listener.onInvitationCanceled(context, conferenceId);
        }

        return conferenceId;
    }

    /***********************************************************************************************
     *
     *
     * Private management
     *
     ***********************************************************************************************/

    private void onInvitationReceived(@NonNull Context context,
                                      @NonNull Map<String, String> remoteMessage) {
        IIncomingInvitationListener listener = getIncomingInvitation();
        if (null != listener) {
            InvitationBundle invitationBundle = new InvitationBundle(remoteMessage);
            listener.onInvitation(context, invitationBundle);
        }
    }

    boolean manageRemoteMessage(@NonNull Context context, @NonNull Map<String, String> remoteMessage) {
        if (remoteMessage.containsKey(Constants.NOTIF_TYPE)) {
            String notificationType = remoteMessage.get(Constants.NOTIF_TYPE);
            if (null == notificationType || TextUtils.isEmpty(notificationType))
                notificationType = "";

            switch (notificationType) {
                case Constants.NOTIF_TYPE_INVITATION_RECEIVED:
                    onInvitationReceived(context, remoteMessage);
                    break;
                case Constants.NOTIF_TYPE_LOCAL_CONFERENCE_DESTROYED:
                    String conferenceId = Opt.of(new InvitationBundle(remoteMessage)).then(c -> c.conferenceId).or("");

                    conferenceId = onInvitationCanceledReceived(context, conferenceId);
                    EventBus.getDefault().post(new PushConferenceDestroyed(conferenceId));
                    break;
                default:
            }

            return true;
        }
        return false;
    }

    @Nullable
    private IIncomingInvitationListener getIncomingInvitation() {
        for (NotificationMode notificationMode : enforcedNotificationMode.getModes()) {
            Holder holder = invitationProviderForModes.get(notificationMode);

            if (null != holder && !holder.mustFilter(Build.MANUFACTURER, Build.VERSION.SDK_INT)) {
                return holder.invitationHolder;
            }
        }

        return null;
    }

    private class Holder {
        @Nullable
        IIncomingInvitationListener invitationHolder;

        @NonNull
        public List<VersionFilter> filters = new CopyOnWriteArrayList<>();

        public boolean mustFilter(@NonNull String manufacturer, int systemAPIVersion) {
            for (VersionFilter filter : filters) {
                if (filter.mustFilter(manufacturer, systemAPIVersion)) return true;
            }
            return false;
        }
    }

    public class ReceiveSpecificEvents {

        @Subscribe(threadMode = ThreadMode.MAIN)
        public void onEvent(@NonNull ConferenceEndedNotificationEvent event) {
            onInvitationCanceledReceived(context, event.conferenceId);
        }

        @Subscribe(threadMode = ThreadMode.MAIN)
        public void onEvent(@NonNull InvitationReceivedNotificationEvent event) {
            onInvitationReceived(context, event.invitation);
        }
    }
}
