package com.voxeet.sdk.push.center.invitation;

import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.sdk.json.ParticipantInfo;
import com.voxeet.sdk.models.ParticipantNotification;
import com.voxeet.sdk.models.v1.ConferenceParticipantStatus;
import com.voxeet.sdk.push.center.management.Constants;
import com.voxeet.sdk.push.utils.Annotate;
import com.voxeet.sdk.utils.Opt;

import java.util.HashMap;
import java.util.Map;

/**
 * The InvitationBundle model represents a typical push notification invitation for a conference. The [InvitationBundle](/documentation/sdk/reference/android/models/invitationbundle#invitationbundle) constructs an invitation that can be transformed [as a map](/documentation/sdk/reference/android/models/invitationbundle#asmap) or [as a bundle](/documentation/sdk/reference/android/models/invitationbundle#asbundle). There are two ways of creating invitations: using the [name](/documentation/sdk/reference/android/models/invitationbundle#invitername), [ID](/documentation/sdk/reference/android/models/invitationbundle#inviterid), [external ID](/documentation/sdk/reference/android/models/invitationbundle#inviterexternalid), [avatar URL](/documentation/sdk/reference/android/models/invitationbundle#inviteravatarurl) of the invited participant, and the [conference ID](/documentation/sdk/reference/android/models/invitationbundle#conferenceid). The other way uses a native bundle.
 */
@Annotate
public class InvitationBundle {

    /**
     * The inviter. A valid invitation must contain non null values.
     */
    @NonNull
    public ParticipantNotification inviter;

    /**
     * The conference ID. A valid invitation must contain a non-null value.
     */
    @Nullable
    public String conferenceId;

    /**
     * Constructs an invitation using raw values.
     *
     * @param inviterId         inviter's ID, it has to contain a non-null value to be valid
     * @param inviterName       inviter's name, it can contain a null value
     * @param inviterExternalId inviter's external ID, it can contain a null value
     * @param conferenceId      ID of the conference, it has to contain a non-null value to be valid
     * @param inviterAvatarUrl  URL of the inviter's avatar, it can contain a null value
     */
    public InvitationBundle(@Nullable String inviterId, @Nullable String inviterName, @Nullable String inviterExternalId, @Nullable String conferenceId, @Nullable String inviterAvatarUrl) {
        this.conferenceId = conferenceId;
        inviter = new ParticipantNotification(
                Opt.of(inviterId).or(""),
                new ParticipantInfo(
                        Opt.of(inviterName).orNull(),
                        Opt.of(inviterExternalId).orNull(),
                        Opt.of(inviterAvatarUrl).orNull()
                ),
                ConferenceParticipantStatus.UNKNWON);
    }

    /**
     * Constructs an invitation using a native Bundle. Refer to the Constants for the available keys.
     *
     * @param bundle Bundle, any null Bundle will trigger an exception
     */
    public InvitationBundle(@NonNull Bundle bundle) {
        this(
                bundle.getString(Constants.INVITER_ID, null),
                bundle.getString(Constants.INVITER_NAME, null),
                bundle.getString(Constants.INVITER_EXTERNAL_ID, null),
                bundle.getString(Constants.CONF_ID, null),
                bundle.getString(Constants.INVITER_URL, null)
        );
    }

    /**
     * Construct an invitation from a Map. Refer to the Constants for the available keys.
     *
     * @param remoteMessage The remote message. must be non-null, Any null Map will trigger an exception
     */
    public InvitationBundle(@NonNull Map<String, String> remoteMessage) {
        this(
                get(remoteMessage, Constants.INVITER_ID),
                get(remoteMessage, Constants.INVITER_NAME),
                get(remoteMessage, Constants.INVITER_EXTERNAL_ID),
                get(remoteMessage, Constants.CONF_ID),
                get(remoteMessage, Constants.INVITER_URL)
        );
    }

    @Nullable
    private static String get(@NonNull Map<String, String> map, @NonNull String key) {
        if (map.containsKey(key)) return map.get(key);
        return null;
    }

    /**
     * Transforms the current invitation into a Map. If the invitation is valid, the Map is considered as valid.
     *
     * @return The Map. Valid if the invitation is valid.
     */
    @NonNull
    public Map<String, String> asMap() {
        String inviterId = inviter.getId();
        String inviterExternalId = Opt.of(inviter.getInfo()).then(ParticipantInfo::getExternalId).orNull();
        String inviterName = Opt.of(inviter.getInfo()).then(ParticipantInfo::getName).orNull();
        String inviterAvatarUrl = Opt.of(inviter.getInfo()).then(ParticipantInfo::getAvatarUrl).orNull();

        Map<String, String> map = new HashMap<>();
        if (null != conferenceId) map.put(Constants.CONF_ID, conferenceId);
        if (null != inviterExternalId) map.put(Constants.INVITER_EXTERNAL_ID, inviterExternalId);
        if (null != inviterId) map.put(Constants.INVITER_ID, inviterId);
        if (null != inviterName) map.put(Constants.INVITER_NAME, inviterName);
        if (null != inviterAvatarUrl) map.put(Constants.INVITER_URL, inviterAvatarUrl);
        return map;
    }

    /**
     * Transforms this current invitation into a native Bundle. If the invitation is valid, the Bundle is considered as valid.
     *
     * @return The Bundle. Valid if the invitation is valid.
     */
    @NonNull
    public Bundle asBundle() {
        Bundle bundle = new Bundle();
        bundle.putString(Constants.CONF_ID, conferenceId);
        bundle.putString(Constants.INVITER_ID, inviter.getId());
        bundle.putString(Constants.INVITER_EXTERNAL_ID, Opt.of(inviter.getInfo()).then(ParticipantInfo::getExternalId).orNull());
        bundle.putString(Constants.INVITER_NAME, Opt.of(inviter.getInfo()).then(ParticipantInfo::getName).orNull());
        bundle.putString(Constants.INVITER_URL, Opt.of(inviter.getInfo()).then(ParticipantInfo::getAvatarUrl).orNull());
        return bundle;
    }
}
