package com.voxeet.sdk.push.center;

import android.content.Context;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.voxeet.sdk.push.center.invitation.IIncomingInvitationListener;
import com.voxeet.sdk.push.center.invitation.InvitationBundle;
import com.voxeet.sdk.push.center.management.Constants;
import com.voxeet.sdk.push.center.management.EnforcedNotificationMode;
import com.voxeet.sdk.push.center.management.NotificationMode;
import com.voxeet.sdk.push.center.management.PushConferenceDestroyed;
import com.voxeet.sdk.push.center.management.VersionFilter;

import org.greenrobot.eventbus.EventBus;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

public class NotificationCenterFactory {

    private static final String TAG = NotificationCenterFactory.class.getSimpleName();
    public static NotificationCenterFactory instance = new NotificationCenterFactory();
    private Map<NotificationMode, Holder> invitationProviderForModes = new HashMap<>();

    @NonNull
    private EnforcedNotificationMode enforcedNotificationMode = EnforcedNotificationMode.FULLSCREEN_INCOMING_CALL;

    {
        invitationProviderForModes.put(NotificationMode.NONE, new Holder());
    }

    private NotificationCenterFactory() {

    }

    public boolean manageRemoteMessage(@NonNull Context context, @NonNull Map<String, String> remoteMessage) {
        if (remoteMessage.containsKey(Constants.NOTIF_TYPE)) {
            String notificationType = remoteMessage.get(Constants.NOTIF_TYPE);
            if (null == notificationType || TextUtils.isEmpty(notificationType))
                notificationType = "";

            switch (notificationType) {
                case Constants.NOTIF_TYPE_INVITATION_RECEIVED:
                    onInvitationReceived(context, remoteMessage,
                            Build.MANUFACTURER,
                            Build.VERSION.SDK_INT);
                    break;
                case Constants.NOTIF_TYPE_LOCAL_CONFERENCE_DESTROYED:
                    String conferenceId = onInvitationCanceledReceived(context, remoteMessage,
                            Build.MANUFACTURER,
                            Build.VERSION.SDK_INT);
                    EventBus.getDefault().post(new PushConferenceDestroyed(conferenceId));
                    break;
                default:
            }

            return true;
        }
        return false;
    }

    public NotificationCenterFactory setEnforcedNotificationMode(@NonNull EnforcedNotificationMode enforcedNotificationMode) {
        this.enforcedNotificationMode = enforcedNotificationMode;
        return this;
    }

    public NotificationCenterFactory register(@NonNull NotificationMode notificationMode, @Nullable IIncomingInvitationListener invitationProvider) {
        Holder holder = invitationProviderForModes.get(notificationMode);
        if (holder == null) {
            holder = new Holder();
            invitationProviderForModes.put(notificationMode, holder);
        }
        holder.invitationHolder = invitationProvider;
        return this;
    }

    public NotificationCenterFactory register(@NonNull NotificationMode notificationMode, @NonNull VersionFilter filter) {
        Holder holder = invitationProviderForModes.get(notificationMode);
        if (holder == null) {
            holder = new Holder();
            invitationProviderForModes.put(notificationMode, holder);
        }

        if (!holder.filters.contains(filter)) holder.filters.add(filter);
        return this;
    }

    @Nullable
    private IIncomingInvitationListener getIncomingInvitation(@NonNull String manufacturer, int systemAPIVersion) {
        for (NotificationMode notificationMode : enforcedNotificationMode.getModes()) {
            Holder holder = invitationProviderForModes.get(notificationMode);

            if (null != holder && !holder.mustFilter(manufacturer, systemAPIVersion)) {
                return holder.invitationHolder;
            }
        }

        return null;
    }

    public void onInvitationReceived(@NonNull Context context,
                                     @NonNull Map<String, String> remoteMessage,
                                     @NonNull String manufacturer,
                                     int systemAPIVersion) {
        IIncomingInvitationListener listener = getIncomingInvitation(manufacturer, systemAPIVersion);
        if (null != listener) {
            InvitationBundle invitationBundle = new InvitationBundle(remoteMessage);
            listener.onInvitation(context, invitationBundle);
        }
    }

    @NonNull
    public String onInvitationCanceledReceived(@NonNull Context context,
                                               @NonNull Map<String, String> remoteMessage,
                                               @NonNull String manufacturer,
                                               int systemAPIVersion) {
        String conferenceId = "";
        if (remoteMessage.containsKey(Constants.CONF_ID)) {
            conferenceId = remoteMessage.get(Constants.CONF_ID);
        }
        if (null == conferenceId || TextUtils.isEmpty(conferenceId)) {
            conferenceId = "";
        }

        IIncomingInvitationListener listener = getIncomingInvitation(manufacturer, systemAPIVersion);
        if (null != listener) {
            InvitationBundle invitationBundle = new InvitationBundle(remoteMessage);
            listener.onInvitationCanceled(context, conferenceId);
        }

        return conferenceId;
    }

    private class Holder {
        @Nullable
        IIncomingInvitationListener invitationHolder;

        @NonNull
        public List<VersionFilter> filters = new CopyOnWriteArrayList<>();

        public boolean mustFilter(@NonNull String manufacturer, int systemAPIVersion) {
            for (VersionFilter filter : filters) {
                if (filter.mustFilter(manufacturer, systemAPIVersion)) return true;
            }
            return false;
        }

    }
}
