package com.voxeet.push.center.invitation;

import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.push.center.management.Constants;
import com.voxeet.push.utils.Annotate;

import java.util.HashMap;
import java.util.Map;

/**
 * The InvitationBundle model represents a typical push notification invitation for a conference. The [InvitationBundle](/documentation/sdk/reference/android/models/invitationbundle#invitationbundle) constructs an invitation that can be transformed [as a map](/documentation/sdk/reference/android/models/invitationbundle#asmap) or [as a bundle](/documentation/sdk/reference/android/models/invitationbundle#asbundle). There are two ways of creating invitations: using the [name](/documentation/sdk/reference/android/models/invitationbundle#invitername), [ID](/documentation/sdk/reference/android/models/invitationbundle#inviterid), [external ID](/documentation/sdk/reference/android/models/invitationbundle#inviterexternalid), [avatar URL](/documentation/sdk/reference/android/models/invitationbundle#inviteravatarurl) of the invited participant, and the [conference ID](/documentation/sdk/reference/android/models/invitationbundle#conferenceid). The other way uses a native bundle.
 */
@Annotate
public class InvitationBundle {

    /**
     * Displays an inviter name or a null value if no one was provided.
     */
    @Nullable
    public String inviterName;

    /**
     * The participant ID. A valid invitation must contain a non-null value.
     */
    @Nullable
    public String inviterId;

    /**
     * The inviter's external ID (if it was provided).
     */
    @Nullable
    public String inviterExternalId;

    /**
     * The conference ID. A valid invitation must contain a non-null value.
     */
    @Nullable
    public String conferenceId;

    /**
     * The participant's avatar URL (if one was provided).
     */
    @Nullable
    public String inviterAvatarUrl;

    /**
     * Constructs an invitation using raw values.
     *
     * @param inviterId The ID of the inviter. It has to contain the non-null value to be valid.
     * @param inviterName The name of the inviter. It can contain the null value.
     * @param inviterExternalId The external ID of the inviter. It can contain the null value.
     * @param conferenceId The ID of the conference. It has to contain the non-null value to be valid.
     * @param inviterAvatarUrl The URL of the inviter's avatar. It can contain the null value.
     */
    public InvitationBundle(@Nullable String inviterId, @Nullable String inviterName, @Nullable String inviterExternalId, @Nullable String conferenceId, @Nullable String inviterAvatarUrl) {
        this.inviterId = inviterId;
        this.inviterExternalId = inviterExternalId;
        this.conferenceId = conferenceId;
        this.inviterAvatarUrl = inviterAvatarUrl;
        this.inviterName = inviterName;
    }

    /**
     * Constructs an invitation using a native Bundle. Refer to the Constants for the available keys.
     * @param bundle The bundle. It has to be non null, any null Bundle will trigger an exception.
     */
    public InvitationBundle(@NonNull Bundle bundle) {
        this.inviterId = bundle.getString(Constants.INVITER_ID, null);
        this.inviterExternalId = bundle.getString(Constants.INVITER_EXTERNAL_ID, null);
        this.conferenceId = bundle.getString(Constants.CONF_ID, null);
        this.inviterAvatarUrl = bundle.getString(Constants.INVITER_URL, null);
        this.inviterName = bundle.getString(Constants.INVITER_NAME, null);
    }

    /**
     * Construct an invitation from a Map. Refer to the Constants for the available keys.
     * @param remoteMessage The remote message. must be non-null, Any null Map will trigger an exception
     */
    public InvitationBundle(@NonNull Map<String, String> remoteMessage) {
        this.inviterId = get(remoteMessage, Constants.INVITER_ID);
        this.inviterExternalId = get(remoteMessage, Constants.INVITER_EXTERNAL_ID);
        this.conferenceId = get(remoteMessage, Constants.CONF_ID);
        this.inviterAvatarUrl = get(remoteMessage, Constants.INVITER_URL);
        this.inviterName = get(remoteMessage, Constants.INVITER_NAME);
    }

    @Nullable
    private String get(@NonNull Map<String, String> map, @NonNull String key) {
        if (map.containsKey(key)) return map.get(key);
        return null;
    }

    /**
     * Transforms the current invitation into a Map. If the invitation is valid, the Map is considered as valid.
     * @return The Map. Valid if the invitation is valid.
     */
    @NonNull
    public Map<String, String> asMap() {
        Map<String, String> map = new HashMap<>();
        if (null != conferenceId) map.put(Constants.CONF_ID, conferenceId);
        if (null != inviterExternalId) map.put(Constants.INVITER_EXTERNAL_ID, inviterExternalId);
        if (null != inviterId) map.put(Constants.INVITER_ID, inviterId);
        if (null != inviterName) map.put(Constants.INVITER_NAME, inviterName);
        if (null != inviterAvatarUrl) map.put(Constants.INVITER_URL, inviterAvatarUrl);
        return map;
    }

    /**
     * Transforms this current invitation into a native Bundle. If the invitation is valid, the Bundle is considered as valid.
     * @return The Bundle. Valid if the invitation is valid.
     */
    @NonNull
    public Bundle asBundle() {
        Bundle bundle = new Bundle();
        bundle.putString(Constants.CONF_ID, conferenceId);
        bundle.putString(Constants.INVITER_EXTERNAL_ID, inviterExternalId);
        bundle.putString(Constants.INVITER_ID, inviterId);
        bundle.putString(Constants.INVITER_NAME, inviterName);
        bundle.putString(Constants.INVITER_URL, inviterAvatarUrl);
        return bundle;
    }
}
