package com.voxeet.sdk.services.conference.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.promise.Promise;
import com.voxeet.sdk.events.restapi.ReplayConferenceResult;
import com.voxeet.sdk.events.sdk.ConferenceStatusUpdatedEvent;
import com.voxeet.sdk.json.SdkConferenceReplayBody;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.v2.ServerErrorOrigin;
import com.voxeet.sdk.network.endpoints.IRestApiConferenceAccess;
import com.voxeet.sdk.services.AbstractPromiseable;
import com.voxeet.sdk.services.ConferenceService;
import com.voxeet.sdk.services.MediaDeviceService;
import com.voxeet.sdk.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.services.conference.information.ConferenceParticipantType;
import com.voxeet.sdk.services.conference.information.ConferenceStatus;
import com.voxeet.sdk.services.conference.information.LocalConferenceType;
import com.voxeet.sdk.utils.HttpHelper;

import org.greenrobot.eventbus.EventBus;

import retrofit2.Call;

public class ReplayPromise extends AbstractPromiseable<Conference, IRestApiConferenceAccess> {
    private final static String TAG = ReplayPromise.class.getSimpleName();
    private final String conferenceId;
    private final long offset;

    public ReplayPromise(@NonNull ConferenceService parent,
                         @NonNull MediaDeviceService mediaDeviceService,
                         @NonNull IRestApiConferenceAccess provider,
                         @Nullable ConferenceInformation information,
                         @NonNull EventBus eventBus,
                         @NonNull String conferenceId,
                         long offset) {
        super(parent, mediaDeviceService, provider, information, eventBus);

        this.conferenceId = conferenceId;
        this.offset = offset;
    }

    @NonNull
    @Override
    public Promise<Conference> createPromise() {
        return new Promise<>(solver -> {
            final ConferenceInformation information = getConferenceInformation(conferenceId);

            information.setConferenceParticipantType(ConferenceParticipantType.LISTENER);

            final Call<ReplayConferenceResult> user = getApiRestCall().replay(conferenceId, new SdkConferenceReplayBody(offset));
            HttpHelper.enqueue(user, (response, object, exception) -> {
                if (null != exception || null == object) {
                    if (null == exception) {
                        exception = new IllegalStateException("invalid object received in replay promise");
                    }
                    ConferenceStatusUpdatedEvent state = new ConferenceStatusUpdatedEvent(conferenceId, ConferenceStatus.ERROR);
                    getEventBus().post(state);

                    log("Error while replay conference : creation process part");
                    solver.reject(HttpHelper.manageThrowableOrServerError(exception, ServerErrorOrigin.CREATE));
                } else {
                    Log.d(TAG, "Success server answer for replaying mConference with id: " +
                            conferenceId + "at offset: " + offset + " given := " + object.conferenceId + " " + object.conferenceAlias
                            + " " + information);

                    moveConference(object.conferenceId, information);

                    information.getConference().setConferenceId(object.conferenceId).setConferenceAlias(object.conferenceAlias);
                    information.setConferenceType(LocalConferenceType.REPLAY);

                    solver.resolve(joinConference(information));
                }
            });

        });
    }
}
