package com.voxeet.sdk.services.conference.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.promise.Promise;
import com.voxeet.sdk.events.promises.NotInConferenceException;
import com.voxeet.sdk.events.sdk.ConferenceStatusUpdatedEvent;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.v2.ServerErrorOrigin;
import com.voxeet.sdk.network.endpoints.IRestApiConferenceAccess;
import com.voxeet.sdk.services.AbstractPromiseable;
import com.voxeet.sdk.services.ConferenceService;
import com.voxeet.sdk.services.MediaDeviceService;
import com.voxeet.sdk.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.services.conference.information.ConferenceStatus;
import com.voxeet.sdk.utils.HttpHelper;
import com.voxeet.sdk.utils.Opt;

import org.greenrobot.eventbus.EventBus;

import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Response;

public class LeavePromise extends AbstractPromiseable<Boolean, IRestApiConferenceAccess> {
    private final static String TAG = LeavePromise.class.getSimpleName();

    public LeavePromise(@NonNull ConferenceService parent,
                        @NonNull MediaDeviceService mediaDeviceService,
                        @NonNull IRestApiConferenceAccess provider,
                        @Nullable ConferenceInformation information,
                        @NonNull EventBus eventBus) {
        super(parent, mediaDeviceService, provider, information, eventBus);
    }

    @NonNull
    @Override
    public Promise<Boolean> createPromise() {
        return new Promise<>(solver -> {
            final ConferenceInformation infos = getInformation();
            if (null == infos) {
                Promise.reject(solver, new NotInConferenceException());
                return;
            }

            final Conference conference = infos.getConference();

            final String conferenceId = infos.getConference().getId();
            infos.setConferenceState(ConferenceStatus.LEAVING);

            log("Attempting to leave mConference with mConference id " + conferenceId);

            closeMedia();

            final Call<ResponseBody> user = getApiRestCall().leave(conferenceId);
            HttpHelper.promise(user, ServerErrorOrigin.LEAVE)
                    .then(answer -> {
                        Log.d("ConferenceService", "Conference left successfully " + Opt.of(answer).then(a -> a.response).then(Response::code).or(0));
                        removeTimeoutCallbacks();
                        infos.setConferenceState(ConferenceStatus.LEFT);
                        getEventBus().post(new ConferenceStatusUpdatedEvent(conference, conference.getState()));

                        solver.resolve(true);
                    })
                    .error(error -> {
                        Log.d(TAG, "Something went wrong while leaving the mConference");
                        removeTimeoutCallbacks();

                        infos.setConferenceState(ConferenceStatus.ERROR);
                        getEventBus().post(new ConferenceStatusUpdatedEvent(conference, conference.getState()));
                        solver.reject(error);
                    });
        });
    }
}
