package com.voxeet.sdk.services.conference.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.VoxeetSDK;
import com.voxeet.promise.Promise;
import com.voxeet.promise.solve.PromiseSolver;
import com.voxeet.promise.solve.Solver;
import com.voxeet.promise.solve.ThenPromise;
import com.voxeet.promise.solve.ThenVoid;
import com.voxeet.sdk.events.sdk.ConferenceStatusUpdatedEvent;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.network.endpoints.IRestApiConferenceAccess;
import com.voxeet.sdk.services.AbstractPromiseable;
import com.voxeet.sdk.services.AudioService;
import com.voxeet.sdk.services.ConferenceService;
import com.voxeet.sdk.services.MediaDeviceService;
import com.voxeet.sdk.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.services.conference.information.ConferenceParticipantType;
import com.voxeet.sdk.services.conference.information.ConferenceStatus;
import com.voxeet.sdk.services.conference.information.LocalConferenceType;

import org.greenrobot.eventbus.EventBus;

public class JoinPromise extends AbstractPromiseable<Conference, IRestApiConferenceAccess> {
    private final static String TAG = JoinPromise.class.getSimpleName();
    private final String conferenceId;
    private final ConferenceParticipantType userType;

    private AudioService audioService;

    public JoinPromise(@NonNull ConferenceService parent,
                       @NonNull AudioService audioService,
                       @NonNull MediaDeviceService mediaDeviceService,
                       @NonNull IRestApiConferenceAccess provider,
                       @Nullable ConferenceInformation information,
                       @NonNull EventBus eventBus,
                       @NonNull VoxeetSDK instance,
                       @NonNull String conferenceId,
                       @NonNull ConferenceParticipantType userType) {
        super(parent, mediaDeviceService, provider, information, eventBus);

        this.audioService = audioService;
        this.conferenceId = conferenceId;
        this.userType = userType;
    }

    @NonNull
    @Override
    public Promise<Conference> createPromise() {
        final Promise<Conference> promise = new Promise<>(new PromiseSolver<Conference>() {
            @Override
            public void onCall(@NonNull final Solver<Conference> solver) {
                //locking
                joinLock();

                //init the conference with the proper conferenceId and alias to null
                ConferenceInformation information = createOrSetConferenceWithParams(conferenceId, null);
                information.setConferenceParticipantType(userType);

                String currentConferenceId = getParent().getConferenceId();
                ConferenceInformation holder = getConferenceInformation(conferenceId);

                boolean init_done = ConferenceStatus.JOINING.equals(holder.getConferenceState()) || ConferenceStatus.JOINED.equals(holder.getConferenceState());

                if (null != currentConferenceId && currentConferenceId.equals(conferenceId) && init_done) {
                    Log.d(TAG, "onCall: join the conference " + conferenceId + " // invalid call ! already calling it");
                    Log.d(TAG, "onCall: btw, the current " + conferenceId + " state is := " + holder.getConferenceState());

                    solver.resolve(holder.getConference());
                    joinUnlock();
                    return;
                }

                //prevent issue with the join
                setIsInConference(true);

                Conference conference = holder.getConference();

                Log.d(TAG, "Attempting to join conference alias:=" + conference.getAlias() + " conferenceId:=" + conferenceId);

                Log.d(TAG, "onNext: having conference");

                audioService.enable();

                //TODO wassent here but also after
                //getEventBus().post(new ConferencePreJoinedEvent(conferenceId, conference.getAlias()));

                information.setConferenceState(ConferenceStatus.JOINING);
                information.setConferenceType(LocalConferenceType.CONFERENCE);
                information.setTelecomMode(getParent().ConferenceConfigurations.telecomMode);

                Log.d(TAG, "onNext: join with := " + conference.getId() + " " + conference.getAlias());

                joinUnlock();

                //set the current conference if none is set in the SDK
                setCurrentConferenceIfNotInPreviousConference(information);

                //try to join the conference
                joinConference(information).then((result, internal_solver) -> {
                    Log.d(TAG, "onCall: joinConference done " + result);
                    solver.resolve(result);
                }).error(error -> {
                    //send error -> left
                    information.setConferenceState(ConferenceStatus.ERROR);
                    ConferenceStatusUpdatedEvent event = new ConferenceStatusUpdatedEvent(conference, conference.getState());
                    getEventBus().post(event);
                    error.printStackTrace();
                    solver.reject(error);
                });
            }
        });

        if (VoxeetSDK.session().isSocketOpen()) {
            return promise;
        } else {
            return new Promise<>(solver -> {
                Log.d(TAG, "onCall: joining but the socket is disconnected");
                VoxeetSDK.session().open()
                        .then((ThenPromise<Boolean, Conference>) aBoolean -> promise)
                        .then((ThenVoid<Conference>) solver::resolve)
                        .error(solver::reject);
            });
        }
    }
}
