package com.voxeet.sdk.services;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.graphics.Point;
import android.media.projection.MediaProjectionManager;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;
import android.view.WindowManager;

import com.voxeet.VoxeetSDK;
import com.voxeet.promise.Promise;
import com.voxeet.sdk.network.endpoints.IRestApiScreenShare;
import com.voxeet.sdk.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.services.conference.promises.StartScreensharePromise;
import com.voxeet.sdk.services.conference.promises.StopScreenSharePromise;
import com.voxeet.sdk.services.screenshare.RequestScreenSharePermissionEvent;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.ScreenHelper;

/**
 * The ScreenShareService allows an application to share a screen during the conference.
 * <p>
 * **Typical application workflow:**
 * <p>
 * **1.** The application calls the [sendRequestStartScreenShare](/documentation/sdk/reference/android/screenshare#sendrequeststartscreenshare) method that asks for permission to share the screen.
 * <p>
 * **2.** The application receives the [RequestScreenSharePermissionEvent](/documentation/sdk/reference/android/screenshare#requestscreensharepermissionevent) asking the participant about permission to share the screen.
 * <p>
 * **3.** The application calls the [sendUserPermissionRequest](/documentation/sdk/reference/android/screenshare#senduserpermissionrequest) method allowing to share the screen.
 * <p>
 * **4.** The application then calls the [consumeRightsToScreenShare](/documentation/sdk/reference/android/screenshare#consumerightstoscreenshare) method which manages previously received information and grants the right to share the screen.
 * <p>
 * **5.** The application starts sharing the screen through the [startScreenShare](/documentation/sdk/reference/android/screenshare#startscreenshare) method.
 * <p>
 * **6.** Then, the application can get information about the screen size ([getScreenSize](/documentation/sdk/reference/android/screenshare#getscreensize)), change it ([setScreenSizeInformation](/documentation/sdk/reference/android/screenshare#setscreensizeinformation)), or scale the screen ([getScreenSizeScaled](/documentation/sdk/reference/android/screenshare#getscreensizescaled)).
 * <p>
 * **7.** To stop sharing the screen, the application calls the [stopScreenShare](/documentation/sdk/reference/android/screenshare#stopscreenshare) method.
 */
@Annotate
public class ScreenShareService extends AbstractVoxeetService {
    private static final int REQUEST_SCREEN_CAPTURE = 672;
    private static final String TAG = ScreenShareService.class.getSimpleName();
    private static final int DEFAULT_WIDTH = 640;
    private static final int DEFAULT_HEIGHT = 480;

    private int mLastResultCode;

    @Nullable
    private Point mLastPoint;

    @Nullable
    private Intent mLastData;

    @NoDocumentation
    public ScreenShareService(@NonNull SdkEnvironmentHolder instance) {
        super(instance);

        //Set the default screenshare information
        try {
            //setScreenSizeInformation(ScreenHelper.getScreenSize(instance.voxeetSdk.getApplicationContext()));
            Point size = ScreenHelper.getScreenSize(instance.voxeetSdk.getApplicationContext());
            size.y = size.x = Math.max(size.x, size.y);

            setScreenSizeInformation(size);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Start a screenshare if currently not started or stop it.
     * <p>
     * Warning : this method is only making call internally, no information is sent directly after this call.
     * <p>
     * Developers should use the startScreenshare or stopScreenshare only
     */
    @NoDocumentation
    @Deprecated
    public void toggleScreenShare() {
        if (isScreenShareOn()) {
            stopScreenShare().then((result, solver) -> Log.d(TAG, "onSuccess: toggleScreenShare " + result))
                    .error(manageError());
        } else {
            VoxeetSDK.screenShare().sendRequestStartScreenShare();
        }
    }

    /**
     * Starts sharing the screen.
     *
     * @param intent intent information obtained when the participant receives a permission to share a screen
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Boolean> startScreenShare(@NonNull Intent intent) {
        int width = DEFAULT_WIDTH;
        int height = DEFAULT_HEIGHT;

        if (null != mLastPoint) {
            width = mLastPoint.x;
            height = mLastPoint.y;
        }


        return new StartScreensharePromise(VoxeetSDK.conference(),
                VoxeetSDK.mediaDevice(),
                getService(IRestApiScreenShare.class),
                VoxeetSDK.conference().getCurrentConference(),
                getEventBus(),
                intent,
                width,
                height).createPromise();
    }

    /**
     * Stops sharing the screen.
     *
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Boolean> stopScreenShare() {
        return new StopScreenSharePromise(VoxeetSDK.conference(),
                VoxeetSDK.mediaDevice(),
                getService(IRestApiScreenShare.class),
                VoxeetSDK.conference().getCurrentConference(),
                getEventBus()).createPromise();
    }

    /**
     * Changes future ScreenShare propagation information.
     *
     * @param point point representing the width and height of the shared screen
     * @return the current service instance.
     */
    public ScreenShareService setScreenSizeInformation(@NonNull Point point) {
        Log.d(TAG, "sendUserPermissionRequest: set the screenshare dimension to " + point.x + "x" + point.y);
        mLastPoint = point;

        return this;
    }

    /**
     * Sends a request to share the screen to the VoxeetAppCompatActivity instance which presents the permission request modal to the participant. After granting the permission, the VoxeetAppCompatActivity requests the start call.
     */
    public void sendRequestStartScreenShare() {
        getEventBus().post(new RequestScreenSharePermissionEvent());
    }

    /**
     * Enables the proper permission dialog to the given Activity instance.
     *
     * @param activity valid activity used to start the system and get the callback result
     * @return the compatibility result.
     */
    public boolean sendUserPermissionRequest(@NonNull Activity activity) {
        return sendUserPermissionRequest(activity, false);
    }

    /**
     * Enables the proper permission dialog to the given Activity instance.
     *
     * @param activity     valid activity used to start the system and get the callback result
     * @param full_quality flag indicating if the screenshare will in the same time force the dimension sent to be equals to the maximum pixels dimension natively displayed to the user.
     * @return the compatibility result.
     */
    public boolean sendUserPermissionRequest(@NonNull Activity activity, boolean full_quality) {

        if (full_quality) {
            Point size = ScreenHelper.getScreenSize(activity);
            size.y = size.x = Math.max(size.x, size.y);

            setScreenSizeInformation(size);
        }

        if (Build.VERSION_CODES.LOLLIPOP <= Build.VERSION.SDK_INT) {
            MediaProjectionManager manager = (MediaProjectionManager)
                    activity.getSystemService(Context.MEDIA_PROJECTION_SERVICE);
            Intent request = manager.createScreenCaptureIntent();
            activity.startActivityForResult(request, ScreenShareService.REQUEST_SCREEN_CAPTURE);
            return true;
        }
        return false;
    }

    /**
     * Automatically manages information obtained from the Activity lifecycle and calls the [startScreenshare](/documentation/sdk/reference/android/screenshare#startscreenshare) method.
     */
    public void consumeRightsToScreenShare() {
        if (mLastResultCode == Activity.RESULT_OK && null != mLastData) {
            startScreenShare(mLastData)
                    .then((result, solver) -> Log.d(TAG, "onCall: starting screen share"))
                    .error(Throwable::printStackTrace);
            mLastData = null;
        }
    }

    /**
     * Manages the Activity result in the context of sharing the screen.
     *
     * @param requestCode the obtained requestCode
     * @param resultCode  the obtained resultCode
     * @param data        incoming data
     * @return true if the logic was managed, false otherwise.
     */
    public boolean onActivityResult(int requestCode, int resultCode, @NonNull Intent data) {
        if (requestCode == REQUEST_SCREEN_CAPTURE) {
            if (Activity.RESULT_OK != resultCode) {
                VoxeetSDK.conference().onParticipantCanceledScreenShare();
            } else {
                mLastResultCode = resultCode;
                mLastData = data;
                return true;
            }
        }
        return false;
    }

    /**
     * Gets the current screen size information.
     *
     * @param context context that is a source of screen size information
     * @return the valid point information.
     */
    @NonNull
    public Point getScreenSize(@NonNull Context context) {
        Point point = new Point();

        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        if (null != windowManager) {
            windowManager.getDefaultDisplay().getSize(point);
        }

        return point;
    }

    /**
     * Gets the scaled width (from the given screen information) according to the requested height.
     *
     * @param screen_size point object containing information about the width and the hight of the screen
     * @param height      the requested height of the screen
     * @return the scaled up/down screen size.
     */
    @NonNull
    public Point getScreenSizeScaled(@NonNull Point screen_size, int height) {
        Point point = new Point(screen_size);
        int screen_width = screen_size.x;
        int screen_height = screen_size.y;

        if (screen_height > 0) {
            point.y = height;
            point.x = (int) (screen_width * height * 1f / screen_height);
        }
        return point;
    }

    private boolean isScreenShareOn() {
        ConferenceInformation information = VoxeetSDK.conference().getCurrentConference();
        return null != information && information.isScreenShareOn();
    }
}
