package com.voxeet.sdk.services;

import android.support.annotation.NonNull;

import com.voxeet.VoxeetSDK;
import com.voxeet.promise.Promise;
import com.voxeet.sdk.json.RecordingStatusUpdatedEvent;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.models.v1.RecordingStatus;
import com.voxeet.sdk.network.endpoints.IRestApiRecording;
import com.voxeet.sdk.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.services.conference.promises.StartRecordingPromiseable;
import com.voxeet.sdk.services.conference.promises.StopRecordingPromiseable;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.Date;

/**
 * The RecordingService allows an application to record conferences by using the [start](/documentation/sdk/reference/android/recording#start) and [stop](/documentation/sdk/reference/android/recording#stop) methods that turn the recording on and off. The [RecordingStatusUpdatedEvent](/documentation/sdk/reference/android/recording#recordingstatusupdatedevent) informs about the status of the recording and additional conference details.
 */
@Annotate
public class RecordingService extends AbstractVoxeetService {

    private final static String TAG = RecordingService.class.getSimpleName();

    private boolean isRecording = false;

    private boolean isICERestartEnabled = false;

    @NoDocumentation
    public RecordingService(@NonNull SdkEnvironmentHolder instance) {
        super(instance);
        registerEventBus();
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Starts recording a part of the conference that is currently joined by the local participant.
     * When the conference is ended or recording is stopped, the recorded parts are automatically concentrated by the server. It is possible to replay them by using the [replay](/documentation/sdk/reference/android/conference#replay) method of the [ConferenceService](/documentation/sdk/reference/android/conference).
     *
     * @return the promise to resolve that indicates the result of the request.
     */
    public Promise<Boolean> start() {
        ConferenceService service = VoxeetSDK.conference();
        return new StartRecordingPromiseable(service,
                VoxeetSDK.mediaDevice(),
                getService(IRestApiRecording.class),
                service.getCurrentConference(),
                getEventBus()).createPromise();
    }

    /**
     * Stops recording the conference.
     *
     * @return the promise to resolve that indicates the result of the request.
     */
    public Promise<Boolean> stop() {
        ConferenceService service = VoxeetSDK.conference();
        return new StopRecordingPromiseable(service,
                VoxeetSDK.mediaDevice(),
                getService(IRestApiRecording.class),
                service.getCurrentConference(),
                getEventBus()).createPromise();
    }


    @Subscribe(threadMode = ThreadMode.MAIN, priority = 999)
    public void onEvent(final RecordingStatusUpdatedEvent recordingStatusUpdateEvent) {
        ConferenceService service = VoxeetSDK.conference();
        String conferenceId = recordingStatusUpdateEvent.conferenceId;

        ConferenceInformation information = service.getConferenceInformation(conferenceId);

        Conference conference = information.getConference();
        Participant participant = service.findParticipantById(recordingStatusUpdateEvent.participantId);
        if (participant != null) {

            Conference.RecordingInformation recording = new Conference.RecordingInformation();
            RecordingStatus status = RecordingStatus.valueOf(recordingStatusUpdateEvent.recordingStatus);
            if (status == RecordingStatus.RECORDING) {
                isRecording = true;
                recording.setStartRecordTimestamp(new Date(recordingStatusUpdateEvent.timeStamp));
                recording.setRecordingStatus(RecordingStatus.RECORDING);
                recording.setRecordingParticipant(recordingStatusUpdateEvent.participantId);
                //TODO participant owner ?
                //participant.setIsRecordingOwner(true);
            } else {
                isRecording = false;
                recording.setStartRecordTimestamp(null);
                recording.setRecordingStatus(RecordingStatus.NOT_RECORDING);
                recording.setRecordingParticipant(null);
                //TODO participant owner ?
                //participant.setIsRecordingOwner(false);
            }

            conference.setRecordingInformation(recording);
        }
    }

}