package com.voxeet.sdk.services;

import android.support.annotation.NonNull;
import android.util.Log;

import com.voxeet.VoxeetSDK;
import com.voxeet.promise.Promise;
import com.voxeet.sdk.json.ParticipantInfo;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.models.v2.ServerErrorOrigin;
import com.voxeet.sdk.network.endpoints.IRestApiSubscribe;
import com.voxeet.sdk.push.center.NotificationCenter;
import com.voxeet.sdk.push.center.SubscriptionCenter;
import com.voxeet.sdk.push.center.subscription.register.BaseSubscription;
import com.voxeet.sdk.push.center.subscription.register.SubscribeInvitation;
import com.voxeet.sdk.push.utils.Annotate;
import com.voxeet.sdk.services.notification.events.ConferenceStatusNotificationEvent;
import com.voxeet.sdk.services.notification.internal.WebSocketConferenceCreated;
import com.voxeet.sdk.services.notification.internal.WebSocketConferenceEnded;
import com.voxeet.sdk.services.notification.internal.WebSocketParticipantJoined;
import com.voxeet.sdk.services.notification.internal.WebSocketParticipantLeft;
import com.voxeet.sdk.services.notification.internal.WebsocketConferenceStatusNotificationEvent;
import com.voxeet.sdk.utils.Callback;
import com.voxeet.sdk.utils.Getter;
import com.voxeet.sdk.utils.HttpHelper;
import com.voxeet.sdk.utils.Map;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.Opt;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import okhttp3.ResponseBody;

/**
 * The NotificationService allows the application to set the desired way of notifying about conference invitations.
 *<p>
 * **Typical application workflow for the inviter:**
 *<p>
 * - The application calls the [invite](/documentation/sdk/reference/android/notification#invite) method to invite specific participants to a conference.
 *<p>
 *<p>
 * **Typical application workflow for the receiver:**
 *<p>
 * **1.** The application calls the [register](/documentation/sdk/reference/android/models/notificationcenter#register) method to register the effective notification mode.
 *<p>
 * **2.** The application calls the [register](/documentation/sdk/reference/android/models/notificationcenter#register-1) method to customize notifications. This method registers filters which reject specific invitations.
 *<p>
 * **3.** The application calls the [setNotificationTokenProvider](/documentation/sdk/reference/android/models/notificationcenter#setnotificationtokenprovider) method to change the notification provider. The [getNotificationTokenProvider](/documentation/sdk/reference/android/notification#getnotificationtokenprovider) method checks if the [INotificationTokeProvider](/documentation/sdk/reference/android/models/inotificationtokenprovider) model is registered to the SDK.
 *<p>
 * **4.** Optionally, the application can change the notification management mode into the [enforcedNotificationMode](/documentation/sdk/reference/android/models/enforcednotificationmode) by calling the [setEnforcedNotificationMode](/documentation/sdk/reference/android/models/notificationcenter#setenforcednotificationmode) method.
 *<p>
 * **5.** The application calls the [InvitationBundle](/documentation/sdk/reference/android/models/invitationbundle#invitationbundle) constructor from the [InvitationBundle](/documentation/sdk/reference/android/models/invitationbundle) model to transform notifications.
 *<p>
 * **6.** The application calls the [onInvitationReceived](/documentation/sdk/reference/android/models/notificationcenter#oninvitationreceived) or [onInvitationCanceledReceived](/documentation/sdk/reference/android/models/notificationcenter#oninvitationcanceledreceived) method to show the proper notification.
 *<p>
 * The Subscriptions can be :
 *<p>
 * - [SubscribeConferenceCreated](/documentation/sdk/reference/android/models/subscribeConferencecreated)
 *<p>
 *  - [SubscribeConferenceEnded](/documentation/sdk/reference/android/models/subscribeconferenceended)
 *<p>
 *  - [SubscribeInvitation](/documentation/sdk/reference/android/models/subscribeinvitation)
 *<p>
 *  - [SubscribeParticipantJoined](/documentation/sdk/reference/android/models/subscribeparticipantjoined)
 *<p>
 *  - [SubscribeParticipantLeft](/documentation/sdk/reference/android/models/subscribeparticipantleft)
 */
@Annotate
public class NotificationService extends AbstractVoxeetService {

    private Events events = new Events();

    @NoDocumentation
    public NotificationService(@NonNull SdkEnvironmentHolder instance) {
        super(instance);

        NotificationCenter.instance.init(instance.voxeetSdk.getEventBus(), instance.voxeetSdk.getApplicationContext());
        instance.voxeetSdk.register(events);

        subscribe(new SubscribeInvitation()).execute();
    }

    /**
     * Instance of the NotificationCenter responsible of every settings and event regarding the invitations and modes to use
     */
    public final NotificationCenter center = NotificationCenter.instance;

    final SubscriptionCenter subscription = SubscriptionCenter.instance;

    /**
     * Turns on the subscription for the specified notifications.
     *
     * @param subscriptions list of Subscription
     * @return a flag indicating if the subscription could have been done
     */
    public Promise<Boolean> subscribe(@NonNull BaseSubscription... subscriptions) {
        return subscribe(new ArrayList<>(Arrays.asList(subscriptions)));
    }

    /**
     * Turns on the subscription for the specified notifications.
     *
     * @param subscriptions list of Subscription
     * @return a flag indicating if the subscription could have been done
     */
    public Promise<Boolean> subscribe(@NonNull List<BaseSubscription> subscriptions) {
        return apply(
                subscriptions,
                SubscriptionCenter.instance::subscribe,
                () -> getService(IRestApiSubscribe.class).subscribe(new IRestApiSubscribe.Subscriptions(subscriptions)),
                ServerErrorOrigin.SUBSCRIBE
        );
    }

    /**
     * Turns off the subscription for the specified notifications.
     *
     * @param subscriptions list of Subscription
     * @return a flag indicating if the subscription could have been done
     */
    public Promise<Boolean> unsubscribe(@NonNull BaseSubscription... subscriptions) {
        return unsubscribe(new ArrayList<>(Arrays.asList(subscriptions)));
    }

    /**
     * Turns off the subscription for the specified notifications.
     *
     * @param subscriptions list of Subscription
     * @return a flag indicating if the subscription could have been done
     */
    public Promise<Boolean> unsubscribe(@NonNull List<BaseSubscription> subscriptions) {
        return apply(
                subscriptions,
                SubscriptionCenter.instance::unsubscribe,
                () -> getService(IRestApiSubscribe.class).unsubscribe(new IRestApiSubscribe.Subscriptions(subscriptions)),
                ServerErrorOrigin.UNSUBSCRIBE
        );
    }

    private Promise<Boolean> apply(@NonNull List<BaseSubscription> list,
                                   @NonNull Callback<BaseSubscription> centerApply,
                                   @NonNull Getter<retrofit2.Call<ResponseBody>> endpoint,
                                   @NonNull ServerErrorOrigin origin) {
        if (Opt.of(list).then(List::size).or(0) == 0)
            return Promise.reject(new IllegalStateException("Need to have subscribable object"));
        boolean needNetworkCall = Map.find(list, iSubscribe -> Opt.of(iSubscribe).then(BaseSubscription::isNetworkCall).or(false)) != null;

        if (!needNetworkCall) {
            Map.safeApply(list, centerApply);
            return Promise.resolve(true);
        }

        return new Promise<>(solver -> {
            retrofit2.Call<ResponseBody> call = endpoint.get();
            HttpHelper.promise(call, origin).then(result -> {
                Map.safeApply(list, centerApply);
                solver.resolve(true);
            }).error(solver::reject);
        });
    }


    /**
     * Invites a list of users for a specific conference.
     *
     * @param conferenceId       conference ID of the conference
     * @param participantOptions information about the participants
     * @return a non null promise to resolve.
     */
    @NoDocumentation
    @Deprecated
    @NonNull
    public Promise<List<Participant>> invite(@NonNull final String conferenceId, @NonNull final List<ParticipantInfo> participantOptions) {
        return VoxeetSDK.conference().invite(conferenceId, participantOptions);
    }

    private class Events {
        @Subscribe(threadMode = ThreadMode.MAIN)
        public void onEvent(@NonNull WebSocketConferenceCreated event) {
            subscription.onEvent(event.to());
        }

        @Subscribe(threadMode = ThreadMode.MAIN)
        public void onEvent(@NonNull WebSocketConferenceEnded event) {
            subscription.onEvent(event.to());
        }

        @Subscribe(threadMode = ThreadMode.MAIN)
        public void onEvent(@NonNull WebSocketParticipantJoined event) {
            subscription.onEvent(event.to());
        }

        @Subscribe(threadMode = ThreadMode.MAIN)
        public void onEvent(@NonNull WebSocketParticipantLeft event) {
            subscription.onEvent(event.to());
        }

        @Subscribe(threadMode = ThreadMode.MAIN)
        public void onEvent(@NonNull WebsocketConferenceStatusNotificationEvent event) {
            getEventBus().post(new ConferenceStatusNotificationEvent(event.conferenceId,
                    event.conferenceAlias,
                    event.isLive,
                    event.startTimestamp,
                    event.asParticipants()));
        }
    }

}
