package com.voxeet.sdk.media.audio;

import android.content.Context;
import android.media.Ringtone;
import android.media.RingtoneManager;
import android.net.Uri;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.audio.listeners.IAudioRouteListener;
import com.voxeet.audio.utils.Constants;
import com.voxeet.audio.utils.Log;
import com.voxeet.audio.utils.__Call;
import com.voxeet.audio2.AudioDeviceManager;
import com.voxeet.audio2.devices.BluetoothDevice;
import com.voxeet.audio2.devices.MediaDevice;
import com.voxeet.audio2.devices.description.ConnectionState;
import com.voxeet.audio2.devices.description.DeviceType;
import com.voxeet.promise.Promise;
import com.voxeet.promise.solve.ThenPromise;
import com.voxeet.sdk.events.sdk.AudioRouteChangeEvent;
import com.voxeet.sdk.media.MediaPowerManager;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.AudioType;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.Opt;
import com.voxeet.sdk.utils.SoundPool;
import com.voxeet.sdk.utils.Validate;

import org.greenrobot.eventbus.EventBus;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * The SoundManager model manages calls to the lower audio stack management. It allows the application to:
 * - Get the [available](/documentation/sdk/reference/android/models/soundmanager#getavailableroutes) or [current](/documentation/sdk/reference/android/models/soundmanager#currentroute) routes and [set](/documentation/sdk/reference/android/models/soundmanager#setaudioroute) audio routes.
 * - Get [system ringtones](/documentation/sdk/reference/android/models/soundmanager#getsystemringtone).
 * - Check if [Bluetooth](/documentation/sdk/reference/android/models/soundmanager#isbluetoothheadsetconnected) or [wired](/documentation/sdk/reference/android/models/soundmanager#iswiredheadseton) headset is connected.
 * - Check if an external [speaker](/documentation/sdk/reference/android/models/soundmanager#isspeakeron) is connected and [change](/documentation/sdk/reference/android/models/soundmanager#setspeakermode) its state.
 * - [Acquire](/documentation/sdk/reference/android/models/soundmanager#acquirelocks) and [release](/documentation/sdk/reference/android/models/soundmanager#releaselocks) locks for internal managers.
 * - [Set](/documentation/sdk/reference/android/models/soundmanager#setmediaroute) and [unset](/documentation/sdk/reference/android/models/soundmanager#unsetmediaroute) media mode and call mode. It also [sets](/documentation/sdk/reference/android/models/soundmanager#enablemedia) the current mode to the media mode.
 * - [Calculate](/documentation/sdk/reference/android/models/soundmanager#checkoutputroute) the output route and set the proper internal state.
 * - Set the instance to [acquire](/documentation/sdk/reference/android/models/soundmanager#requestaudiofocus) or [release](/documentation/sdk/reference/android/models/soundmanager#abandonaudiofocusrequest) the audio focus.
 * - [Set](/documentation/sdk/reference/android/models/soundmanager#setsound) the audio type to a proper asset reference and [set](/documentation/sdk/reference/android/models/soundmanager#setsound) the asset reference name for a specific sound and audio type.
 * - [Play](/documentation/sdk/reference/android/models/soundmanager#playsoundtype) a specific audio type in a stream voice call. It also enables to [force](/documentation/sdk/reference/android/models/soundmanager#playsoundtypeforce) it and [stop specific audio type](/documentation/sdk/reference/android/models/soundmanager#stopsoundtype) or [stop playing sound in a specific mode](/documentation/sdk/reference/android/models/soundmanager#stop).
 * - [Reset](/documentation/sdk/reference/android/models/soundmanager#resetdefaultsoundtype) internal sound type to the default type.
 * - [Get](/documentation/sdk/reference/android/models/soundmanager#getsystemringtone) the ringtone established by the Android system.
 * - [Enable](/documentation/sdk/reference/android/models/soundmanager#enable) and [disable](/documentation/sdk/reference/android/models/soundmanager#disable) the sound and internal manager.
 * - Indicate that the conference is in the [joining](/documentation/sdk/reference/android/models/soundmanager#onconferenceprejoinedevent) state, or it is [destroyed](/documentation/sdk/reference/android/models/soundmanager#onconferencedestroyedpush).
 */
@Annotate
public class SoundManager implements IAudioRouteListener {

    private static final String TAG = SoundManager.class.getSimpleName();
    private AudioDeviceManager mAudioDeviceManager;
    private MediaPowerManager mMediaPowerManager;
    private Context mContext;
    private HashMap<Integer, SoundPool> _sound_pools;
    private HashMap<AudioType, String> mSounds;

    private boolean enabled;
    private List<MediaDevice> mediaDevices;
    private MediaDevice current;

    private ArrayList<Call<List<MediaDevice>>> callbacks = new ArrayList<>();

    private SoundManager() {
        disable();
    }

    @NoDocumentation
    public SoundManager(@NonNull Context context) {
        this();

        mediaDevices = new ArrayList<>();

        _sound_pools = new HashMap<>();

        mAudioDeviceManager = new AudioDeviceManager(context, SoundManager.this::updateDevices);

        //mAudioDeviceManager.setMediaRoute();
        //mAudioDeviceManager.configureVolumeStream(Constants.STREAM_VOICE_CALL, Constants.STREAM_MUSIC);

        mMediaPowerManager = new MediaPowerManager(context,
                currentRoute());

        mContext = context;

        mSounds = new HashMap<>();
        configure();
    }

    /**
     * Register external listeners for update on platform MediaDevice.
     *
     * @param callback callback to register
     * @return informs if the register call has been properly be done.
     */
    public boolean registerUpdateDevices(Call<List<MediaDevice>> callback) {
        if (!callbacks.contains(callback)) {
            callbacks.add(callback);
            return true;
        }
        return false;
    }

    /**
     * Unregister an external listener for update on platform MediaDevice.
     *
     * @param callback callback to unregister
     */
    public void unregisterUpdateDevices(Call<List<MediaDevice>> callback) {
        callbacks.remove(callback);
    }

    private void updateDevices(Promise<List<MediaDevice>> update) {
        update.then((ThenPromise<List<MediaDevice>, MediaDevice>) mediaDevices -> {
            this.mediaDevices = mediaDevices;
            mMediaPowerManager.setAudioRoute(currentRoute());
            return mAudioDeviceManager.current();
        }).then(current -> {
            this.current = current;
            for (Call<List<MediaDevice>> callback : callbacks) {
                try {
                    callback.apply(mediaDevices);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }).error(Throwable::printStackTrace);
    }

    /**
     * Gets currently available routes.
     *
     * @return a non-null route.
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        List<AudioRoute> routes = new ArrayList<>();
        for (MediaDevice device : mediaDevices) {
            AudioRoute route = AudioRoute.from(device);
            if (!routes.contains(route)) {
                routes.add(route);
            }
        }
        return routes;
    }

    /**
     * Gets the current route calculated on the current internal manager state.
     *
     * @return the currently calculated route.
     */
    @NonNull
    public AudioRoute currentRoute() {
        if (null != current) {
            return AudioRoute.from(current);
        }
        return AudioRoute.ROUTE_MEDIA;
    }

    /**
     * List the audio media devices on the platform.
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<List<MediaDevice>> enumerateDevices() {
        return mAudioDeviceManager.enumerateDevices();
    }

    /**
     * Connect the SDK to the specified device.
     *
     * @param device device
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Boolean> connect(@NonNull MediaDevice device) {
        return mAudioDeviceManager.connect(device);
    }

    /**
     * Disconnect the SDK from the specified device.
     *
     * @param device device
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Boolean> disconnect(@NonNull MediaDevice device) {
        return mAudioDeviceManager.disconnect(device);
    }

    /**
     * Sets the current route forcing a newly calculated route that is compatible with the current system state. For example, it is impossible to enable Bluetooth if Bluetooth headset is not connected. It starts working after connecting Bluetooth headset.
     *
     * @param route valid audio route
     * @return the result of an internal call.
     */
    @Deprecated
    public boolean setAudioRoute(@NonNull AudioRoute route) {
        Validate.runningOnUiThread();
        MediaDevice device = oneOf(route);
        if (null != device) {
            _connect(device, "setAudioRoute to " + device.id());
            return true;
        }
        return false;
    }

    /**
     * Gets the Ringtone from an Android system.
     *
     * @return the Ringtone instance corresponding to the instance of the current system.
     */
    @Nullable
    public Ringtone getSystemRingtone() {
        Uri uri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_RINGTONE);
        if (null == uri) return null;
        return RingtoneManager.getRingtone(mContext, uri);
    }

    /**
     * Checks for a headset that is connected to the device.
     *
     * @return the connection state.
     */
    public boolean isBluetoothHeadsetConnected() {
        return isPlatformConnected(DeviceType.BLUETOOTH, ConnectionState.CONNECTED);
    }

    /**
     * Checks for wired headset connected to the device.
     *
     * @return the connection state.
     */
    public boolean isWiredHeadsetOn() {
        return isPlatformConnected(DeviceType.WIRED_HEADSET, ConnectionState.CONNECTED);
    }

    /**
     * Check if one of the specified DeviceType is currently connected to the native platform
     *
     * @param deviceType device type
     * @param state connection state to the native platform
     * @return informs about the state.
     */
    public boolean isPlatformConnected(@NonNull DeviceType deviceType, @NonNull ConnectionState state) {
        return Opt.of(oneOf(deviceType)).then(c -> state.equals(c.platformConnectionState())).or(false);
    }

    /**
     * Changes the state of output or an internal speaker.
     *
     * @param isSpeaker option to change the speaker to the external one
     * @return the current instance to chain calls.
     */
    @Deprecated
    @NonNull
    public SoundManager setSpeakerMode(boolean isSpeaker) {
        MediaDevice toConnect = null;
        Log.d(TAG, "setSpeakerMode: enabled ?" + enabled);

        MediaDevice[] devices = new MediaDevice[]{oneOf(DeviceType.WIRED_HEADSET), oneOf(DeviceType.BLUETOOTH),};
        //first, check for bluetooth or wired
        for (MediaDevice device : devices) {
            if (toConnect == null && null != device && ConnectionState.CONNECTED.equals(device.platformConnectionState())) {
                toConnect = device;
            }
        }

        //then get the external/internal if not the primary ones are available
        if (null == toConnect) {
            toConnect = oneOf(isSpeaker ? DeviceType.EXTERNAL_SPEAKER : DeviceType.INTERNAL_SPEAKER);
        }

        if (null != toConnect) {
            _connect(toConnect, "setSpeakerMode for " + toConnect.id());
        } else {
            Log.d(TAG, "unexpected null for setSpeakerMode !!");
        }

        return this;
    }

    /**
     * Checks an external speaker mode.
     *
     * @return the current speaker state.
     */
    @Deprecated
    public boolean isSpeakerOn() {
        //return AudioRoute.ROUTE_SPEAKER.equals(mAudioDeviceManager.outputRoute());
        return Opt.of(oneOf(AudioRoute.ROUTE_SPEAKER)).then(c -> ConnectionState.CONNECTED.equals(c.connectionState())).or(false);
    }

    /**
     * Acquires locks for an internal manager.
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager acquireLocks() {
        mMediaPowerManager.acquire();
        return this;
    }

    /**
     * Releases locks for an internal manager.
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager releaseLocks() {
        mMediaPowerManager.release();
        return this;
    }

    /**
     * Resets an internal Sound Type to the default type.
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager resetDefaultSoundType() {
        abandonAudioFocusRequest();

        return this;
    }

    /**
     * Call the internal UiSoundsStreamType accessible from Android
     * <p>
     * For more information, go to the [android-sdk-audio](https://github.com/voxeet/android-sdk-audio) library
     *
     * @return the current integer representing the UiSoundsStreamType
     */
    private int getUiSoundsStreamType() {
        return Constants.STREAM_MUSIC;
    }

    /**
     * Sets the current mode to Media.
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager enableMedia() {
        enabled = true;
        setMediaRoute();
        return this;
    }

    /**
     * Unsets the Call mode to set the standard Media or a System type. Should not be used in conferences.
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager setMediaRoute() {
        MediaDevice device = oneOf(AudioRoute.ROUTE_MEDIA);
        if (null != device) {
            _connect(device, "setMediaRoute connect for " + device.id() + " done");
        }
        return this;
    }

    /**
     * Unsets the Media mode to set the Call mode. Should be used in conferences.
     *
     * @return the current instance to chain calls
     */
    @NonNull
    @Deprecated
    public SoundManager unsetMediaRoute() {
        MediaDevice device = oneOf(AudioRoute.ROUTE_MEDIA);
        if (null != device) {
            mAudioDeviceManager.disconnect(device).then(result -> {
                Log.d(TAG, "connect for " + device.id() + " done ? " + result);
            }).error(Throwable::printStackTrace);
        }
        return this;
    }

    /**
     * Makes the instance of the current application release the AudioFocus. For more information, see [android-sdk-audio](https://github.com/voxeet/android-sdk-audio).
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager abandonAudioFocusRequest() {
        mAudioDeviceManager.current().then((ThenPromise<MediaDevice, Boolean>) mediaDevice -> {
            if (null != mediaDevice) return mAudioDeviceManager.disconnect(mediaDevice);
            return Promise.resolve(true);
        }).then(result -> {
            Log.d(TAG, "abandonAudioFocusRequest done");
        }).error(Throwable::printStackTrace);

        return this;
    }

    /**
     * Makes the instance of the current application acquire the AudioFocus. For more information, see [android-sdk-audio](https://github.com/voxeet/android-sdk-audio).
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager requestAudioFocus() {
        mAudioDeviceManager.current().then((ThenPromise<MediaDevice, Boolean>) current ->
                Opt.of(current).then(c -> mAudioDeviceManager.connect(c))
                        .or(Promise.resolve(true)))
                .then(result -> {
                    Log.d(TAG, "requestAudioFocus done");
                })
                .error(Throwable::printStackTrace);
        return this;
    }

    /**
     * Calculates the output route and sets the internal state accordingly. Use this method each time it must be recalculated.
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager checkOutputRoute() {
        if (isEnabled()) {
            Log.d(TAG, "check the current route ! will call reconnect of the current connected device");
            requestAudioFocus();
        }
        return this;
    }

    private SoundPool getSoundPool(int soundMode) {
        SoundPool pool = _sound_pools.get(soundMode);
        if (pool == null) {
            pool = new SoundPool(mContext, soundMode, getVolume(soundMode));
            _sound_pools.put(soundMode, pool);
        }
        return pool;
    }

    private void configure() {
        //default disable the audio manager
        mAudioDeviceManager.enumerateDevices().then(mediaDevices -> {
            this.mediaDevices = mediaDevices;
        }).error(Throwable::printStackTrace);

        enabled = false;

        mSounds.put(AudioType.RING, "out.mp3");
        mSounds.put(AudioType.HANGUP, "leave.mp3");

        setSound(AudioType.RING, mSounds.get(AudioType.RING));
        setSound(AudioType.HANGUP, mSounds.get(AudioType.HANGUP));
    }

    /**
     * Sets the AudioType to the given asset reference in the voice call mode.
     *
     * @param type      AudioType to set in the stream voice call
     * @param assetName name of the asset
     * @return the current instance to chain calls.
     */
    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return setSound(type, assetName, Constants.STREAM_VOICE_CALL);
    }

    /**
     * Sets the asset reference name for a given soundMode and an AudioType.
     *
     * @param type      AudioType to set in the stream voice call
     * @param assetName name of the asset
     * @param soundMode sound mode for the system
     * @return the current instance to chain calls.
     */
    public boolean setSound(@NonNull AudioType type, @NonNull String assetName, int soundMode) {
        return getSoundPool(soundMode).release(type).setShortResource(type, assetName);
    }

    /**
     * Plays given AudioType in a stream voice call.
     *
     * @param type AudioType
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager playSoundType(@NonNull AudioType type) {
        return playSoundType(type, Constants.STREAM_VOICE_CALL);
    }

    /**
     * Plays a given AudioType in a stream voice call for a given sound mode.
     *
     * @param type      AudioType
     * @param soundMode sound mode
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager playSoundType(@NonNull AudioType type, int soundMode) {
        if (isEnabled()) {
            getSoundPool(soundMode).playShortResource(type, mSounds.get(type));
        }
        return this;
    }

    /**
     * Forces a given AudioType sound to be played in a stream voice call.
     *
     * @param type AudioType
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager playSoundTypeForce(@NonNull AudioType type) {
        return playSoundType(type, Constants.STREAM_VOICE_CALL);
    }

    /**
     * Forces a given AudioType sound to be played in the given sound mode.
     *
     * @param type      AudioType
     * @param soundMode sound mode
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager playSoundTypeForce(@NonNull AudioType type, int soundMode) {
        getSoundPool(soundMode).playShortResource(type, mSounds.get(type));
        return this;
    }

    /**
     * Stops playing the specified type of sound for the stream voice call.
     *
     * @param audioType AudioType
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager stopSoundType(@NonNull AudioType audioType) {
        return stopSoundType(audioType, Constants.STREAM_VOICE_CALL);
    }

    /**
     * Stops playing the specified type of sound in a given mode.
     *
     * @param audioType AudioType
     * @param soundMode sound mode
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager stopSoundType(@NonNull AudioType audioType, int soundMode) {
        getSoundPool(soundMode).stop(audioType);
        return this;
    }

    /**
     * Stops the sound that is played in a given mode.
     *
     * @param soundMode sound mode
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager stop(int soundMode) {
        getSoundPool(soundMode).stop();
        return this;
    }

    /**
     * Stops the sound that is played in a stream voice call mode.
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager stop() {
        stop(Constants.STREAM_VOICE_CALL);
        return this;
    }

    /**
     * Calls when a conference is in a joining state.
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager onConferencePreJoinedEvent() {
        MediaDevice[] devices = new MediaDevice[]{
                oneOf(DeviceType.BLUETOOTH),
                oneOf(DeviceType.WIRED_HEADSET),
                oneOf(DeviceType.EXTERNAL_SPEAKER),
                oneOf(DeviceType.INTERNAL_SPEAKER)
        };

        for (MediaDevice device : devices) {
            if (null != device && ConnectionState.CONNECTED.equals(device.platformConnectionState())) {
                _connect(device, "onConferencePreJoinedEvent done ");
            }
        }

        return this;
    }

    private void _connect(@NonNull MediaDevice device, String text) {
        mAudioDeviceManager.connect(device).then(result -> {
            Log.d(TAG, text + " ? " + result);
        }).error(Throwable::printStackTrace);
    }

    /**
     * Calls when a conference is destroyed.
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager onConferenceDestroyedPush() {
        abandonAudioFocusRequest();
        return this;
    }

    /**
     * Enables this and the internal manager.
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager enable() {
        enabled = true;
        return this;
    }

    /**
     * Disables this and the internal manager.
     *
     * @return the current instance to chain calls.
     */
    @NonNull
    public SoundManager disable() {
        enabled = false;
        return this;
    }

    private boolean isEnabled() {
        return enabled;
    }

    private float getVolume(int soundMode) {
        if (soundMode == Constants.STREAM_VOICE_CALL) {
            return 0.1f;
        }
        return 1.f;
    }

    @NoDocumentation
    @Override
    public void onAudioRouteChanged() {
        EventBus.getDefault().post(new AudioRouteChangeEvent());
    }

    @NoDocumentation
    @Nullable
    public MediaDevice oneOf(@NonNull AudioRoute route) {
        for (MediaDevice device : mediaDevices) {
            if (route.isCompatible(device) && isNotFiltered(device)) {
                return device;
            }
        }
        return null;
    }

    @NoDocumentation
    @Nullable
    public MediaDevice oneOf(@NonNull DeviceType deviceType) {
        for (MediaDevice device : mediaDevices) {
            if (deviceType.equals(device.deviceType()) && isNotFiltered(device)) {
                return device;
            }
        }
        return null;
    }

    /**
     * Check if the device is not to be filtered. For instance, SmartWatches may be registered as compatible by the system but most won't be actually be compatible.
     * @param device device
     * @return informs about the compatibility.
     */
    public boolean isNotFiltered(@NonNull MediaDevice device) {
        if (device instanceof BluetoothDevice) {
            android.bluetooth.BluetoothDevice bt = ((BluetoothDevice) device).bluetoothDevice();
            return Opt.of(bt).then(android.bluetooth.BluetoothDevice::getName)
                    .then(String::toLowerCase)
                    .then(s -> !s.contains("watch")).or(false);
        }
        return true;
    }

    /**
     * Get the currently connected media device.
     *
     * @return the promise to resolve.
     */
    public Promise<MediaDevice> current() {
        return Promise.resolve(current);
    }

    public static interface Call<TYPE> extends __Call<TYPE> {

    }
}
