package com.voxeet.sdk.media.audio;

import android.support.annotation.NonNull;

import com.voxeet.audio2.devices.MediaDevice;
import com.voxeet.sdk.utils.Annotate;

/**
 * Specific routes for audio devices on Android. The usage of this has been deprecated in favor of direct media devices invocation in the AudioService instance. Using them
 *
 * The possible routes are :
 * - ROUTE_HEADSET
 * - ROUTE_PHONE
 * - ROUTE_SPEAKER
 * - ROUTE_BLUETOOTH
 * - ROUTE_MEDIA
 */
@Annotate
public enum AudioRoute {
    ROUTE_HEADSET(false),
    ROUTE_PHONE(true),
    ROUTE_SPEAKER(false),
    ROUTE_BLUETOOTH(false),
    ROUTE_MEDIA(false);

    private boolean mProximitySensor;

    AudioRoute(boolean proximity_sensor) {
        mProximitySensor = proximity_sensor;
    }

    public boolean useProximitySensor() {
        return mProximitySensor;
    }

    public static AudioRoute valueOf(int value) {
        switch (value) {
            case 0:
                return ROUTE_HEADSET;
            case 1:
                return ROUTE_PHONE;
            case 2:
                return ROUTE_SPEAKER;
            case 3:
                return ROUTE_BLUETOOTH;
            case 4:
                return ROUTE_MEDIA;
            default:
                return ROUTE_SPEAKER;
        }
    }

    /**
     * Informs about the AudioRoute type for a given MediaDevice
     *
     * @param device device
     * @return The corresponding AudioRoute. _(ROUTE_MEDIA by default)_
     */
    public static AudioRoute from(@NonNull MediaDevice device) {
        switch (device.deviceType()) {
            case BLUETOOTH:
                return AudioRoute.ROUTE_BLUETOOTH;
            case NORMAL_MEDIA:
                return AudioRoute.ROUTE_MEDIA;
            case WIRED_HEADSET:
                return AudioRoute.ROUTE_HEADSET;
            case EXTERNAL_SPEAKER:
                return AudioRoute.ROUTE_SPEAKER;
            case INTERNAL_SPEAKER:
                return AudioRoute.ROUTE_PHONE;
            case USB:
            default:
                return AudioRoute.ROUTE_MEDIA;
        }
    }

    /**
     * Check if the given MediaDevice is compatible with the given AudioRoute
     *
     * @param device media device
     * @return compatibility flag
     */
    public boolean isCompatible(@NonNull MediaDevice device) {
        switch (device.deviceType()) {
            case BLUETOOTH:
                return this.equals(AudioRoute.ROUTE_BLUETOOTH);
            case NORMAL_MEDIA:
                return this.equals(AudioRoute.ROUTE_MEDIA);
            case WIRED_HEADSET:
                return this.equals(AudioRoute.ROUTE_HEADSET);
            case EXTERNAL_SPEAKER:
                return this.equals(AudioRoute.ROUTE_SPEAKER);
            case INTERNAL_SPEAKER:
                return this.equals(AudioRoute.ROUTE_PHONE);
            case USB:
            default:
                return this.equals(AudioRoute.ROUTE_MEDIA);
        }
    }
}