package com.voxeet.sdk.media;

import android.content.Context;
import android.media.MediaCodecInfo;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.SdpCandidate;
import com.voxeet.android.media.crypto.AbstractNativeMediaCryptoCallback;
import com.voxeet.sdk.media.peer.PendingPeerCallback;
import com.voxeet.sdk.media.peer.PendingPeerOperation;
import com.voxeet.sdk.media.peer.SdpCandidates;
import com.voxeet.sdk.media.peer.SdpDescription;
import com.voxeet.sdk.media.peer.Type;

import org.greenrobot.eventbus.EventBus;
import org.webrtc.CameraVideoCapturer;
import org.webrtc.EglBase;
import org.webrtc.MediaCodecVideoHelperFactory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Wrapper around the Media class
 *
 * @author Kévin Le Perf
 */
public class MediaSDK extends MediaEngine {
    public static final String TAG = MediaSDK.class.getSimpleName();
    private static long DescriptionTimeout = 5000;

    private final static Object mOperationLock = new Object();
    private final static Object mCandidateLock = new Object();

    @NonNull
    private SdpCandidates mPeerCandidates = new SdpCandidates();

    @NonNull
    private List<PendingPeerOperation> mPeerOperations = new ArrayList<>();

    public static void setDescriptionTimeout(long descriptionTimeout) {
        DescriptionTimeout = descriptionTimeout;
    }

    public static Context context() {
        return MediaEngine.Context;
    }

    public MediaSDK(@NonNull final Context context,
                    @NonNull final String peer,
                    @NonNull StreamListener listener,
                    @NonNull CameraVideoCapturer.CameraEventsHandler cameraListener,
                    boolean startWithVideo,
                    boolean microphone,
                    @Nullable AbstractNativeMediaCryptoCallback delegate) throws MediaEngineException {
        super(context, peer, listener, cameraListener, startWithVideo, microphone, delegate);

        try {
            Log.d(TAG, "MediaSDK: constructed an instance of Media with the following codecs available");
            ArrayList<MediaCodecInfo> list = MediaCodecVideoHelperFactory.getCodecs();
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                for (MediaCodecInfo info : list) {
                    Log.d(TAG, "codec :: " + info.getName() + " encoder:=" + info.isEncoder() + " types:=" + Arrays.toString(info.getSupportedTypes()));
                }
            }
        } catch (Exception e) {

        }
    }

    @Override
    protected void onEglBaseRecreated(@NonNull EglBase base) {
        EventBus.getDefault().post(new EglBaseRefreshEvent(base));
    }

    public void createAnswerForPeer(@NonNull final String peer,
                                    final long ssrc,
                                    @NonNull final SdpDescription offer,
                                    @NonNull final List<SdpCandidate> offerCandidates,
                                    boolean isMaster,
                                    @NonNull final PendingPeerCallback listener) throws MediaEngineException {
        createConnection(peer, isMaster);

        String exception = setPeerDescription(peer, ssrc, offer.type, offer.sdp);

        if (null != exception) {
            throw new MediaEngineException(String.format("Unable set remote SDP for Peer: %s :: %s", peer, exception));
        } //else, exception == null -> everything went fine

        for (SdpCandidate c : offerCandidates) {
            setPeerCandidate(peer, c.sdpMid, c.sdpMLineIndex, c.sdp);
        }

        WaitForSdpMessage(peer, Type.ANSWER, listener);

        if (!createAnswer(peer)) {
            throw new MediaEngineException(String.format("Unable to create answer for Peer: %s", peer));
        }
    }

    public void addPeerFromAnswer(@NonNull String peer,
                                  long ssrc,
                                  @NonNull SdpDescription answer,
                                  @NonNull List<SdpCandidate> candidates) throws MediaEngineException {
        String exception = setPeerDescription(peer, ssrc, answer.type, answer.sdp);

        if (null != exception) {
            throw new MediaEngineException(String.format("Unable set remote SDP for Peer: %s :: %s", peer, exception));
        } //else, exception == null -> everything went fine

        for (SdpCandidate candidate : candidates) {
            setPeerCandidate(peer, candidate.sdpMid, candidate.sdpMLineIndex, candidate.sdp);
        }
    }

    //-------------------------------------------------------------------------------------------------
    // Callbacks
    //-------------------------------------------------------------------------------------------------

    @Override
    public void onSessionCreated(@NonNull String peer,
                                 @NonNull String type,
                                 @NonNull String sdp) {
        Log.d(TAG, String.format("Session description received for peer: %s with type: %s and SDP: %s", peer, type, sdp));

        Type operationType = Type.fromString(type);

        if (null != operationType) {
            unlockPeerOperation(operationType, peer, new SdpDescription(type, sdp));
        } else {
            Log.w(TAG, String.format("Session description received from an unknown type: %s", type));
        }
    }

    @Override
    public void onIceCandidateDiscovered(@NonNull String peer,
                                         @NonNull SdpCandidate[] candidates) {
        Log.d(TAG, String.format("ICE candidate discovered for peer: %s with id", peer));


        mStreamListener.onIceCandidateDiscovered(peer, candidates);

        synchronized (mCandidateLock) {
            for (SdpCandidate candidate : candidates) {
                mPeerCandidates.add(peer, candidate);
            }
        }
    }

    @Override
    public void onIceGatheringComplete(@NonNull String peer) {
        Log.d(TAG, String.format("ICE candidate gathering complete for peer: %s", peer));

        List<SdpCandidate> candidates = null;
        SdpDescription description = null;

        synchronized (mCandidateLock) {
            candidates = mPeerCandidates.candidates(peer);
        }

        if (candidates.isEmpty()) {
            Log.w(TAG, String.format("No ICE candidate gathered for peer: %s", peer));
        }

        unlockPeerOperation(Type.CANDIDATES, peer, description);
    }

    /**
     * For compatibility reasons, set an empty one
     * <p>
     * TODO null check in the wrapper
     */
    public void unsetStreamListener() {
        mStreamListener = new StreamListener() {
            @Override
            public void onStreamAdded(@NonNull String peer, @NonNull MediaStream stream) {

            }

            @Override
            public void onStreamUpdated(@NonNull String peer, @NonNull MediaStream stream) {

            }

            @Override
            public void onStreamRemoved(@NonNull String peer) {

            }

            @Override
            public void onScreenStreamAdded(@NonNull String peer, @NonNull MediaStream stream) {

            }

            @Override
            public void onScreenStreamRemoved(@NonNull String peer) {

            }

            @Override
            public void onShutdown() {

            }

            @Override
            public void onIceCandidateDiscovered(String peer, SdpCandidate[] candidates) {

            }
        };
    }

    //-------------------------------------------------------------------------------------------------
    // Helper methods
    //-------------------------------------------------------------------------------------------------

    private void createConnection(final String peer, final boolean master) throws MediaEngineException {
        if (!createPeerConnection(peer, master)) {
            throw new MediaEngineException(String.format("Unable to create connection with Peer: %s", peer));
        }
    }

    @NonNull
    private PendingPeerOperation createPeerOperation(@NonNull Type type,
                                                     @NonNull String peer) throws MediaEngineException {
        PendingPeerOperation operation = new PendingPeerOperation(type, peer);

        synchronized (mOperationLock) {
            if (mPeerOperations.contains(operation)) {
                throw new MediaEngineException(String.format("An operation of type: %s is already pending for peer: %s", type.name(), peer));
            }
        }

        mPeerOperations.add(operation);

        return operation;
    }

    private void unlockPeerOperation(@NonNull Type type,
                                     @NonNull String peer,
                                     @Nullable SdpDescription value) {
        Log.d(TAG, "unlockPeerOperation: " + type + " " + peer + " " + value);
        synchronized (mOperationLock) {
            boolean isFound = false;

            for (PendingPeerOperation operation : mPeerOperations) {
                Log.d(TAG, "unlockPeerOperation: " + operation + " :: valid value ? " + (null == value));
                if (operation.tryUnlock(type, peer, value)) {
                    isFound = true;
                    mPeerOperations.remove(operation);
                    break;
                }
            }

            if (!isFound) {
                Log.w(TAG, String.format("No pending operation found for peer: %s with type: %s", peer, type.name()));
            }
        }
    }

    private void WaitForSdpMessage(@NonNull String peer,
                                   @NonNull Type type,
                                   @NonNull PendingPeerCallback listener) throws MediaEngineException {
        PendingPeerOperation descriptionOperation = createPeerOperation(type, peer);

        descriptionOperation.setListener(listener);

        descriptionOperation.waitOperation(DescriptionTimeout);
    }

    public static void setContext(@NonNull Context context) {
        MediaEngine.Context = context;
    }

}
