package com.voxeet;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.voxeet.promise.Promise;
import com.voxeet.sdk.authent.token.RefreshTokenCallback;
import com.voxeet.sdk.exceptions.ExceptionManager;
import com.voxeet.sdk.exceptions.VoxeetSentry;
import com.voxeet.sdk.log.endpoints.LogEndpoints;
import com.voxeet.sdk.log.factory.LogFactory;
import com.voxeet.sdk.preferences.VoxeetPreferences;
import com.voxeet.sdk.services.AbstractVoxeetService;
import com.voxeet.sdk.services.AudioService;
import com.voxeet.sdk.services.ChatService;
import com.voxeet.sdk.services.CommandService;
import com.voxeet.sdk.services.ConferenceService;
import com.voxeet.sdk.services.FilePresentationService;
import com.voxeet.sdk.services.LocalStatsService;
import com.voxeet.sdk.services.MediaDeviceService;
import com.voxeet.sdk.services.NotificationService;
import com.voxeet.sdk.services.RecordingService;
import com.voxeet.sdk.services.ScreenShareService;
import com.voxeet.sdk.services.SdkEnvironmentHolder;
import com.voxeet.sdk.services.SessionService;
import com.voxeet.sdk.services.TelemetryService;
import com.voxeet.sdk.services.VideoPresentationService;
import com.voxeet.sdk.services.VoxeetHttp;
import com.voxeet.sdk.utils.AbstractVoxeetEnvironmentHolder;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.BackendAccessHolder;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.VoxeetEnvironmentHolder;
import com.voxeet.sdk.utils.converter.EnumConverterFactory;
import com.voxeet.sdk.utils.converter.JacksonConverterFactory;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.EventBusException;

import java.util.HashMap;
import java.util.Iterator;

import retrofit2.Retrofit;

/**
 * VoxeetSDK is the main object providing methods that interact with the Voxeet service. It is the entry point to call the embedded services and the static elements provided to the developers. The SDK is asynchronous and uses promise at its core.
 * <p>
 * **Typical application workflow:**
 * <p>
 * **1.** The application accesses the Voxeet SDK instance by calling the [instance](/documentation/sdk/reference/android/voxeetsdk#instance) method.
 * <p>
 * **2.** The application initializes the Voxeet SDK through the [initialize](/documentation/sdk/reference/android/voxeetsdk#initialize) method.
 * <p>
 * **3.** The application uses the proper method to retrieve the unique instance for a service that will be used (such as using the [conference](/documentation/sdk/reference/android/voxeetsdk#conference) method to retrieve the [ConferenceService](/documentation/sdk/reference/android/conference) instance).
 * <p>
 * **4.** To use the EventBus, the application calls the [getEventBus](/documentation/sdk/reference/android/voxeetsdk#geteventbus) method to retrieve its unique instance. Then, it calls the [register](/documentation/sdk/reference/android/voxeetsdk#register) method to register the object into the EventBus and the [unregister](/documentation/sdk/reference/android/voxeetsdk#unregister) to unregister from the instance.
 */
@Annotate
public final class VoxeetSDK {

    private final static String TAG = VoxeetSDK.class.getSimpleName();
    private final static long DEFAULT_TIMEOUT_MS = -1;

    private static Context CurrentApplication = null;
    private static VoxeetEnvironmentHolder VoxeetEnvironmentHolder;
    private static VoxeetSDK CurrentInstance;
    private final Retrofit _retrofit;

    private SdkEnvironmentHolder _sdk_environment_holder;
    private VoxeetHttp _voxeet_http;
    private EventBus _event_bus;
    private BackendAccessHolder _backend_access_holder;
    private HashMap<Class<? extends AbstractVoxeetService>, AbstractVoxeetService> _services;
    private VoxeetSentry voxeetSentry;

    @NoDocumentation
    protected static void setInstance(@NonNull VoxeetSDK sdk) {
        CurrentInstance = sdk;

        VoxeetPreferences.init(sdk.getApplicationContext(),
                sdk.getVoxeetEnvironmentHolder());
    }

    /**
     * Accesses the instance of the SDK which enables to use methods specific for the SDK.
     * Services are available using static methods.
     * <p>
     * Note: Every service method can return a null object if the SDK is initialized for the first time.
     *
     * @return The SDK instance if the static initialize method was called.
     */
    @NonNull
    public static VoxeetSDK instance() {
        return CurrentInstance;
    }

    @NoDocumentation
    public static void setApplication(@NonNull Context application) {
        CurrentApplication = application;
        VoxeetEnvironmentHolder = new VoxeetEnvironmentHolder(application);
    }

    /**
     * Initializes the Voxeet SDK. This function determines Voxeet SDK functions. It should be called as soon as possible.
     *
     * @param appId    application ID
     * @param password password
     */
    public static synchronized void initialize(@NonNull String appId,
                                               @NonNull String password) {
        if (null == CurrentInstance) {
            VoxeetSDK sdk = new VoxeetSDK(appId,
                    password,
                    null,
                    null);

            VoxeetSDK.setInstance(sdk);
        } else {
            Log.d(TAG, "initialize: Instance already started !");
        }
    }

    /**
     * Initializes the Voxeet SDK. This function invokes the SDK with a third-party authentication feature. It determines Voxeet SDK functions, so it should be called as soon as possible.
     *
     * @param accessToken  default accessToken
     * @param refreshToken refreshToken callback
     */
    public static synchronized void initialize(@NonNull String accessToken,
                                               @NonNull RefreshTokenCallback refreshToken) {
        if (null == CurrentInstance) {
            VoxeetSDK sdk = new VoxeetSDK(null,
                    null,
                    accessToken,
                    refreshToken);
            VoxeetSDK.setInstance(sdk);
        } else {
            Log.d(TAG, "initialize: Instance already started !");
        }
    }

    private VoxeetSDK(@Nullable String appId,
                      @Nullable String password,
                      @Nullable String tokenAccess,
                      @Nullable RefreshTokenCallback tokenRefresh) {
        initExceptionIfOk(CurrentApplication);

        ObjectMapper mapper = new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        //VisibilityChecker checker = mapper.getVisibilityChecker();
        //mapper.setVisibility(checker.withFieldVisibility(JsonAutoDetect.Visibility.PROTECTED_AND_PUBLIC));

        //set log factory init
        Retrofit retrofit = new Retrofit.Builder()
                //.addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(
                        JacksonConverterFactory.create(mapper)
                )
                .addConverterFactory(new EnumConverterFactory())
                .baseUrl(LogFactory.ENDPOINT) //TODO refacto
                .build();
        _backend_access_holder = new BackendAccessHolder(appId,
                password, tokenAccess, tokenRefresh);

        LogEndpoints endpoint = retrofit.create(LogEndpoints.class);
        LogFactory.instance.init(endpoint);

        _event_bus = EventBus.getDefault();

        //make sure the looper is the main looper for promise execution
        Promise.setHandler(new Handler(Looper.getMainLooper()));

        Log.d("VoxeetSDK", "VoxeetSdk: version := " + VoxeetEnvironmentHolder.getVersionName());

        //the rest needs to be use from the static which will call init();

        _voxeet_http = new VoxeetHttp(this, _backend_access_holder);

        _retrofit = _voxeet_http.getRetrofit();

        _sdk_environment_holder = new SdkEnvironmentHolder(_retrofit,
                _voxeet_http.getClient(),
                this,
                _voxeet_http,
                VoxeetEnvironmentHolder);

        _services = new HashMap<>();

        VoxeetPreferences.init(getApplicationContext(), getVoxeetEnvironmentHolder());

        initServices();
    }

    /**
     * Retrieve the unique instance of the ChatService
     *
     * @return With an initialized SDK, the instance of the ChatService
     */
    @NoDocumentation
    @NonNull
    public static ChatService chat() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(ChatService.class);
    }


    /**
     * Retrieves the unique [ConferenceService](/documentation/sdk/reference/android/conference) instance which enables the participant to interact with conferences.
     *
     * @return With an initialized SDK, the instance of the ConferenceService.
     */
    @NonNull
    public static ConferenceService conference() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(ConferenceService.class);
    }

    /**
     * Retrieves the unique [MediaDeviceService](/documentation/sdk/reference/android/mediadevice) instance which enables the participant to interact with devices through the system.
     *
     * @return With an initialized SDK, the instance of the MediaDeviceService.
     */
    @NonNull
    public static MediaDeviceService mediaDevice() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(MediaDeviceService.class);
    }

    /**
     * Retrieves the unique [ScreenShareService](/documentation/sdk/reference/android/screenshare) instance which enables the participant to use the screen sharing option.
     *
     * @return With an initialized SDK, the instance of the ScreenShareService.
     */
    @NonNull
    public static ScreenShareService screenShare() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(ScreenShareService.class);
    }

    /**
     * Retrieves the unique [SessionService](/documentation/sdk/reference/android/session) instance which enables the participant to use sessions.
     *
     * @return With an initialized SDK, the instance of the SessionService.
     */
    @NonNull
    public static SessionService session() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(SessionService.class);
    }

    /**
     * Retrieves the unique [FilePresentationService](/documentation/sdk/reference/android/filepresentation) instance which enables the participant to use file presentations.
     *
     * @return With an initialized SDK, the instance of the FilePresentationService.
     */
    @NonNull
    public static FilePresentationService filePresentation() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(FilePresentationService.class);
    }

    /**
     * Retrieves the unique [videoPresentationService](/documentation/sdk/reference/android/videopresentation) instance which enables the participant to use video presentations.
     * <p>
     * The UXKit also provides components extending this service with automatic injection of various plugins such as Youtube or MP4.
     *
     * @return With an initialized SDK, the instance of the VideoPresentationService.
     */
    @NonNull
    public static VideoPresentationService videoPresentation() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(VideoPresentationService.class);
    }

    /**
     * Retrieves the unique [AudioService](/documentation/sdk/reference/android/audio) instance which enables the participant to manage the audio connection.
     *
     * @return With an initialized SDK, the instance of the AudioService.
     */
    @NonNull
    public static AudioService audio() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(AudioService.class);
    }

    /**
     * Retrieves the unique [RecordingService](/documentation/sdk/reference/android/recording) instance which enables the participant to use the recording option.
     * <p>
     * This service wraps the recording related calls.
     *
     * @return With an initialized SDK, the instance of the RecordingService.
     */
    @NonNull
    public static RecordingService recording() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(RecordingService.class);
    }

    /**
     * Retrieves the unique [CommandService](/documentation/sdk/reference/android/command) instance which enables the participant to send messages in a string formatted shape into specified conferences.
     *
     * @return With an initialized SDK, the instance of the CommandService.
     */
    @NonNull
    public static CommandService command() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(CommandService.class);
    }

    /**
     * Retrieves the unique instance of the LocalStatsService which enables the participant to call to the statistics inside conferences.
     *
     * @return With an initialized SDK, the instance of the LocalStatsService.
     */
    @NoDocumentation
    @NonNull
    public static LocalStatsService localStats() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(LocalStatsService.class);
    }

    /**
     * Retrieves the unique [NotificationService](/documentation/sdk/reference/android/notification) instance which enables forwarding notifications from developer to the properly registered manager.
     *
     * @return With an initialized SDK, the instance of the NotificationService.
     */
    @NonNull
    public static NotificationService notification() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(NotificationService.class);
    }

    @NoDocumentation
    @NonNull
    public static TelemetryService telemetry() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(TelemetryService.class);
    }

    @NoDocumentation
    public Context getApplicationContext() {
        return CurrentApplication;
    }

    @NoDocumentation
    public AbstractVoxeetEnvironmentHolder getVoxeetEnvironmentHolder() {
        return VoxeetEnvironmentHolder;
    }

    /**
     * Retrieves the instance of EventBus used by the SDK.
     *
     * @return The instance of the EventBus used internally.
     */
    @NonNull
    public EventBus getEventBus() {
        return _event_bus;
    }

    /**
     * Registers object into the EventBus of the SDK. Call this method before creating or joining any conference to trigger events.
     *
     * @param subscriber subscriber
     */
    public boolean register(@Nullable Object subscriber) {
        try {
            //MediaSDK.setContext(context);
            EventBus eventBus = getEventBus();

            if (null != subscriber && !eventBus.isRegistered(subscriber))
                eventBus.register(subscriber);
        } catch (EventBusException error) {
            error.printStackTrace();
            return false;
        }

        return true;
    }

    /**
     * Unregisters from the EventBus instance. Call this method to remove your Activity or Fragment.
     *
     * @param subscriber subscriber
     */
    public void unregister(@NonNull Object subscriber) {
        try {
            if (getEventBus().isRegistered(subscriber))
                getEventBus().unregister(subscriber);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private HashMap<Class<? extends AbstractVoxeetService>, AbstractVoxeetService> getServices() {
        return _services;
    }

    private void initExceptionIfOk(@NonNull Context application_context) {
        Log.d(TAG, "initExceptionIfOk: checking for exception to log internally");
        voxeetSentry = new VoxeetSentry(application_context);
        ExceptionManager.register(voxeetSentry);
        Log.d(TAG, "initExceptionIfOk: finished to try implementing error management");
    }

    private <T extends AbstractVoxeetService> T getServiceForKlass(Class<T> klass) {
        T service = null;
        Iterator<Class<? extends AbstractVoxeetService>> iterator = _services.keySet().iterator();
        while (iterator.hasNext()) {
            Class<? extends AbstractVoxeetService> next = iterator.next();
            AbstractVoxeetService item = _services.get(next);

            if (klass.isInstance(item)) {
                service = (T) item;
            }
        }

        if (service == null) {
            Log.d(TAG, klass.getSimpleName() + " not found in the list of services");
        }
        return service;
    }

    /**
     * Change the region used by the VoxeetSdk. This method must be used outside of any connected sessions.
     *
     * @param region
     */
    public static void region(@NonNull String region) {
        VoxeetEnvironmentHolder.setRegion(region);
    }

    private void initServices() {
        getServices().put(AudioService.class, new AudioService(_sdk_environment_holder));
        getServices().put(ConferenceService.class, new ConferenceService(_sdk_environment_holder, DEFAULT_TIMEOUT_MS));
        getServices().put(SessionService.class, new SessionService(_sdk_environment_holder));
        getServices().put(FilePresentationService.class, new FilePresentationService(_sdk_environment_holder));
        getServices().put(VideoPresentationService.class, new VideoPresentationService(_sdk_environment_holder));
        getServices().put(ScreenShareService.class, new ScreenShareService(_sdk_environment_holder));
        getServices().put(MediaDeviceService.class, new MediaDeviceService(_sdk_environment_holder));
        getServices().put(LocalStatsService.class, new LocalStatsService(_sdk_environment_holder));
        getServices().put(TelemetryService.class, new TelemetryService(_sdk_environment_holder));
        getServices().put(ChatService.class, new ChatService(_sdk_environment_holder));
        getServices().put(CommandService.class, new CommandService(_sdk_environment_holder));
        getServices().put(RecordingService.class, new RecordingService(_sdk_environment_holder));
        getServices().put(NotificationService.class, new NotificationService(_sdk_environment_holder));
    }
}
