package com.voxeet.sdk.services.conference.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.VoxeetSDK;
import com.voxeet.promise.Promise;
import com.voxeet.sdk.json.SdkConferenceInvitation;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.models.v1.ConferenceParticipantStatus;
import com.voxeet.sdk.models.v2.ServerErrorOrigin;
import com.voxeet.sdk.network.endpoints.IRestApiConferenceAccess;
import com.voxeet.sdk.services.AbstractPromiseable;
import com.voxeet.sdk.services.ConferenceService;
import com.voxeet.sdk.services.MediaDeviceService;
import com.voxeet.sdk.services.SessionService;
import com.voxeet.sdk.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.utils.HttpHelper;
import com.voxeet.sdk.utils.Opt;

import org.greenrobot.eventbus.EventBus;

import java.util.ArrayList;
import java.util.List;

import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Response;

public class InvitePromise extends AbstractPromiseable<List<Participant>, IRestApiConferenceAccess> {

    private final String TAG = InvitePromise.class.getSimpleName();

    private final List<String> ids;
    private final String conferenceId;

    public InvitePromise(@NonNull ConferenceService parent,
                         @NonNull MediaDeviceService mediaDeviceService,
                         @NonNull IRestApiConferenceAccess provider,
                         @Nullable ConferenceInformation information,
                         @NonNull EventBus eventBus,
                         @NonNull String conferenceId,
                         @NonNull List<String> ids) {
        super(parent, mediaDeviceService, provider, information, eventBus);

        this.ids = ids;
        this.conferenceId = conferenceId;
    }

    @NonNull
    @Override
    public Promise<List<Participant>> createPromise() {
        return new Promise<>(solver -> {
            //remove the timeout
            removeTimeoutCallbacks();

            //a new one is now sent
            sendTimeoutCallbacks();

            boolean sdk = isSDK();
            List<String> voxeetIds = sdk ? null : new ArrayList<>();

            final List<String> externalIds = new ArrayList<>(ids);
            if (!sdk) voxeetIds.addAll(ids);

            //send an error if you're not in a conference
            if (null == conferenceId) {
                Promise.reject(new IllegalStateException("You're not in a conference"));
                return;
            }

            final Call<ResponseBody> user = getApiRestCall().invite(conferenceId, new SdkConferenceInvitation(voxeetIds, externalIds));
            HttpHelper.enqueue(user, (response, object, exception) -> {
                if (null != exception) {
                    solver.reject(HttpHelper.manageThrowableOrServerError(exception, ServerErrorOrigin.INVITE));
                } else {
                    List<Participant> list = new ArrayList<>();
                    String localId = Opt.of(VoxeetSDK.session()).then(SessionService::getParticipantId).or("");

                    if (Opt.of(response).then(Response::code).or(0) == 200) {
                        for (String userId : externalIds) {
                            if (!userId.equals(localId)) {
                                list.add(updateConferenceParticipants(userId, ConferenceParticipantStatus.IN_PROGRESS));
                            }
                        }
                    }
                    solver.resolve(list);
                }
            });
        });
    }
}
