package com.voxeet.sdk.services.conference.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.VoxeetSDK;
import com.voxeet.promise.Promise;
import com.voxeet.promise.solve.Solver;
import com.voxeet.promise.solve.ThenPromise;
import com.voxeet.promise.solve.ThenVoid;
import com.voxeet.sdk.events.sdk.ConferenceStatusUpdatedEvent;
import com.voxeet.sdk.json.CreateConferenceParams;
import com.voxeet.sdk.json.internal.MetadataHolder;
import com.voxeet.sdk.json.internal.ParamsHolder;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.v1.CreateConferenceResult;
import com.voxeet.sdk.models.v2.ServerErrorOrigin;
import com.voxeet.sdk.network.endpoints.IRestApiConferenceAccess;
import com.voxeet.sdk.services.AbstractPromiseable;
import com.voxeet.sdk.services.ConferenceService;
import com.voxeet.sdk.services.MediaDeviceService;
import com.voxeet.sdk.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.services.conference.information.ConferenceStatus;
import com.voxeet.sdk.services.conference.information.LocalConferenceType;
import com.voxeet.sdk.utils.HttpHelper;

import org.greenrobot.eventbus.EventBus;

import retrofit2.Call;

public class CreateConferencePromiseable extends AbstractPromiseable<CreateConferenceResult, IRestApiConferenceAccess> {

    private String conferenceAlias;
    private MetadataHolder metadata;
    private ParamsHolder params = null;

    public CreateConferencePromiseable(@NonNull ConferenceService parent,
                                       @NonNull MediaDeviceService mediaDeviceService,
                                       @NonNull IRestApiConferenceAccess provider,
                                       @Nullable ConferenceInformation information,
                                       @NonNull EventBus eventBus,
                                       @Nullable String conferenceAlias,
                                       @Nullable MetadataHolder metadata,
                                       @Nullable ParamsHolder params) {
        super(parent, mediaDeviceService, provider, information, eventBus);

        this.conferenceAlias = conferenceAlias;
        this.metadata = metadata;
        this.params = params;
    }

    @NonNull
    @Override
    public Promise<CreateConferenceResult> createPromise() {
        final Promise<CreateConferenceResult> promise = new Promise<>(solver -> {
            getEventBus().post(new ConferenceStatusUpdatedEvent(conferenceAlias, ConferenceStatus.CREATING));
            log("Attempting to create mConferene alias:=" + conferenceAlias);

            CreateConferenceParams arg = new CreateConferenceParams()
                    .setMetadataHolder(metadata)
                    .setParamsHolder(params);
            //        .setStats(mEnableStats);

            if (null != conferenceAlias) arg.setConferenceAlias(conferenceAlias);

            //now create the conference and will retry in case of issue...
            internalCreate(arg, solver);
        });

        if (VoxeetSDK.session().isSocketOpen()) {
            return promise;
        } else {

            return new Promise<>(solver -> VoxeetSDK.session().open()
                    .then((ThenPromise<Boolean, CreateConferenceResult>) result -> promise)
                    .then((ThenVoid<CreateConferenceResult>) solver::resolve)
                    .error(solver::reject));
        }
    }


    /**
     * Create a conference and retry if neede
     *
     * @param params the params to send
     * @param solver the solver to resolve
     */
    private void internalCreate(final CreateConferenceParams params, final Solver<CreateConferenceResult> solver) {
        Call<CreateConferenceResult> observable = getApiRestCall().create(params);
        HttpHelper.enqueue(observable, (response, object, e) -> {
            if (null != e) {
                e.printStackTrace();

                ConferenceStatusUpdatedEvent state = new ConferenceStatusUpdatedEvent(params.conferenceAlias, ConferenceStatus.ERROR);
                setIsInConference(false);
                closeMedia();
                getEventBus().post(state);

                log("Error while create conference : creation process part");
                solver.reject(HttpHelper.manageThrowableOrServerError(e, ServerErrorOrigin.CREATE));
            } else {
                ConferenceInformation information = createOrSetConferenceWithParams(object.conferenceId, object.conferenceAlias);

                information.setConferenceState(ConferenceStatus.CREATED);
                information.setConferenceType(LocalConferenceType.CONFERENCE);

                Conference conference = information.getConference();
                getEventBus().post(new ConferenceStatusUpdatedEvent(conference, conference.getState()));

                log("internalCreate onNext: join with := " + object.conferenceId + " " + object.conferenceAlias);
                //TODO check for issue in here but for now, setting back id...
                information.getConference().setConferenceId(object.conferenceId);
                log("now conference is " + information.getConference().getId());

                solver.resolve(object);
            }
        });
    }
}
