package com.voxeet.sdk.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.voxeet.VoxeetSDK;
import com.voxeet.promise.Promise;
import com.voxeet.sdk.authent.KeySecretThroughOAuthTokenResponseProvider;
import com.voxeet.sdk.authent.OAuthTokenResponseProvider;
import com.voxeet.sdk.authent.endpoints.IVoxeetRService;
import com.voxeet.sdk.authent.exceptions.AuthentExceptionListener;
import com.voxeet.sdk.authent.models.DeviceType;
import com.voxeet.sdk.authent.models.UserInfoBody;
import com.voxeet.sdk.authent.token.RefreshTokenCallback;
import com.voxeet.sdk.authent.token.TokenResponseProvider;
import com.voxeet.sdk.events.sdk.SocketStateChangeEvent;
import com.voxeet.sdk.json.ParticipantInfo;
import com.voxeet.sdk.log.factory.LogFactory;
import com.voxeet.sdk.network.endpoints.IRestApiTelemetry;
import com.voxeet.sdk.preferences.VoxeetPreferences;
import com.voxeet.sdk.services.authenticate.VoxeetCookieJar;
import com.voxeet.sdk.services.authenticate.WebSocketState;
import com.voxeet.sdk.services.notification.INotificationTokenProvider;
import com.voxeet.sdk.services.notification.NotificationTokenHolderFactory;
import com.voxeet.sdk.utils.AbstractVoxeetEnvironmentHolder;
import com.voxeet.sdk.utils.BackendAccessHolder;
import com.voxeet.sdk.utils.Callback;
import com.voxeet.sdk.utils.converter.JacksonConverterFactory;

import org.greenrobot.eventbus.EventBus;

import java.util.HashMap;
import java.util.List;

import javax.net.ssl.HostnameVerifier;

import okhttp3.Cookie;
import okhttp3.OkHttpClient;
import retrofit2.Retrofit;

/**
 * VoxeetHttp manages every network call made by the SDK
 * <p>
 * It will identify, re-identify and make the calls
 */
public class VoxeetHttp implements AuthentExceptionListener {

    private static final String TAG = VoxeetHttp.class.getSimpleName();
    private final AbstractVoxeetEnvironmentHolder environment_holder;
    private final BackendAccessHolder holder;

    private Callback<AbstractVoxeetEnvironmentHolder> onRegion = this::onRegion;

    private void onRegion(AbstractVoxeetEnvironmentHolder abstractVoxeetEnvironmentHolder) {
        Log.d(TAG, "onRegion: check that no user are currently logged in!");
        init();
    }


    private HashMap<Class, Object> services = new HashMap<>();
    /**
     * The Retrofit.
     */
    private Retrofit retrofit;
    private Retrofit retrofitTelemetry;

    private OkHttpClient client;

    private TokenResponseProvider tokenResponseProvider;

    private boolean debug = false;
    private OkHttpClient clientIdentify;
    private OkHttpClient clientSubIdentify;
    private Retrofit retrofitIdentify;
    private Retrofit retrofitSubIdentify;
    private VoxeetCookieJar cookieJar;

    /**
     * Init the Voxeet Http wrapper
     *
     * @param sdk
     * @param holder
     */
    public VoxeetHttp(VoxeetSDK sdk, BackendAccessHolder holder) {

        this.holder = holder;

        environment_holder = sdk.getVoxeetEnvironmentHolder();
        environment_holder.register(onRegion);
        init();
    }

    private void init() {
        initAuthenticators();
        initClient();
        initRetrofit();
        initService();
    }


    private void initAuthenticators() {
        String appId = holder.appId;
        String password = holder.password;
        String tokenAccess = holder.tokenAccess;
        RefreshTokenCallback tokenRefresh = holder.tokenRefresh;

        //set 1 provider according to the 2 different cases which be made available
        if (null != appId && null != password) {
            Log.d(TAG, "VoxeetHttp: initializing using appId/password");
            tokenResponseProvider = new KeySecretThroughOAuthTokenResponseProvider(appId, password, environment_holder.getVersionName(), environment_holder.getServerUrl(), this);
        } else if (null != tokenRefresh) {
            Log.d(TAG, "VoxeetHttp: initializing using access/refresh");

            //in such case, the participant's servers are here to manage oauth
            tokenResponseProvider = new OAuthTokenResponseProvider(tokenAccess, tokenRefresh, environment_holder.getVersionName(), this);
        }
    }

    /**
     * Overrode retrofit client.
     */
    private void initClient() {
        this.cookieJar = new VoxeetCookieJar(getBuiltServerUrl());

        OkHttpClient.Builder builderIdentify = new OkHttpClient.Builder()
                .cookieJar(cookieJar)
                .addNetworkInterceptor(chain -> {
                    Log.d(TAG, "intercept: builderIdentify");
                    return tokenResponseProvider.executeIdentify(chain, "builderIdentify");
                });

        OkHttpClient.Builder builderSubIdentify = new OkHttpClient.Builder()
                .cookieJar(cookieJar)
                .addNetworkInterceptor(chain -> {
                    Log.d(TAG, "intercept: builderSubIdentify");
                    return tokenResponseProvider.executeIdentify(chain, "builderSubIdentify");
                });

        OkHttpClient.Builder builder = new OkHttpClient.Builder()
                .cookieJar(cookieJar)
                .addNetworkInterceptor(chain -> {
                    Log.d(TAG, "intercept: builder");
                    return tokenResponseProvider.execute(chain);
                });

        tokenResponseProvider.configureOkHttpClientBuilder(builder, true);
        tokenResponseProvider.configureOkHttpClientBuilder(builderIdentify, true);
        tokenResponseProvider.configureOkHttpClientBuilder(builderSubIdentify, false);

        if (this.debug) {
            HostnameVerifier hostnameVerifier = (hostname, session) -> true;

            builder = builder.hostnameVerifier(hostnameVerifier);
            builderIdentify = builderIdentify.hostnameVerifier(hostnameVerifier);
            builderSubIdentify = builderSubIdentify.hostnameVerifier(hostnameVerifier);
        }

        client = builder.build();
        clientIdentify = builderIdentify.build();
        clientSubIdentify = builderSubIdentify.build();
    }

    private void initRetrofit() {
        retrofit = new Retrofit.Builder()
                //.addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create(new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)))
                .baseUrl(getBuiltServerUrl())
                .client(client)
                .build();
        retrofitTelemetry = new Retrofit.Builder()
                //.addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create(new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)))
                .baseUrl(getBuiltTelemetryServerUrl())
                .client(client)
                .build();

        //TODO put the services creation ahead of time in the VoxeeHttp instance
        IRestApiTelemetry object = retrofitTelemetry.create(IRestApiTelemetry.class);
        services.put(IRestApiTelemetry.class, object);


        retrofitIdentify = new Retrofit.Builder()
                //.addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create(new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)))
                .baseUrl(getBuiltServerUrl())
                .client(clientIdentify)
                .build();

        retrofitSubIdentify = new Retrofit.Builder()
                //.addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create(new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)))
                .baseUrl(getBuiltServerUrl())
                .client(clientSubIdentify)
                .build();
    }

    private void initService() {
        tokenResponseProvider.initService(retrofitIdentify.create(IVoxeetRService.class),
                retrofitSubIdentify.create(IVoxeetRService.class));
    }

    /**
     * Identify. Similar to a login but for the REST call only
     *
     * @param participantInfo the participant info
     */
    public Promise<Boolean> identifyChain(final ParticipantInfo participantInfo) {
        return new Promise<>(final_solver -> {
            LogFactory.instance.connect();
            UserInfoBody userInfoBody = new UserInfoBody(participantInfo.getName(), participantInfo.getExternalId(), participantInfo.getAvatarUrl());
            tokenResponseProvider.setUserInfo(userInfoBody);

            VoxeetPreferences.setExternalUserInfo(participantInfo);

            userInfoBody.deviceIdentifier = VoxeetPreferences.deviceIdentifier();
            userInfoBody.deviceType = DeviceType.ANDROID;


            INotificationTokenProvider provider = NotificationTokenHolderFactory.provider;
            if (null != provider && provider.isTokenUploadAllowed()) {
                String token = provider.getToken();
                if (token != null) {
                    Log.d(TAG, "onCall: token obtained");
                    userInfoBody.devicePushToken = token;
                }
            }

            tokenResponseProvider.retrieve().then((result) -> {
                if (null != result) {
                    Log.d(TAG, "Successful login with id " + result.getId());

                    VoxeetPreferences.setId(result.getId());
                    VoxeetPreferences.saveLoginCookie(result.getUserToken());

                    List<Cookie> cookies = cookieJar.getCookies(getBuiltServerUrl());
                    if (cookies != null && cookies.size() > 0)
                        VoxeetPreferences.saveLoginCookie(cookies.get(0).toString());
                    final_solver.resolve(true);

                    VoxeetSDK.session().connectSocket();
                } else {
                    EventBus.getDefault().post(new SocketStateChangeEvent(WebSocketState.CLOSED));
                    final_solver.resolve(false);
                }
            }).error(error -> {
                error.printStackTrace();
                //previously made here
                EventBus.getDefault().post(new SocketStateChangeEvent(WebSocketState.CLOSED));
                final_solver.reject(error);
            });
        });
    }

    public Promise<Boolean> logout(@NonNull String id) {
        return new Promise<>(
                solver -> tokenResponseProvider.logout().then((result) -> {
                    cleanParticipantSession(id);
                    solver.resolve(result);
                }).error(solver::reject)
        );
    }

    /**
     * Gets client.
     *
     * @return the client
     */

    public OkHttpClient getClient() {
        return client;
    }

    public Retrofit getRetrofit() {
        return retrofit;
    }

    @Nullable
    String getToken() {
        return tokenResponseProvider.getToken();
    }

    @Nullable
    public String getJwtToken() {
        return tokenResponseProvider.getJwtToken();
    }

    public void resetVoxeetHttp() {
        tokenResponseProvider.resetVoxeetHttp();
    }

    public void cleanParticipantSession(@NonNull String id) {
        tokenResponseProvider.cleanUserSession(id);
    }

    String getBuiltServerUrl() {
        String port = environment_holder.getServerPort();
        if (!port.startsWith(":")) port = ":" + port;
        return environment_holder.getServerUrl() + port;
    }

    String getBuiltTelemetryServerUrl() {
        return environment_holder.getTelemetryServerUrl();
    }

    //TODO build services ahead of time
    @Nullable
    public <RETROFIT_SERVICE_KLASS> RETROFIT_SERVICE_KLASS get(@Nullable Class<RETROFIT_SERVICE_KLASS> service) {
        if (null == service) return null;

        if (services.containsKey(service)) {
            return (RETROFIT_SERVICE_KLASS) services.get(service);
        }
        Object object = retrofit.create(service);
        services.put(service, object);
        return (RETROFIT_SERVICE_KLASS) object;
    }

    @Override
    public void onException(@NonNull Throwable throwable) {
        //previously sent to exception manager
        throwable.printStackTrace();
    }
}