package com.voxeet.sdk.services;

import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.VoxeetSDK;
import com.voxeet.audio2.devices.MediaDevice;
import com.voxeet.promise.Promise;
import com.voxeet.promise.solve.ThenPromise;
import com.voxeet.promise.solve.ThenVoid;
import com.voxeet.sdk.BuildConfig;
import com.voxeet.sdk.events.sdk.ConferenceStatusUpdatedEvent;
import com.voxeet.sdk.events.sdk.SocketStateChangeEvent;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.network.endpoints.IRestApiTelemetry;
import com.voxeet.sdk.services.telemetry.SdkEnvironment;
import com.voxeet.sdk.services.telemetry.TelemetryAnswer;
import com.voxeet.sdk.services.telemetry.TelemetryState;
import com.voxeet.sdk.services.telemetry.TelemetryStateUpdated;
import com.voxeet.sdk.services.telemetry.TelemetryTickState;
import com.voxeet.sdk.services.telemetry.TelemetryTickUpdated;
import com.voxeet.sdk.services.telemetry.device.DeviceInformation;
import com.voxeet.sdk.services.telemetry.device.DeviceStats;
import com.voxeet.sdk.services.telemetry.device.HardwareInfo;
import com.voxeet.sdk.services.telemetry.network.NetworkInformationProvider;
import com.voxeet.sdk.services.telemetry.promises.TelemetryConfigurationPromise;
import com.voxeet.sdk.services.telemetry.promises.WebRTCDeviceInfoPromise;
import com.voxeet.sdk.services.telemetry.promises.WebRTCStatsBatchPromise;
import com.voxeet.sdk.services.telemetry.promises.WebRTCStatsPromise;
import com.voxeet.sdk.services.telemetry.rest.TelemetryConfiguration;
import com.voxeet.sdk.telemetry.MetricsHolder;
import com.voxeet.sdk.telemetry.WebRTCStats;
import com.voxeet.sdk.telemetry.utils.TeleLog;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.Map;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.Opt;
import com.voxeet.stats.LocalStats;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;

/**
 * Telemetry service used by the Voxeet SDK. Provides information about the environment.
 */
@Annotate
public class TelemetryService extends AbstractVoxeetService<IRestApiTelemetry> {
    private final static String TAG = TelemetryService.class.getSimpleName();
    private static ConcurrentHashMap<SdkEnvironment, String> environment = new ConcurrentHashMap<>();
    private final NetworkInformationProvider networkProvider;
    private WaitForConference waitForConference;
    private Handler _runnable;

    private int _next_publish_in;

    private TelemetryState state = TelemetryState.STOPPED;
    private TelemetryTickState tick_state = TelemetryTickState.AGGREGATE;
    private List<WebRTCStats> temp_stats = new ArrayList<>();

    @Nullable
    private MetricsHolder previous_metrics = null;

    private Runnable _tick = () -> {
        String conferenceId = Opt.of(VoxeetSDK.conference()).then(ConferenceService::getConferenceId).or("");
        String userId = Opt.of(VoxeetSDK.session()).then(SessionService::getParticipantId).orNull();

        _next_publish_in--;
        List<Participant> participants = VoxeetSDK.conference().getParticipants();
        LocalStatsService service = VoxeetSDK.localStats();
        List<LocalStats> localStats = Map.map(participants, in -> service.getLocalStats(in.getId()));

        TeleLog.log = true;

        TeleLog.d("TICK", "TICK: >>>>>>>>>>>>>>>>>>>>>>>>>>>>>");
        TeleLog.d("TICK", "has previous := " + (null != previous_metrics));

        for (LocalStats stat : localStats) {
            TeleLog.d("TICK", "json := " + stat.userId + " " + stat.getRawJson());
        }

        WebRTCStats stats = new WebRTCStats(conferenceId, userId, localStats, previous_metrics, System.currentTimeMillis());
        TeleLog.d("TICK", "json := " + stats.toJson().toString());
        TeleLog.d("TICK", "TICK: <<<<<<<<<<<<<<<<<<<<<<<<<<<<<");
        temp_stats.add(stats);

        //now set the previous metrics holder...
        previous_metrics = stats.metricsHolder;

        if (_next_publish_in <= 0) {
            tick_state = TelemetryTickState.POST;
            _next_publish_in = batch();

            flush(conferenceId);
        } else {
            tick_state = TelemetryTickState.AGGREGATE;
        }

        getEventBus().post(new TelemetryTickUpdated(tick_state));

        repost();
    };

    @Nullable
    private TelemetryConfiguration configuration;
    private boolean device_information_sent;

    private Runnable _ttl = () -> {
        if (null != configuration && configuration.needRefresh()) {
            configuration().then(this::setConfigurationAfterRetrieval).error(Throwable::printStackTrace);
        }
    };

    @NoDocumentation
    public TelemetryService(SdkEnvironmentHolder sdk_environment_holder) {
        super(sdk_environment_holder, IRestApiTelemetry.class);
        waitForConference = new WaitForConference();

        getEventBus().register(waitForConference);
        networkProvider = new NetworkInformationProvider(sdk_environment_holder.voxeetSdk.getApplicationContext());
        registerEnvironment(SdkEnvironment.SDK, BuildConfig.VERSION_NAME);
        registerEnvironment(SdkEnvironment.MEDIA, com.voxeet.android.media.BuildConfig.VERSION_NAME);
    }

    /**
     * Gets the device information.
     *
     * @return the promise to resolve.
     */
    public Promise<DeviceInformation> deviceInformation() {
        return new Promise<>(solver -> {
            HardwareInfo info = new HardwareInfo(context);
            VoxeetSDK.audio().enumerateDevices().then((ThenVoid<List<MediaDevice>>) devices -> solver.resolve(new DeviceInformation(info, true, environment,
                    devices,
                    networkProvider.carriers(),
                    networkProvider.wifiInfo())))
                    .error(solver::reject);
        });
    }

    private void trySendDeviceInformation() {
        if (!device_information_sent) {
            device_information_sent = true;

            deviceInformation().then((ThenPromise<DeviceInformation, TelemetryAnswer<Boolean>>) deviceInformation -> {
                DeviceStats stats = new DeviceStats(VoxeetSDK.session().getParticipantId(),
                        deviceInformation,
                        System.currentTimeMillis());
                return new WebRTCDeviceInfoPromise(VoxeetSDK.conference(),
                        VoxeetSDK.mediaDevice(),
                        getService(),
                        null,
                        getEventBus(),
                        stats).createPromise();
            }).then(booleanTelemetryAnswer -> {
                //device information properly sent
                Log.d(TAG, "call: device information sent");
            }).error(error -> {
                device_information_sent = false;
                error.printStackTrace();
            });
        }
    }


    @NonNull
    private Promise<TelemetryConfiguration> configuration() {
        if (null != configuration && !configuration.needRefresh()) {
            return Promise.resolve(configuration);
        } else {
            return new TelemetryConfigurationPromise(VoxeetSDK.conference(),
                    VoxeetSDK.mediaDevice(),
                    getService(),
                    null,
                    getEventBus()).createPromise();
        }
    }

    @NonNull
    private Promise<TelemetryAnswer<Boolean>> upload(@NonNull String conferenceId, WebRTCStats stats) {
        //String conferenceId = Opt.of(VoxeetSdk.conference().getConferenceId()).or("");
        return new WebRTCStatsPromise(VoxeetSDK.conference(),
                VoxeetSDK.mediaDevice(),
                getService(),
                VoxeetSDK.conference().getConferenceInformation(conferenceId),
                getEventBus(),
                stats).createPromise();
    }

    @NonNull
    private Promise<TelemetryAnswer<Boolean>> upload(@NonNull String conferenceId, List<WebRTCStats> stats) {
        //String conferenceId = Opt.of(VoxeetSdk.conference().getConferenceId()).or("");
        return new WebRTCStatsBatchPromise(VoxeetSDK.conference(),
                VoxeetSDK.mediaDevice(),
                getService(),
                VoxeetSDK.conference().getConferenceInformation(conferenceId),
                getEventBus(),
                stats).createPromise();
    }

    @NoDocumentation
    @NonNull
    public TelemetryState getState() {
        return state;
    }

    @NoDocumentation
    @NonNull
    public TelemetryTickState getTickState() {
        return tick_state;
    }

    private int batch() {
        return Opt.of(configuration).then(TelemetryConfiguration::batch).or(3);
    }

    private int samplingFreq() {
        return Opt.of(configuration).then(TelemetryConfiguration::samplingFreqSec).or(10) * 1000;
    }

    private void flush(@NonNull String conferenceId) {
        if (temp_stats.size() > 0) {
            upload(conferenceId, temp_stats).then(result -> {
                Log.d(TAG, "uploaded stats " + result);
            }).error(Throwable::printStackTrace);

            //now resetting
            temp_stats = new ArrayList<>();
        }
    }

    private synchronized void start() {
        //try
        trySendDeviceInformation();

        if (null == _runnable) {
            device_information_sent = false;

            String conferenceId = Opt.of(VoxeetSDK.conference().getConferenceId()).orNull();
            setState(TelemetryState.STARTING);

            Handler handler = new Handler(Looper.getMainLooper());
            _runnable = handler;

            configuration().then(configuration -> {
                if (null == configuration) throw new IllegalStateException("Couldn't load");
                if (!configuration.enabled()) throw new IllegalStateException("Stopped telemetry");


                setConfigurationAfterRetrieval(configuration);

                Log.d(TAG, "start: configuration := " + configuration);
                trySendDeviceInformation();

                _next_publish_in = batch();
                setState(TelemetryState.STARTED);

                handler.removeCallbacks(_tick);
                handler.postDelayed(_tick, samplingFreq());
            }).error(error -> stop(conferenceId));
        }
    }

    private void setConfigurationAfterRetrieval(@Nullable TelemetryConfiguration configuration) {
        if (null != configuration) {
            this.configuration = configuration; //reset the configuration

            this.postForConfigurationInvalidation();
        }
    }

    private void postForConfigurationInvalidation() {
        handler.removeCallbacks(_ttl);
        if (null != configuration) {
            handler.postDelayed(_ttl, (configuration.ttl + 10) * 1000); //post at ttl + 10s
        }
    }

    private synchronized void stop(@Nullable String conferenceId) {
        if (null != _runnable) {
            previous_metrics = null;
            device_information_sent = false;
            setState(TelemetryState.STOPPED);
            _runnable.removeCallbacks(_tick);
            _runnable = null;

            if (null != conferenceId) {
                flush(conferenceId);
            }
        }
    }

    private void repost() {
        if (null != configuration) {
            _runnable.postDelayed(_tick, samplingFreq());
        }
    }

    private void setState(@NonNull TelemetryState state) {
        this.state = state;
        getEventBus().post(new TelemetryStateUpdated(state));
    }

    private class WaitForConference {
        public WaitForConference() {

        }

        @Subscribe(threadMode = ThreadMode.MAIN)
        public void onEvent(@NonNull SocketStateChangeEvent event) {
            switch (event.state) {
                case CONNECTED:
                    configuration().then(configuration -> {
                        if (null != configuration)
                            TelemetryService.this.configuration = configuration;
                    }).error(Throwable::printStackTrace);
                    break;
                default:
            }
        }

        @Subscribe(threadMode = ThreadMode.MAIN)
        public void onEvent(@NonNull ConferenceStatusUpdatedEvent event) {
            switch (event.state) {
                case CREATING:
                case CREATED:
                    //nothing to do
                    break;
                case JOINING:
                case JOINED:
                case FIRST_PARTICIPANT:
                case NO_MORE_PARTICIPANT:
                    start();
                    break;
                default:
                    stop(Opt.of(event.conference).then(Conference::getId).orNull());
            }
        }
    }

    @NoDocumentation
    public void registerEnvironment(@Nullable SdkEnvironment name, @Nullable String value) {
        if (null != name && null != value && !environment.contains(name)) {
            environment.put(name, value);
        }
    }
}
