package com.voxeet.sdk.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.VoxeetSDK;
import com.voxeet.promise.Promise;
import com.voxeet.promise.solve.Solver;
import com.voxeet.sdk.events.error.SdkLogoutErrorEvent;
import com.voxeet.sdk.events.error.SocketConnectErrorEvent;
import com.voxeet.sdk.events.sdk.SdkLogoutSuccessEvent;
import com.voxeet.sdk.events.sdk.SocketStateChangeEvent;
import com.voxeet.sdk.json.ParticipantInfo;
import com.voxeet.sdk.log.factory.LogFactory;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.network.websocket.VoxeetWebSocket;
import com.voxeet.sdk.preferences.VoxeetPreferences;
import com.voxeet.sdk.services.authenticate.WebSocketState;
import com.voxeet.sdk.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;

/**
 * The SessionService allows an application to register its participants' information in the Voxeet service. It is mandatory for the application to open a session before it can interact with the service further. During the life cycle of the application, it may open and close sessions multiple times.
 * <p>
 * **Typical application workflow:**
 * <p>
 * **1.** The application opens a session using the [open](/documentation/sdk/reference/android/session#open) method.
 * <p>
 * **2.** The application can check the [SocketStateChangeEvent](/documentation/sdk/reference/android/session#socketstatechangeevent) informing if the WebSocket changed state. To check the state outside of the event, it can use the [isSocketOpen](/documentation/sdk/reference/android/session#issocketopen) method.
 * <p>
 * **3.** The application closes the session using the [close](/documentation/sdk/reference/android/session#close) method.
 */
@Annotate
public class SessionService extends AbstractVoxeetService {

    private static final String TAG = SessionService.class.getSimpleName();
    private VoxeetWebSocket _voxeet_websocket;
    private List<Solver<Boolean>> mWaitForLogInSocket;
    private ReentrantLock lockConnectAttempt = new ReentrantLock();

    @Nullable
    private ParticipantInfo _participant_infos;

    /**
     * Instantiates a new Participant service.
     *
     * @param instance the parent instance
     */
    @NoDocumentation
    public SessionService(@NonNull SdkEnvironmentHolder instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        mWaitForLogInSocket = new ArrayList<>();
        _voxeet_websocket = new VoxeetWebSocket(instance.voxeetSdk,
                instance.voxeetHttp);

        registerEventBus();
    }

    /**
     * Connects the WebSocket for the current session. Does not check for any issue with participant logged in or not.
     */
    @NoDocumentation
    public void connectSocket() {
        _voxeet_websocket.connect();
    }

    /**
     * Opens a session with the specified ParticipantInfo information.
     *
     * @param participantInfo ParticipantInfo, it should contain at least one ParticipantName
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Boolean> open(@NonNull ParticipantInfo participantInfo) {
        _participant_infos = participantInfo;
        return open();
    }

    /**
     * Logs in using information from the previously unclosed session opened earlier. It can also create a temporary “ghost” session.
     *
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Boolean> open() {
        if (null == _participant_infos) rejectParticipantNull("login");

        if (isSocketOpen()) {
            return new Promise<>(solver -> {
                Log.d(TAG, "onCall: socket opened");
                lockConnectAttempt();
                resolveLoginSockets();
                unlockConnectAttempt();

                solver.resolve(true);
            });
        }

        return new Promise<>(solver -> {
            if (null != _voxeet_websocket && isSocketOpen()) {
                _voxeet_websocket.close(true);
            }

            Log.d(TAG, "onCall: start login in promise");
            lockConnectAttempt();

            Log.d(TAG, "onCall: having the list with elements := " + mWaitForLogInSocket);
            // the current list if not empty, we make sure that we are not calling it more than one 1 time
            if (mWaitForLogInSocket.size() == 0) {
                mWaitForLogInSocket.add(solver);

                unlockConnectAttempt(); //unlock here not before
                IdentifyInlogin().then((result) -> {
                    //nothing, will be in the events
                    Log.d(TAG, "onCall: first login part done");
                }).error(error -> {
                    Log.d(TAG, "onError: login error " + error.getMessage());
                    //solver.reject(error);

                    lockConnectAttempt();
                    rejectLoginSockets(error);
                    clearLoginSockets();
                    unlockConnectAttempt(); //unlock here not before
                    Log.d(TAG, "onError: login error managed");
                });
            } else {
                Log.d(TAG, "onCall: already have a login attempt in progress");
                Promise.reject(new IllegalStateException("Can not open a session while an other one is trying to be started"));
                unlockConnectAttempt(); //unlock here not before

                //nothing to do, it will be resolved later
            }
            Log.d(TAG, "onCall: start login done... can take a time");
        });
    }

    /**
     * Closes the current WebSocket without logging out participants. This method is recommended to perform soft log out locally and keeping participant information on a server.
     */
    @NoDocumentation
    public void closeSocket() {
        _voxeet_websocket.close(true);
    }

    @NoDocumentation
    @Deprecated
    public void cleanParticipantSession(@NonNull String id) {
        getVoxeetHttp().cleanParticipantSession(id);
    }

    /**
     * In the SDK, the WebSocket is the last stage of the login process (the first stage is the authentication using the REST API).
     *
     * @return the current connectivity check.
     */
    public boolean isSocketOpen() {
        return _voxeet_websocket.isOpen();
    }

    /**
     * Get the registered ParticipantInfo for the current session
     * <p>
     * If no ParticipantInfo have been provided, e.g. no login done, the reference will be null
     *
     * @return the internal reference or null
     */
    @NoDocumentation
    @Nullable
    public ParticipantInfo getParticipantInfo() {
        return _participant_infos;
    }

    /**
     * Get the identifier for the current participant obtained during the login process
     *
     * @return the identifier, a string, or null if no login or a logout has been performed
     */
    @NoDocumentation
    @Nullable
    public String getParticipantId() {
        return VoxeetPreferences.id();
    }

    /**
     * Gets a corresponding currently logged in participants' representation. It is not an object related to any conference.
     *
     * @return a new instance aggregating the ID and participantInfo.
     */
    @NoDocumentation
    @Nullable
    public Participant getParticipant() {
        String id = getParticipantId();
        ParticipantInfo participantInfo = getParticipantInfo();

        return null != id ? new Participant(id, participantInfo) : null;
    }

    /**
     * Checks if the participant is local or external.
     *
     * @param participant valid participant's ID
     * @return informs if the conference and locally connected participants have the same Voxeet IDs.
     */
    public boolean isLocalParticipant(@NonNull Participant participant) {
        String id = participant.getId();
        return null != id && id.equals(getParticipantId());
    }

    /**
     * Logs out from the current session.
     * Logging out cancels every login progress and leaves the conference.
     *
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Boolean> close() {
        return new Promise<>(solver -> {
            LogFactory.instance.disconnect();
            Log.d(TAG, "onCall: logout called");
            ConferenceService service = VoxeetSDK.conference();

            lockConnectAttempt();
            if (!isInProperServiceState(service)) {
                try {
                    throwInvalidServices();
                } catch (IllegalStateException e) {
                    rejectLoginSockets(e);
                    solver.reject(e);
                }
                return;
            }
            try {
                throw new IllegalStateException("You have awaiting login, the logout automatically canceled those");
            } catch (Exception e) {
                rejectLoginSockets(e);
            }
            clearLoginSockets();
            unlockConnectAttempt();

            try {
                service.clearConferencesInformation();

                if (service.isLive()) {
                    service.leave()
                            .then((result, solver1) -> Log.d(TAG, "onCall: leaving while logging out done"))
                            .error(error -> {
                                Log.d(TAG, "onError: leaving when logout error -- for information :");
                                error.printStackTrace();
                            });
                }
            } catch (Exception e) {
                Log.d(TAG, "onCall: WARNING :: please report this error :");
                e.printStackTrace();
            }

            Log.d(TAG, "onCall: clean awaiting sockets done");

            closeInSafeState()
                    .then((result, s) -> {
                        Log.d(TAG, "onCall: logout result := " + result + " ... propagating...");
                        solver.resolve(result);
                    })
                    .error(solver::reject);
        });
    }

    private Promise<Boolean> closeInSafeState() {
        return new Promise<>(solver -> {
            String id = getParticipantId();
            if (null != id) {
                Log.d(TAG, "Attempting to logout");
                getVoxeetHttp().logout(id).then((result, internal_solver) -> {
                    VoxeetPreferences.onLogout();
                    closeSocket();

                    String message;
                    if (null != result && result) {
                        message = "Logout success";

                        getEventBus().post(new SdkLogoutSuccessEvent(message));
                        solver.resolve(true);
                    } else {
                        message = "Logout failed";

                        getEventBus().post(new SdkLogoutErrorEvent(message));
                        solver.resolve(false);
                    }
                }).error(error -> {
                    error.printStackTrace();

                    VoxeetPreferences.onLogout();
                    closeSocket();


                    getEventBus().post(new SdkLogoutErrorEvent(handleError(error)));
                    solver.reject(error);
                });
            } else {
                final String message = "Already logged out";
                Log.d(TAG, message);
                getEventBus().post(new SdkLogoutSuccessEvent("Already logged out"));
                solver.resolve(true);
            }
        });
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(SocketStateChangeEvent event) {
        if (WebSocketState.CONNECTED.equals(event.state)) {
            lockConnectAttempt();
            resolveLoginSockets();
            unlockConnectAttempt(); //unlock here not before
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(SocketConnectErrorEvent event) {
        lockConnectAttempt();
        int i = 0;
        while (i < mWaitForLogInSocket.size()) {
            Log.d(TAG, "onEvent: calling resolved false");
            Solver<Boolean> solver = mWaitForLogInSocket.get(i);
            try {
                solver.resolve(false);
            } catch (Exception e) {
                e.printStackTrace();
            }
            i++;
        }

        clearLoginSockets();
        unlockConnectAttempt(); //unlock here not before
    }

    @NonNull
    private Promise<Boolean> IdentifyInlogin() {
        if (null == _participant_infos) return rejectParticipantNull("on login attempt");
        return getVoxeetHttp().identifyChain(_participant_infos);
    }

    private void lockConnectAttempt() {
        try {
            lockConnectAttempt.lock();
        } catch (Exception e) {

        }
    }

    private void unlockConnectAttempt() {
        try {
            if (lockConnectAttempt.isLocked()) {
                lockConnectAttempt.unlock();
            }
        } catch (Exception e) {

        }
    }

    private void rejectLoginSockets(@NonNull Throwable error) {
        try {
            int i = 0;
            while (i < mWaitForLogInSocket.size()) {
                Log.d(TAG, "onError: calling reject");
                Solver<Boolean> solver = mWaitForLogInSocket.get(i);
                try {
                    solver.reject(error);
                } catch (Exception e) {
                    e.printStackTrace();
                }
                i++;
            }
        } catch (Exception err) {
            err.printStackTrace();
        }
    }

    private void resolveLoginSockets() {
        try {
            int i = 0;
            while (i < mWaitForLogInSocket.size()) {
                Log.d(TAG, "onEvent: calling resolved true");
                Solver<Boolean> solver = mWaitForLogInSocket.get(i);
                try {
                    solver.resolve(true);
                } catch (Exception e) {
                    e.printStackTrace();
                }
                i++;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        clearLoginSockets();
    }

    private void clearLoginSockets() {
        try {
            mWaitForLogInSocket.clear();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private Promise<Boolean> rejectParticipantNull(@NonNull String origin) {
        return Promise.reject(new IllegalStateException("Invalid ! participant not set ! sent from " + origin));
    }
}