package com.voxeet.sdk.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.VoxeetSDK;
import com.voxeet.sdk.events.sdk.QualityIndicators;
import com.voxeet.sdk.json.ConferenceStats;
import com.voxeet.sdk.media.MediaSDK;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.services.localstats.LocalStatsHandler;
import com.voxeet.sdk.services.localstats.LocalStatsParticipantInfo;
import com.voxeet.sdk.services.localstats.events.LocalStatsEvent;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.stats.LocalStats;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * Service responsible for statistics management from the WebRTC implementation. Every statistics are local only
 */
@Annotate
public class LocalStatsService extends AbstractVoxeetService implements Runnable {
    public final static int NEXT_LOOP_FETCH_MS = 5000;

    private LocalStatsHandler mHandler;
    private HashMap<String, HashMap<String, LocalStatsParticipantInfo>> mConferencesHolder;

    @NoDocumentation
    public LocalStatsService(@NonNull SdkEnvironmentHolder instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        mConferencesHolder = new HashMap<>();
        mHandler = null;
    }

    /**
     * Retrieve the in-memory holder of the participant specifically designed for the participant
     *
     * @param conferenceId  the conferenceId - not the alias
     * @param participantId the participantId - not the externalId
     * @return a valid participant info holder
     */
    public LocalStatsParticipantInfo getParticipantInfo(@NonNull String conferenceId, @NonNull String participantId) {
        HashMap<String, LocalStatsParticipantInfo> holder = getParticipantStatsHolder(conferenceId);
        LocalStatsParticipantInfo info = holder.get(participantId);

        if (null == info) {
            info = new LocalStatsParticipantInfo(participantId);
            holder.put(participantId, info);
        }
        return info;
    }

    /**
     * Get the specified participant stats from a local point of view
     *
     * @param peer the participant id
     * @return LocalStats or null
     */
    @Nullable
    public LocalStats getLocalStats(@Nullable String peer) {
        if (null == peer) return null;
        MediaSDK media = VoxeetSDK.mediaDevice().getMedia();
        if (null != media) {
            /*int counter = 0;
            LocalStats stats;
            do {
                stats = media.getLocalStats(peer);
                counter++;
            } while (counter < 10 && null == stats.getRawJson());
            return stats;*/
            return media.getLocalStats(peer);
        }
        return null;
    }

    /**
     * Start an autofetch loop which will get the stats information at various times
     *
     * @return an indicator telling if the session is now started or already started
     */
    public boolean startAutoFetch() {
        if (null == mHandler) {
            mHandler = new LocalStatsHandler(this, NEXT_LOOP_FETCH_MS);
            mHandler.startAutoFetch();
            return true;
        } else {
            return false;
        }
    }

    /**
     * Stop any autofetching loop
     *
     * @return an indicator telling if it's now stoppped or was already stopped
     */
    public boolean stopAutoFetch() {
        if (null != mHandler) {
            mHandler.stopAutoFetch();
            mHandler = null;
            return true;
        }
        return false;
    }

    @NoDocumentation
    @Override
    public void run() {
        List<LocalStats> stats = new ArrayList<>();
        Conference conference = VoxeetSDK.conference().getConference();

        if (null != conference) {
            List<Participant> users = conference.getParticipants();
            for (int i = 0; i < users.size(); i++) {
                Participant user = users.get(i);
                LocalStats stat = getLocalStats(user.getId());
                getParticipantInfo(conference.getId(), user.getId()).addFromLocalStats(stat);
                stats.add(stat);
            }

            getEventBus().post(new LocalStatsEvent(stats));
        } else {
            //no conference ?
        }
    }

    /**
     * Retrieve the in-memory holder of the conference information specifically designed for the stats
     *
     * @param conferenceId the conferenceId - not the alias
     * @return a non null HashMap instance
     */
    @NonNull
    private HashMap<String, LocalStatsParticipantInfo> getParticipantStatsHolder(@NonNull String conferenceId) {
        HashMap<String, LocalStatsParticipantInfo> holder = mConferencesHolder.get(conferenceId);

        if (null == holder) {
            holder = new HashMap<>();
            mConferencesHolder.put(conferenceId, holder);
        }
        return holder;
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceStats stats) {
        String conferenceId = VoxeetSDK.conference().getConferenceId();
        if (stats.getConference_id() != null && stats.getConference_id().equals(conferenceId)) {
            float mos = stats.getScore(VoxeetSDK.session().getParticipantId());

            getEventBus().post(new QualityIndicators(mos));
        }
    }
}
