package com.voxeet.sdk.services;

import android.support.annotation.NonNull;

import com.voxeet.VoxeetSDK;
import com.voxeet.promise.Promise;
import com.voxeet.sdk.network.endpoints.IRestApiMessaging;
import com.voxeet.sdk.services.conference.promises.SendBroadcastMessagePromise;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

/**
 * The CommandService allows the application to [send](/documentation/sdk/reference/android/command#send) textual [ChatMessage](/documentation/sdk/reference/android/models/chatmessage) messages to all clients and redistributes them across the EventBus. It also emits events informing about the statuses of message transmission. The [BroadcastEvent](/documentation/sdk/reference/android/command#broadcastevent) informs that the message is broadcasted. The [MessageReceived](/documentation/sdk/reference/android/command#messagereceived) event informs that the message is received at the specific conference.
 */
@Annotate
public class CommandService extends AbstractVoxeetService<IRestApiMessaging> {
    @NoDocumentation
    public CommandService(@NonNull SdkEnvironmentHolder instance) {
        super(instance, IRestApiMessaging.class);
    }


    /**
     * Sends the message to the conference. The message must be in the form of a string or a representation of strings (json or base64).
     *
     * @param conferenceId ID of the conference to which the message should be sent
     * @param message      actual message content (any possible string)
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Boolean> send(@NonNull String conferenceId, @NonNull final String message) {
        return new SendBroadcastMessagePromise(VoxeetSDK.conference(),
                VoxeetSDK.mediaDevice(),
                getService(),
                VoxeetSDK.conference().getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId,
                message).createPromise();
    }
}
