package com.voxeet.sdk.services;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.VoxeetSDK;
import com.voxeet.promise.solve.ErrorPromise;
import com.voxeet.sdk.push.utils.NoDocumentation;
import com.voxeet.sdk.services.holder.ServiceProviderHolder;

import org.greenrobot.eventbus.EventBus;

public abstract class AbstractVoxeetService<T> {

    private static final String TAG = AbstractVoxeetService.class.getSimpleName();

    private SdkEnvironmentHolder sdk_environment_holder;

    @NonNull
    private ServiceProviderHolder<T> holder = ServiceProviderHolder.DEFAULT;

    /**
     * The Handler.
     */
    protected final static Handler handler = new Handler(Looper.getMainLooper());

    /**
     * The Context.
     */
    protected Context context;

    private AbstractVoxeetService() {

    }

    /**
     * Instantiates a new Voxeet service.
     *
     * @param holder        the voxeet parent information holder
     * @param service_klass
     */
    protected AbstractVoxeetService(@NonNull SdkEnvironmentHolder holder, @Nullable Class<T> service_klass) {
        this(holder, new ServiceProviderHolder.Builder<T>()
                .setClient(holder.voxeetHttp)
                .setService(service_klass)
                .build());
    }

    /**
     * Instantiates a new Voxeet service.
     *
     * @param instance the instance of the SDK
     */
    protected AbstractVoxeetService(@NonNull SdkEnvironmentHolder instance, @NonNull ServiceProviderHolder<T> holder) {
        this();

        this.sdk_environment_holder = instance;
        this.context = instance.voxeetSdk.getApplicationContext();

        this.holder = holder;
    }

    @Nullable
    final protected T getService() {
        return holder.getService();
    }

    @Nullable
    final protected <TYPE> TYPE getService(Class<TYPE> klass) {
        return holder.getService(klass);
    }

    /**
     * Retrieves the error string from the request.
     *
     * @param e the exception
     * @return the error content
     */
    public String handleError(@Nullable Throwable e) {
        if (null == e) {
            return "invalid error";
        }
        Log.d(TAG, "handleError: " + e.getClass().getSimpleName());
        return e.getMessage();
    }

    /**
     * Reset service.
     */
    public void resetService() {

    }


    /**
     * Get the current endpoint root url
     *
     * @return
     */
    @Nullable
    protected final String getURLRoot() {
        return sdk_environment_holder.voxeetHttp.getBuiltServerUrl();
    }

    /**
     * @return the current token of the session
     */
    @Nullable
    protected final String getInternalToken() {
        return sdk_environment_holder.voxeetHttp.getToken();
    }

    /**
     * @return the current token of the session
     */
    @Nullable
    protected final String getInternalJwtToken() {
        return getVoxeetHttp().getJwtToken();
    }

    protected final VoxeetHttp getVoxeetHttp() {
        return sdk_environment_holder.voxeetHttp;
    }

    protected final SdkEnvironmentHolder getSdkEnvironmentHolder() {
        return sdk_environment_holder;
    }

    protected final VoxeetSDK getVoxeetSDK() {
        return getSdkEnvironmentHolder().voxeetSdk;
    }

    protected EventBus getEventBus() {
        return holder.getEventBus();
    }

    protected void registerEventBus() {
        EventBus eventBus = holder.getEventBus();
        try {
            if (!eventBus.isRegistered(this))
                eventBus.register(this);
        } catch (Exception e) {
            Log.d(TAG, "registerEventBus: logging for debug purposes :");
            e.printStackTrace();
        }
    }

    protected void unRegisterEventBus() {
        EventBus eventBus = holder.getEventBus();
        try {
            if (eventBus.isRegistered(this))
                eventBus.unregister(this);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    protected void postOnMainThread(@NonNull Runnable runnable) {
        handler.post(runnable);
    }

    protected boolean isInProperServiceState(AbstractVoxeetService... services) {
        if (null == services || services.length == 0) return false;

        for (AbstractVoxeetService service : services) {
            if (null == service) return false;
        }
        return true;
    }

    protected boolean throwInvalidServices() {
        throw new IllegalStateException("Invalid service state... stopping...");
    }

    @NoDocumentation
    @NonNull
    protected ErrorPromise manageError() {
        return Throwable::printStackTrace;
    }
}
