package com.voxeet.sdk.json;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonTypeName;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.models.v1.Conference;
import com.voxeet.sdk.models.v1.ConferenceType;
import com.voxeet.sdk.utils.FromWebsocket;
import com.voxeet.sdk.utils.InternalEvent;
import com.voxeet.sdk.utils.Opt;

@InternalEvent
@FromWebsocket
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonTypeName(EventNames.INVITATION_RECEIVED)
public class InvitationReceivedEvent extends Event {
    public String conferenceId;

    @Nullable
    @JsonProperty("confAlias")
    public String conferenceAlias;

    @Nullable
    @JsonProperty("userId")
    public String userId;

    @Nullable
    @JsonProperty("participantId")
    public String participantId;

    @Nullable
    @JsonProperty("externalId")
    public String externalId;

    @Nullable
    @JsonProperty("externalName")
    public String externalName;

    @JsonProperty("externalAvatarUrl")
    public String externalAvatarUrl;

    @Nullable
    public ParticipantInviter inviter;

    @Nullable
    @JsonProperty("conferenceInfo")
    public Conference conference;

    @Override
    public String getType() {
        return EventNames.INVITATION_RECEIVED;
    }

    @NonNull
    public Participant getInviter() {
        String id = Opt.of(participantId).or(userId);
        String externalId = Opt.of(this.externalId).orNull();
        String avatarUrl = Opt.of(this.externalAvatarUrl).orNull();
        String userName = Opt.of(this.externalName).orNull();

        if (null != inviter) {
            if (TextUtils.isEmpty(id)) id = Opt.of(inviter.participantId).orNull();
            if (TextUtils.isEmpty(externalId)) externalId = Opt.of(inviter.externalId).orNull();
            if (TextUtils.isEmpty(userName))
                userName = Opt.of(inviter.externalName).or(Opt.of(inviter.nickName).or(""));
            if (TextUtils.isEmpty(avatarUrl))
                avatarUrl = Opt.of(inviter.externalAvatarUrl).orNull();
        }

        if (TextUtils.isEmpty(id)) id = "";
        if (TextUtils.isEmpty(externalId)) externalId = "";
        if (TextUtils.isEmpty(userName)) userName = "";
        if (TextUtils.isEmpty(avatarUrl)) avatarUrl = "";

        return new Participant(id, new ParticipantInfo(userName, externalId, avatarUrl));
    }

    @NonNull
    public String id() {
        if (!TextUtils.isEmpty(conferenceId)) return conferenceId;

        if (null != conference) {
            String conferenceId = conference.getConferenceId();
            if (null != conferenceId) return conferenceId;
        }
        return "";
    }

    @NonNull
    public ConferenceType type() {
        if (null != conference && null != conference.getConferenceType()) {
            return ConferenceType.fromId(conference.getConferenceType());
        }
        return ConferenceType.SDK_CONF;
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public class ParticipantInviter {

        public ParticipantInviter() {

        }

        public ParticipantInviter(String participantId, String externalId, String externalAvatarUrl, String externalName) {
            this();
            this.participantId = participantId;
            this.externalId = externalId;
            this.externalAvatarUrl = externalAvatarUrl;
            this.externalName = externalName;
            this.nickName = externalName;
        }

        @Nullable
        @JsonProperty("userId")
        public String participantId;

        @Nullable
        public String externalId;

        @Nullable
        public String externalAvatarUrl;

        @Nullable
        public String externalName;

        @Nullable
        public String nickName;
    }
}
