package com.voxeet.sdk.services;

import android.support.annotation.NonNull;

import com.voxeet.promise.Promise;
import com.voxeet.sdk.VoxeetSdk;
import com.voxeet.sdk.events.sdk.MessageReceived;
import com.voxeet.sdk.factories.EventsFactory;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.models.v2.ChatMessage;
import com.voxeet.sdk.services.chat.ChatMessageEvent;
import com.voxeet.sdk.services.chat.InternalChatMessage;
import com.voxeet.sdk.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;

/**
 * Module to manage incoming well formatted ChatMessage and redistribute them accross the EventBus
 */
@Annotate
public class ChatService extends AbstractVoxeetService {

    @NoDocumentation
    public ChatService(@NonNull SdkEnvironmentHolder instance) {
        super(instance, ServiceProviderHolder.DEFAULT);
    }

    @Subscribe
    public void onEvent(MessageReceived event) {
        InternalChatMessage chatMessage = EventsFactory.mapping(event.message, InternalChatMessage.class);

        ConferenceService conferenceService = VoxeetSdk.conference();
        if (null != conferenceService && null != chatMessage) {
            Participant participant = conferenceService.findParticipantById(chatMessage.ownerId);
            //ChatMessage
            //Participant
            EventBus.getDefault().post(new ChatMessageEvent(participant, new ChatMessage(chatMessage)));
        }
    }

    /**
     * Send a message to the various participants
     *
     * @param conference the conference to send the message to
     * @param chatMessage the actual message
     * @return a promise to resolve
     */
    @NonNull
    public Promise<Boolean> sendMessage(@NonNull Conference conference, @NonNull ChatMessage chatMessage) {
        SessionService sessionService = VoxeetSdk.session();
        CommandService commandService = VoxeetSdk.command();

        if(null != sessionService && null != commandService) {
            Participant participant = new Participant(sessionService.getParticipantId(), sessionService.getParticipantInfo());
            InternalChatMessage internalChatMessage = new InternalChatMessage("Chat_Message", participant, chatMessage);

            String json = EventsFactory.getJson(internalChatMessage);
            if(null != json) {
                return commandService.send(conference.getId(), json);
            }
        }

        return Promise.reject(new IllegalStateException("The sdk is not initialized"));
    }

}
