package com.voxeet.sdk.services.media;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaStream;
import com.voxeet.sdk.VoxeetSdk;
import com.voxeet.sdk.services.MediaDeviceService;
import com.voxeet.sdk.utils.Opt;

import org.webrtc.VideoFrame;
import org.webrtc.VideoSink;

import java.lang.ref.WeakReference;
import java.util.ArrayList;

public class VideoSinkHolder {
    @NonNull
    private WeakReference<VideoSink> holder;

    private ArrayList<MediaStreamListener> mediaStreamListeners;

    private VideoSinkHolder() {
        mediaStreamListeners = new ArrayList<>();
    }

    public VideoSinkHolder(@NonNull VideoSink videoSink) {
        this();
        holder = new WeakReference<>(videoSink);
    }

    public boolean hasVideoSink() {
        return holder.get() != null;
    }

    public void clear() {
        MediaEngine media = Opt.of(VoxeetSdk.mediaDevice()).then(MediaDeviceService::getMedia).orNull();
        unattach(media);
    }

    @Nullable
    public MediaStream getMediaStream() {
        for (MediaStreamListener listener : mediaStreamListeners) {
            if (listener.active) return listener.mediaStream;
        }
        return null;
    }

    public void attach(@NonNull MediaEngine media, @NonNull MediaStream stream) {
        VideoSink videoSink = holder.get();
        unattach(media);

        if (null != videoSink) {
            MediaStreamListener mediaStreamListener = new MediaStreamListener(stream);
            mediaStreamListeners.add(mediaStreamListener);
            mediaStreamListener.active = true;
            media.attachMediaStream(mediaStreamListener, stream);
        }
    }

    public void unattach(@Nullable MediaEngine media) {
        VideoSink videoSink = holder.get();
        for (MediaStreamListener mediaStreamListener : mediaStreamListeners) {
            if (null != media && null != videoSink && mediaStreamListener.active) {
                media.unattachMediaStream(mediaStreamListener, mediaStreamListener.mediaStream);
            }
            mediaStreamListener.active = false;
        }
        mediaStreamListeners.clear();
    }

    @Override
    public boolean equals(@Nullable Object obj) {
        VideoSink videoSink = holder.get();

        return obj == this || (obj instanceof VideoSink && obj.equals(videoSink));
    }

    private class MediaStreamListener implements VideoSink {
        public MediaStream mediaStream;
        public boolean active;

        public MediaStreamListener(MediaStream mediaStream) {
            this.mediaStream = mediaStream;
            this.active = true;
        }

        @Override
        public void onFrame(VideoFrame frame) {
            VideoSink videoSink = holder.get();
            if (active) {
                if (null != videoSink) videoSink.onFrame(frame);
            } else {
                MediaEngine media = Opt.of(VoxeetSdk.mediaDevice()).then(MediaDeviceService::getMedia).orNull();

                if (Opt.isNonNull(videoSink, media)) {
                    media.unattachMediaStream(videoSink, mediaStream);
                }
            }
        }
    }


}
