package com.voxeet.sdk.preferences;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.sdk.json.ParticipantInfo;
import com.voxeet.sdk.utils.AbstractVoxeetEnvironmentHolder;

import java.util.UUID;

/**
 * The type Voxeet preferences.
 */
public class VoxeetPreferences {

    private final static String TAG = VoxeetPreferences.class.getSimpleName();

    private final static String ACCOUNT_TYPE = "acct_type";
    private final static String ID = "id";

    private final static String DEVICE_ID = "device_id";
    private final static String UPLOAD_TOKEN = "upload_token";

    /**
     * The constant LOGIN.
     */

    private final static String LOGIN = "login";

    /**
     * The constant COOKIE.
     */
    private final static String COOKIE = "cookie";

    /**
     * The constant PASSWORD.
     */
    private final static String PASSWORD = "password";

    /**
     * The constant APP_VERSION.
     */
    private final static String APP_VERSION = "app_version";

    private final static String DEFAULT_VIDEO_ON = "default_video_on";
    private final static String DEFAULT_BUILT_IN_SPEAKER_ON = "default_video_on";

    /**
     * The constant MANIFEST_URL.
     */
    private final static String MANIFEST_URL = "manifest_url";

    private final static String SDK_EXTERNAL_ID = "SDK_INTERNAL_ID";
    private final static String SDK_EXTERNAL_NAME = "SDK_EXTERNAL_NAME";
    private final static String SDK_EXTERNAL_AVATARURL = "SDK_EXTERNAL_AVATARURL";
    private static AbstractVoxeetEnvironmentHolder VoxeetEnvironmentHolder;

    private static IPreferencesProvider preferences;

    private VoxeetPreferences() {

    }

    /**
     * Init.
     *
     * @param context                 the context
     * @param voxeetEnvironmentHolder
     */
    public static void init(@NonNull Context context, AbstractVoxeetEnvironmentHolder voxeetEnvironmentHolder) {
        VoxeetEnvironmentHolder = voxeetEnvironmentHolder;
        SharedPreferences prefs = context.getSharedPreferences("main", Context.MODE_PRIVATE);


        setCustomPreferencesProvider(context, new SharedPreferencesCacheWrapper(prefs));

        if(null == deviceIdentifier()) {
            save(DEVICE_ID, UUID.randomUUID().toString());
        }
    }

    public static void setCustomPreferencesProvider(@NonNull Context context,
                                                    @NonNull IPreferencesProvider provider) {

        int version = 0;
        try {
            version = context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionCode;
        } catch (PackageManager.NameNotFoundException e) {
            Log.e(TAG, "Error accessing application version number", e);
        }

        preferences = provider;
        preferences.edit()
                .putString(MANIFEST_URL, VoxeetEnvironmentHolder.getManifestUrl())
                .putInt(APP_VERSION, version)
                .apply();

    }


    /**
     * participant id string.
     *
     * @return the string
     */
    @Deprecated
    @Nullable
    public static String id() {
        if (preferences != null && preferences.contains(ID)) {
            return preferences.getString(ID, null);
        } else {
            return null;
        }
    }

    public static boolean isDefaultVideoOn() {
        return preferences != null && preferences.getBoolean(DEFAULT_VIDEO_ON, false);
    }

    public static boolean isDefaultBuiltinSpeakerOn() {
        return preferences != null && preferences.getBoolean(DEFAULT_BUILT_IN_SPEAKER_ON, false);
    }

    public static void setDefaultVideoOn(boolean defaultVideoOn) {
        save(DEFAULT_VIDEO_ON, defaultVideoOn);
    }

    public static void setDefaultBuiltInSpeakerOn(boolean defaultBuiltInSpeakerOn) {
        save(DEFAULT_BUILT_IN_SPEAKER_ON, defaultBuiltInSpeakerOn);
    }

    /**
     * Sets id.
     *
     * @param id the id
     */
    public static void setId(String id) {
        if (preferences != null) {
            preferences.edit()
                    .putString(ID, id)
                    .apply();
        }
    }

    /**
     * Device identifier string.
     *
     * @return the string
     */
    public static String deviceIdentifier() {
        return preferences.getString(DEVICE_ID, null);
    }

    /**
     * Clear preferences on logout.
     */
    public static void onLogout() {
        Log.d(TAG, "onLogout");
        if (preferences != null) {
            SharedPreferencesCacheWrapper.Editor editor = preferences.edit();

            editor.putString(LOGIN, "")
                    .putString(PASSWORD, "")
                    .putString(UPLOAD_TOKEN, "")
                    .putString(COOKIE, "")
                    .putString(ID, "")
                    .putString(SDK_EXTERNAL_NAME, null)
                    .putString(SDK_EXTERNAL_ID, null)
                    .putString(SDK_EXTERNAL_AVATARURL, null)
                    .apply();
        }
    }

    /**
     * Gets login cookie.
     *
     * @return the login cookie
     */
    public static String getLoginCookie() {
        return preferences.getString(COOKIE, null);
    }

    /**
     * Save login cookie.
     *
     * @param cookie the cookie
     */
    public static void saveLoginCookie(String cookie) {
        save(COOKIE, cookie);
    }

    @Nullable
    public static ParticipantInfo getSavedUserInfo() {
        String externalId = getSdkExternalId();
        String avatarUrl = getSdkExternalAvatarurl();
        String externalName = getSdkExternalName();

        if (null != externalId) {
            return new ParticipantInfo(externalName, externalId, avatarUrl);
        }
        return null;
    }

    public static String getSdkExternalId() {
        if (null != preferences)
            return preferences.getString(SDK_EXTERNAL_ID, null);
        return null;
    }

    public static String getSdkExternalName() {
        if (null != preferences)
            return preferences.getString(SDK_EXTERNAL_NAME, null);
        return null;
    }

    public static String getSdkExternalAvatarurl() {
        if (null != preferences)
            return preferences.getString(SDK_EXTERNAL_AVATARURL, null);
        return null;
    }

    public static void setExternalName(String externalName) {
        save(SDK_EXTERNAL_NAME, externalName);
    }

    public static void setExternalId(String externalId) {
        save(SDK_EXTERNAL_ID, externalId);
    }

    public static void setExternalAvatarUrl(String externalAvatarUrl) {
        save(SDK_EXTERNAL_AVATARURL, externalAvatarUrl);
    }

    public static void setExternalUserInfo(@Nullable ParticipantInfo participantInfo) {
        if (null == participantInfo) return;

        setExternalId(participantInfo.getExternalId());
        setExternalAvatarUrl(participantInfo.getAvatarUrl());
        setExternalName(participantInfo.getName());
    }

    private static void save(@NonNull String key, @Nullable String value) {
        if (preferences != null) preferences.edit().putString(key, value).apply();
    }

    private static void save(@NonNull String key, boolean value) {
        if (preferences != null) preferences.edit().putBoolean(key, value).apply();
    }
}
