package com.voxeet.sdk.models;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaStream;
import com.voxeet.sdk.json.ParticipantInfo;
import com.voxeet.sdk.models.v1.ConferenceParticipantStatus;
import com.voxeet.sdk.models.v1.ConferenceUser;
import com.voxeet.sdk.models.v1.RestParticipant;
import com.voxeet.sdk.models.v1.SdkParticipant;
import com.voxeet.sdk.models.v2.ParticipantMediaStreamHandler;
import com.voxeet.sdk.push.center.management.Constants;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.Opt;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * The Participant model contains information about conference participants, such as [ID](/documentation/sdk/reference/android/models/participant#getid), [status](/documentation/sdk/reference/android/models/participant#getstatus), and [information](/documentation/sdk/reference/android/models/participant#getinfo), including the `external ID`, `name`, and `URL of the avatar`. It also [updates participant status](/documentation/sdk/reference/android/models/participant#updatestatus) and enables to check if the Participant [is locally active](/documentation/sdk/reference/android/models/participant#islocallyactive). It accesses [streams](/documentation/sdk/reference/android/models/participant#streams) available for the participant and [streams manager](/documentation/sdk/reference/android/models/participant#streamshandler) that enables stream manipulation.
 *
 * The developers are responsible for the `ParticipantInfo` element. The ID is provided and managed by the SDK.
 */
@Annotate
public class Participant {

    @NonNull
    private ParticipantMediaStreamHandler streamsHandler;

    @NonNull
    private CopyOnWriteArrayList<MediaStream> streams;

    @Nullable
    private String id;

    @Nullable
    private ParticipantInfo participantInfo;

    @NonNull
    private ConferenceParticipantStatus status;
    private String quality; //needed ?

    private Participant() {
        streams = new CopyOnWriteArrayList<>();
        streamsHandler = new ParticipantMediaStreamHandler(this);
    }

    @NoDocumentation
    public Participant(@Nullable String id, @Nullable ParticipantInfo participantInfo) {
        this();
        this.id = id;
        this.participantInfo = participantInfo;
    }

    /**
     * Create a new ConferenceUser
     *
     * @param conferenceUser the conference participant to use as creator
     */
    @NoDocumentation
    public Participant(@NonNull ConferenceUser conferenceUser) {
        this(conferenceUser.getUserId(), conferenceUser.getParticipantInfo());
        updateStatus(conferenceUser.getConferenceStatus());
    }

    /**
     * Create a new ConferenceUser
     *
     * @param participantInfo the developer's management of the information
     */
    @NoDocumentation
    public Participant(@Nullable ParticipantInfo participantInfo) {
        this(null, participantInfo);
    }

    /**
     * Create a new ConferenceUser
     *
     * @param participant from a given participant obtained from a server event
     */
    @NoDocumentation
    public Participant(@NonNull RestParticipant participant) {
        this(participant.getParticipantId(), new ParticipantInfo(participant.getName(), participant.getExternalId(), participant.getAvatarUrl()));
    }

    @NoDocumentation
    public Participant(@NonNull SdkParticipant participant) {
        this(participant.getUserId(),
                new ParticipantInfo(
                        participant.getMetadata().getExternalName(),
                        participant.getMetadata().getExternalId(),
                        participant.getMetadata().getExternalPhotoUrl()
                )
        );
        updateStatus(ConferenceParticipantStatus.valueOf(participant.getStatus()));
    }

    /**
     * Informs if the current Participant instance is connected to the local device.
     *
     * @return the flag indicating if the participant is already connected.
     */
    public boolean isLocallyActive() {
        return ConferenceParticipantStatus.ON_AIR.equals(getStatus())
                ||
                (ConferenceParticipantStatus.CONNECTING.equals(getStatus()) && streams().size() > 0);
    }

    /**
     * Gets the Voxeet identifier of the specific conference participant.
     *
     * @return possibly a null value.
     */
    @Nullable
    public String getId() {
        return id;
    }

    /**
     * Gets information about the current participant.
     *
     * @return displayed information.
     */
    @Nullable
    public ParticipantInfo getInfo() {
        return participantInfo;
    }

    /**
     * An accessor to streams available for the current Participant instance.
     *
     * @return a non-null list containing the `MediaStream`.
     */
    @NonNull
    public List<MediaStream> streams() {
        return streams;
    }

    /**
     * The accessor to the streams manager available for the current Participant instance. The manager adds filtering and methods to manipulate the streams.
     *
     * @return a non-null instance of the `ParticipantMediaStreamHandler`.
     */
    @NonNull
    public ParticipantMediaStreamHandler streamsHandler() {
        return streamsHandler;
    }


    /**
     * Sets the participant's status from a server point of view. It can be used only by the SDK.
     *
     * Warning: the status does not check any lifecycle, set improper information, and does not manage it. Therefore, it can lead to an issue on the developer side.
     *
     * @param status The valid participant's status.
     * @return the current instance of the participant.
     */
    @NoDocumentation
    @NonNull
    public Participant updateStatus(@NonNull ConferenceParticipantStatus status) {
        this.status = status;
        return this;
    }

    /**
     * Gets the current status.
     *
     * @return the instance of the valid status.
     */
    @NonNull
    public ConferenceParticipantStatus getStatus() {
        return status;
    }

    /**
     * Check for the equality of another object
     * - ConferenceUser
     * - RestParticipant
     * <p>
     * In order to proceed to an update of the instance if needed internally in the SDK
     *
     * @param obj a nullable object to check against
     * @return true if both are equals or "compatible" for update
     */
    @NoDocumentation
    @Override
    public boolean equals(@Nullable Object obj) {
        if (null == id && null == participantInfo) return false;
        boolean same = false;

        if (this == obj) return true;

        if (obj instanceof ConferenceUser) {
            ConferenceUser conferenceUser = (ConferenceUser) obj;
            ParticipantInfo conferenceParticipantInfo = conferenceUser.getParticipantInfo();
            String externalId = null != participantInfo ? participantInfo.getExternalId() : null;
            if (null != id) same |= id.equals(conferenceUser.getUserId());
            if (null != externalId && null != conferenceParticipantInfo)
                same |= externalId.equals(conferenceParticipantInfo.getExternalId());
        }

        if (obj instanceof RestParticipant) {
            RestParticipant participant = (RestParticipant) obj;
            String externalId = null != participantInfo ? participantInfo.getExternalId() : null;

            if (null != id) same |= id.equals(participant.getParticipantId());
            if (null != id && null != externalId)
                same |= externalId.equals(participant.getExternalId());
        }

        if (obj instanceof Participant) {
            Participant participant = (Participant) obj;
            String externalId = null != participantInfo ? participantInfo.getExternalId() : null;
            ParticipantInfo other = participant.getInfo();
            if (null != id) same |= id.equals(participant.getId());
            if (null != id && null != externalId && null != other)
                same |= externalId.equals(other.getExternalId());
        }

        return same;
    }

    /**
     * Set the current participant info
     * <p>
     * Note : it does not lead to an update server side, this must be done afterward
     * Every field can be null
     *
     * @param participantInfo a valid instance of information
     * @return the current participant instance
     */
    @NoDocumentation
    public Participant setParticipantInfo(@NonNull ParticipantInfo participantInfo) {
        this.participantInfo = participantInfo;
        return this;
    }

    /**
     * Set the quality for this specific participant
     *
     * @param quality
     * @deprecated
     */
    @NoDocumentation
    @Deprecated
    public void setQuality(@NonNull String quality) {
        this.quality = quality;
    }

    /**
     * Check for a required local update of the field of this current instance
     *
     * @param name      the name to set
     * @param avatarUrl the avatar url - can be null
     */
    @NoDocumentation
    public void updateIfNeeded(@Nullable String name, @Nullable String avatarUrl) {
        if (null == participantInfo) {
            participantInfo = new ParticipantInfo();
        }

        if (null != name && (null == participantInfo.getName() || !name.equals(participantInfo.getName())))
            participantInfo.setName(name);
        if (null != avatarUrl && (null == participantInfo.getAvatarUrl() || avatarUrl.equals(participantInfo.getAvatarUrl())))
            participantInfo.setAvatarUrl(avatarUrl);
    }

    /**
     * Get the string representation of the given instance
     *
     * @return a valid string
     */
    @NoDocumentation
    @NonNull
    @Override
    public String toString() {
        return "Participant{" +
                "id='" + id + '\'' +
                ", participantInfo=" + participantInfo +
                ", status=" + status +
                ", quality='" + quality + '\'' +
                '}';
    }

    public Map<String, String> asMap() {
        HashMap<String, String> infos = new HashMap<>();

        infos.put(Constants.INVITER_ID, Opt.of(getId()).or(""));
        infos.put(Constants.INVITER_NAME, Opt.of(getInfo()).then(ParticipantInfo::getName).or(""));
        infos.put(Constants.INVITER_EXTERNAL_ID, Opt.of(getInfo()).then(ParticipantInfo::getExternalId).or(""));
        infos.put(Constants.INVITER_URL, Opt.of(getInfo()).then(ParticipantInfo::getAvatarUrl).or(""));

        return infos;
    }
}
