package com.voxeet.sdk.services.conference.information;

import android.support.annotation.NonNull;
import android.util.Log;

import com.voxeet.sdk.media.audio.ParticipantPosition;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.models.v1.RestParticipant;
import com.voxeet.sdk.services.media.VideoState;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The ConferenceInformation model contains  and manages information about conferences.
 * The [ConferenceInformation](/documentation/sdk/reference/android/models/conferenceinformation#conferenceinformation) constructor creates a new information holder used internally by the SDK.
 * This model gets:
 * - the participant's [position](/documentation/sdk/reference/android/models/conferenceinformation#getposition) in the audio 3D context, and [sets](/documentation/sdk/reference/android/models/conferenceinformation#setposition) this position
 * - the [Conference instance](/documentation/sdk/reference/android/models/conferenceinformation#getconference)
 * - the [conference type](/documentation/sdk/reference/android/models/conferenceinformation#getconferencetype)
 * - the [conference status](/documentation/sdk/reference/android/models/conferenceinformation#getconferencestatus)
 * - the [participant type](/documentation/sdk/reference/android/models/conferenceinformation#getconferenceusertype) and checks if the participant is a [broadcaster](/documentation/sdk/reference/android/models/conferenceinformation#isbroadcaster) or a [listener](/documentation/sdk/reference/android/models/conferenceinformation#islistener)
 * - the [video state](/documentation/sdk/reference/android/models/conferenceinformation#getvideostate)
 * - the [state of the local video](/documentation/sdk/reference/android/models/conferenceinformation#isownvideostarted) that is displayed during the conference
 *
 * It also checks if the [screen is shared](/documentation/sdk/reference/android/models/conferenceinformation#isscreenshareon) and if the conference is in the [telecom mode](/documentation/sdk/reference/android/models/conferenceinformation#istelecommode).
 */
@Annotate
public class ConferenceInformation {

    //TODO the NoDocumentation must be dealt with a private
    private static final String TAG = ConferenceInformation.class.getSimpleName();

    @NonNull
    private Conference mConference;

    @NonNull
    private Map<String, String> participantIdsCached;

    private LocalConferenceType mConferenceType;

    @NonNull
    private List<Participant> mLastInvitationReceived;
    private ConferenceStatus state = ConferenceStatus.DEFAULT;

    private HashMap<String, ParticipantPosition> mPositions;

    private boolean mOwnScreenShareStarted;
    private ConferenceParticipantType participantType;
    private boolean telecomMode;

    @NonNull
    private VideoState videoState;
    private boolean mOwnVideoStarted = false;

    private ConferenceInformation() {
        videoState = VideoState.STOPPED;
        telecomMode = false;
        participantType = ConferenceParticipantType.NORMAL;
        participantIdsCached = new HashMap<>();
        mLastInvitationReceived = new ArrayList<>();
        mConference = new Conference(this);
        mConferenceType = LocalConferenceType.NONE;
        mOwnScreenShareStarted = false;
        mPositions = new HashMap<>();
    }

    /**
     * Creates a new Information holder used internally by the SDK.
     *
     * @param localParticipant the current local session participant
     * @param conferenceId the conference ID (not the alias)
     */
    @NoDocumentation
    public ConferenceInformation(@NonNull Participant localParticipant, @NonNull String conferenceId) {
        this();

        mConference.updateParticipant(localParticipant);
        mConference.setConferenceId(conferenceId);
    }

    /**
     * Retrieves the participant position in the 3D audio context.
     *
     * @param participant The valid instance of the conference participant.
     * @return the position of the given participant.
     */
    @NonNull
    public ParticipantPosition getPosition(@NonNull Participant participant) {
        return getPosition(participant.getId());
    }

    /**
     * Retrieves the participant's position in the 3D audio context.
     *
     * @param participantId The valid participant's ID
     * @return the position of the given participant.
     */
    @NonNull
    public ParticipantPosition getPosition(@NonNull String participantId) {
        ParticipantPosition p = mPositions.get(participantId);
        if (null == p) p = new ParticipantPosition(0, 0);
        return p;
    }

    /**
     * Sets the participant's position.
     *
     * @param participant The valid instance of the conference participant.
     * @param point       The participant's position.
     */
    public void setPosition(@NonNull Participant participant, @NonNull ParticipantPosition point) {
        setPosition(participant.getId(), point);
    }

    /**
     * Sets the participant's position.
     *
     * @param participantId The participant's ID.
     * @param point       The participant's position.
     */
    public void setPosition(@NonNull String participantId, @NonNull ParticipantPosition point) {
        mPositions.put(participantId, point);
    }

    /**
     * Gets the Conference instance from the information holder.
     *
     * @return the valid instance.
     */
    @NonNull
    public Conference getConference() {
        return mConference;
    }

    /**
     * Set a conference inside this ConferenceInformation holder. To be used internally b the SDK
     *
     * @param conference
     */
    @NoDocumentation
    public void setConference(@NonNull Conference conference) {
        mConference = conference;
    }

    /**
     * Set the current conference's type : None (default when created), Conference or Replay
     * <p>
     * This is not to be changed during the developer's app lifecycle
     *
     * @param type the conference's type, preferrably Conference or Replay
     */
    @NoDocumentation
    public void setConferenceType(@NonNull LocalConferenceType type) {
        mConferenceType = type;
    }

    /**
     * Gets the current local conference type (none, conference, or replay). It is not a conference presentation from the server point of view.v
     *
     * @return the current conference type.
     */
    @NonNull
    public LocalConferenceType getConferenceType() {
        return mConferenceType;
    }

    /**
     * Gets the various of participants in cache in this conference. Will be removed in favor of the list of participants in the Conference object
     *
     * @return a valid list of Participant's Id
     */
    @NoDocumentation
    @Deprecated
    @NonNull
    public Map<String, String> getParticipantIdsCached() {
        return participantIdsCached;
    }

    /**
     * retrieve the list of participants that where invited
     *
     * @return a valid list of RestParticipant
     */
    @NoDocumentation
    @Deprecated
    @NonNull
    public List<Participant> getLastInvitationReceived() {
        return mLastInvitationReceived;
    }

    /**
     * Update the current conference state
     *
     * @param state
     */
    @NoDocumentation
    public void setConferenceState(@NonNull ConferenceStatus state) {
        this.state = state;
    }

    /**
     * Gets the current conference state.
     * All possible states are sufficient to manage a pure conference UX and UI.
     *
     * @return the conference state.
     */
    @NonNull
    public ConferenceStatus getConferenceState() {
        return state;
    }

    /**
     * Set if the participant asked for video in this conference state
     * Helpful to deal with new privacy features in Android P+
     *
     * @param ownVideoStarted a boolean indicating the proper state
     */
    @NoDocumentation
    public void setRequestedOwnVideo(boolean ownVideoStarted) {
        Log.d(TAG, "setRequestedOwnVideo: " + ownVideoStarted);
        mOwnVideoStarted = ownVideoStarted;
    }

    /**
     *  This method is helpful in dealing with new privacy features in Android P+.
     *
     * @return a boolean indicating the proper state.
     */
    public boolean isOwnVideoStarted() {
        Log.d(TAG, "isOwnVideoStarted: " + mOwnVideoStarted);
        return mOwnVideoStarted;
    }

    /**
     * Change the current participant conference type, to be used internally by the SDK
     *
     * @param participantType
     */
    @NoDocumentation
    public void setConferenceParticipantType(@NonNull ConferenceParticipantType participantType) {
        this.participantType = participantType;
    }

    /**
     * Retrieves the current state of the conference participant (for example: listener, normal, or broadcaster).
     *
     * @return the current type for the local session participant.
     */
    public ConferenceParticipantType getConferenceParticipantType() {
        return participantType;
    }

    /**
     * Informs if the currently logged in participant is in the broadcaster mode. This method is useful for webinars.
     *
     * @return informs if there is a broadcaster in the current session.
     */
    public boolean isBroadcaster() {
        return ConferenceParticipantType.BROADCASTER.equals(participantType);
    }

    /**
     * Informs if the currently logged in participant is in a listening mode.
     *
     * @return informs if there is a listener in the current session.
     */
    public boolean isListener() {
        return ConferenceParticipantType.LISTENER.equals(participantType);
    }

    /**
     * Set the screen share's state
     *
     * @param enable
     */
    @NoDocumentation
    public void setScreenShareOn(boolean enable) {
        mOwnScreenShareStarted = enable;
    }

    /**
     * Checks if the screen is currently shared from a perspective of a screen share lifecycle.
     *
     * @return the screen share state.
     */
    public boolean isScreenShareOn() {
        return mOwnScreenShareStarted;
    }

    /**
     * Transforms, from a SDK point of view, a list of participants from the REST Api into Conference's updated participant
     *
     * @param participants
     */
    @NoDocumentation
    public void participantsToConferenceParticipants(List<RestParticipant> participants) {
        if (null != participants) {
            Map<String, String> cache = getParticipantIdsCached();
            for (RestParticipant participant : participants) {
                cache.put(participant.getExternalId(), participant.getExternalId());
            }

            Conference conference = getConference();
            conference.updateRestParticipants(participants);
        }
    }

    /**
     * Set or unset the telecom mode into this conference
     *
     * @param telecomMode the new telecommunication mode
     */
    @NoDocumentation
    public void setTelecomMode(boolean telecomMode) {
        this.telecomMode = telecomMode;
    }

    /**
     * Checks if the conference is in the telecom mode.
     *
     * @return a boolean indicating if the conference is in the telecom mode.
     */
    @NoDocumentation
    public boolean isTelecomMode() {
        return telecomMode;
    }

    /**
     * Gets the state of the local video that is displayed during the conference. The state changes as in the following cycle: stopped ▸ starting ▸ started ▸ stopping ▸ stopped.
     *
     * @return the current non-null enum value.
     */
    @NonNull
    public VideoState getVideoState() {
        return videoState;
    }

    @NoDocumentation
    void clean() {
        mPositions.clear();
        mConference.getParticipants().clear();
    }

    @NoDocumentation
    public List<Participant> getParticipants() {
        return getConference().getParticipants();
    }

    @Override
    public String toString() {
        return "ConferenceInformation{" +
                "Conference='" + mConference + '\'' +
                '}';
    }

    void setVideoState(@NonNull VideoState videoState) {
        this.videoState = videoState;
    }
}
