package com.voxeet.sdk.services;

import android.support.annotation.NonNull;

import com.voxeet.promise.Promise;
import com.voxeet.promise.solve.Solver;
import com.voxeet.sdk.json.VideoPresentationPaused;
import com.voxeet.sdk.json.VideoPresentationPlay;
import com.voxeet.sdk.json.VideoPresentationSeek;
import com.voxeet.sdk.json.VideoPresentationStarted;
import com.voxeet.sdk.json.VideoPresentationStopped;
import com.voxeet.sdk.network.endpoints.IRestApiVideoPresentation;
import com.voxeet.sdk.services.abstracts.AbstractPresentationService;
import com.voxeet.sdk.services.presentation.PresentationState;
import com.voxeet.sdk.services.presentation.video.VideoPresentation;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;

/**
 * The VideoPresentationService allows an application to share a hosted video file with the participants, and synchronously control the playback for all participants. The video file location, specified by the `url` parameter of the [start](/documentation/sdk/reference/android/videopresentation#start) method, needs to be accessible by all participants. The MPEG-4 Part 14 or MP4 file formats are recommended for the hosted video file.
 *
 * **Typical application workflow for the presenter:**
 *
 * **1.** The application prepares the file and uploads it to the network location that is accessible by all participants through the `url`.
 *
 * **2.** The application calls the [start](/documentation/sdk/reference/android/videopresentation#start) method to supply other conference participants with the `url` as a parameter. This method starts sharing the video file during the conference.
 *
 * **2.** The application can call the [pause](/documentation/sdk/reference/android/videopresentation#pause) method to pause the video and [play](/documentation/sdk/reference/android/videopresentation#play) to resume it.
 *
 * **3.** While sharing the video with other conference participants, the application coordinates the timestamp in the local and presented video. It uses the [seek](/documentation/sdk/reference/android/videopresentation#seek) method to inform other participants about the current video position (timestamp).
 *
 * **4.** The application calls the [stop](/documentation/sdk/reference/android/videopresentation#stop) method to end the video presentation.
 *
 *
 * **Typical application workflow for other participants:**
 *
 * **1.** The application receives the [VideoPresentationStarted](/documentation/sdk/reference/android/videopresentation#videopresentationstarted) event informing that the presenter shares the video with other conference participants and it extracts the `url` of the video.
 *
 * **2.** The application receives the [VideoPresentationPaused](/documentation/sdk/reference/android/videopresentation#videopresentationpaused) event informing that the video is paused. When it is resumed, the application receives the [VideoPresentationPlay](/documentation/sdk/reference/android/videopresentation#videopresentationplay) event.
 *
 * **3.** If the timestamp of the video changed, the application is informed about the current video position via [VideoPresentationSeek](/documentation/sdk/reference/android/videopresentation#videopresentationseek) event.
 *
 * **4.** When the presentation is ended, the application receives the [VideoPresentationStopped](/documentation/sdk/reference/android/videopresentation#videopresentationstopped) event.
 */
@Annotate
public class VideoPresentationService extends AbstractPresentationService<IRestApiVideoPresentation, VideoPresentation> {

    private List<Solver<VideoPresentation>> mCacheStartedSolvers;
    private List<Solver<VideoPresentation>> mCacheStoppedSolvers;
    private List<Solver<VideoPresentation>> mCachePausedSolvers;
    private List<Solver<VideoPresentation>> mCachePlaySolvers;
    private List<Solver<VideoPresentation>> mCacheSeekSolvers;


    /**
     * Instantiates a new VideoPresentation service.
     *
     * @param instance the parent instance
     */
    @NoDocumentation
    public VideoPresentationService(@NonNull SdkEnvironmentHolder instance) {
        super(instance, IRestApiVideoPresentation.class);

        mCachePausedSolvers = new ArrayList<>();
        mCacheStartedSolvers = new ArrayList<>();
        mCacheStoppedSolvers = new ArrayList<>();
        mCachePausedSolvers = new ArrayList<>();
        mCachePlaySolvers = new ArrayList<>();
        mCacheSeekSolvers = new ArrayList<>();
        registerEventBus();
    }

    @NoDocumentation
    void onInternalServiceEvent(VideoPresentationStarted event) {
        VideoPresentation information = getPresentationInformation(event.key);
        information.url = event.url;
        presentations.add(information);

        information.state = PresentationState.STARTED;
        information.lastSeekTimestamp = event.timestamp;
        getEventBus().post(event);
        tryUnlock(information, mCacheStartedSolvers);
    }

    @NoDocumentation
    void onInternalServiceEvent(VideoPresentationStopped event) {
        VideoPresentation information = getPresentationInformation(event.key);
        information.state = PresentationState.STOP;
        presentations.remove(information);

        getEventBus().post(event);
        tryUnlock(information, mCacheStoppedSolvers);
    }

    @NoDocumentation
    void onInternalServiceEvent(VideoPresentationPaused event) {
        VideoPresentation information = getPresentationInformation(event.key);
        information.state = PresentationState.PAUSED;
        information.lastSeekTimestamp = event.timestamp;

        getEventBus().post(event);
        tryUnlock(information, mCachePausedSolvers);
    }

    @NoDocumentation
    void onInternalServiceEvent(VideoPresentationPlay event) {
        VideoPresentation information = getPresentationInformation(event.key);
        information.state = PresentationState.PLAY;
        information.lastSeekTimestamp = event.timestamp;

        getEventBus().post(event);
        tryUnlock(information, mCachePlaySolvers);
    }

    @NoDocumentation
    void onInternalServiceEvent(VideoPresentationSeek event) {
        VideoPresentation information = getPresentationInformation(event.key);
        information.state = PresentationState.SEEK;
        information.lastSeekTimestamp = event.timestamp;

        getEventBus().post(event);
        tryUnlock(information, mCacheSeekSolvers);
    }

    /**
     * Starts sharing the video with other conference participants.
     *
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<VideoPresentation> start(final String url) {
        return new Promise<>(solver -> {
            IRestApiVideoPresentation.VideoPresentationUrl holder = new IRestApiVideoPresentation.VideoPresentationUrl(url);
            consumeInternalCall(solver, mCacheStartedSolvers,
                    internalCall(getService().startVideoPresentation(getConferenceId(), holder)));
        });
    }

    /**
     * Stops sharing the video with other conference participants.
     *
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<VideoPresentation> stop() {
        return new Promise<>(solver -> consumeInternalCall(solver, mCacheStoppedSolvers,
                internalCall(getService().stopVideoPresentation(getConferenceId()))));
    }

    /**
     * Resumes the video presentation.
     *
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<VideoPresentation> play() {
        return new Promise<>(solver -> consumeInternalCall(solver, mCachePlaySolvers,
                internalCall(getService().playVideoPresentation(getConferenceId()))));
    }

    /**
     * Pauses the video presentation.
     *
     * @param timestamp the timestamp the video paused at
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<VideoPresentation> pause(final long timestamp) {
        return new Promise<>(solver -> {
            IRestApiVideoPresentation.VideoPresentationSeek body_sent = new IRestApiVideoPresentation.VideoPresentationSeek(timestamp);

            consumeInternalCall(solver, mCachePausedSolvers,
                    internalCall(getService().pauseVideoPresentation(getConferenceId(), body_sent)));
        });
    }

    /**
     * Informs other conference participants about the current video position (timestamp).
     *
     * @param timestamp the new timestamp
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<VideoPresentation> seek(final long timestamp) {
        return new Promise<>(solver -> {

            IRestApiVideoPresentation.VideoPresentationSeek body_sent = new IRestApiVideoPresentation.VideoPresentationSeek(timestamp);

            consumeInternalCall(solver, mCacheSeekSolvers,
                    internalCall(getService().seekVideoPresentation(getConferenceId(), body_sent)));
        });
    }

    /**
     * Gets the description of the presentation due to possible various Activities states.
     *
     * @return the clone of the holder of the key, URL, and the state. A null value if the video presentation does not exist at the moment.
     */
    @Nullable
    public VideoPresentation getCurrentPresentation() {
        for (VideoPresentation information : presentations) {
            if (!PresentationState.STOP.equals(information.state)) return information.clone();
        }
        return null;
    }

    @NonNull
    private VideoPresentation getPresentationInformation(@NonNull String key) {
        for (VideoPresentation information : presentations) {
            if (key.equals(information.key)) return information;
        }
        return new VideoPresentation(key, "");
    }

}