package com.voxeet.sdk.services;

import android.content.Context;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.promise.Promise;
import com.voxeet.sdk.VoxeetSdk;
import com.voxeet.sdk.json.ParticipantInfo;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.push.center.NotificationCenterFactory;
import com.voxeet.sdk.push.center.invitation.IIncomingInvitationListener;
import com.voxeet.sdk.push.center.invitation.InvitationBundle;
import com.voxeet.sdk.push.center.management.EnforcedNotificationMode;
import com.voxeet.sdk.push.center.management.NotificationMode;
import com.voxeet.sdk.push.center.management.VersionFilter;
import com.voxeet.sdk.push.utils.Annotate;
import com.voxeet.sdk.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.services.notification.INotificationTokenProvider;
import com.voxeet.sdk.services.notification.NotificationTokenHolderFactory;
import com.voxeet.sdk.utils.NoDocumentation;

import java.util.List;

/**
 * The NotificationService allows the application to set the desired way of notifying about conference invitations.
 *
 * **Typical application workflow for the inviter:**
 *
 * - The application calls the [invite](/documentation/sdk/reference/android/notification#invite) method to invite specific participants to a conference.
 *
 *
 * **Typical application workflow for the receiver:**
 *
 * **1.** The application calls the [register](/documentation/sdk/reference/android/notification#register) method to register the effective notification mode.
 *
 * **2.** The application calls the [register](/documentation/sdk/reference/android/notification#register-1) method to customize notifications. This method registers filters which reject specific invitations.
 *
 * **3.** The application calls the [setNotificationTokenProvider](/documentation/sdk/reference/android/notification#setnotificationtokenprovider) method to change the notification provider. The [getNotificationTokenProvider](/documentation/sdk/reference/android/notification#getnotificationtokenprovider) method checks if the [INotificationTokeProvider](/documentation/sdk/reference/android/models/inotificationtokenprovider) model is registered to the SDK.
 *
 * **4.** Optionally, the application can change the notification management mode into the [enforcedNotificationMode](/documentation/sdk/reference/android/models/enforcednotificationmode) by calling the [setEnforcedNotificationMode](/documentation/sdk/reference/android/notification#setenforcednotificationmode) method.
 *
 * **5.** The application calls the [InvitationBundle](/documentation/sdk/reference/android/models/invitationbundle#invitationbundle) constructor from the [InvitationBundle](/documentation/sdk/reference/android/models/invitationbundle) model to transform notifications.
 *
 * **6.** The application calls the [onInvitationReceived](/documentation/sdk/reference/android/notification#oninvitationreceived) or [onInvitationCanceledReceived](/documentation/sdk/reference/android/notification#oninvitationcanceledreceived) method to show the proper notification.
 */
@Annotate
public class NotificationService extends AbstractVoxeetService {

    @NoDocumentation
    public NotificationService(@NonNull SdkEnvironmentHolder instance) {
        super(instance, ServiceProviderHolder.DEFAULT);
    }

    /**
     * Accesses the [INotificationTokeProvider](/documentation/sdk/reference/android/models/inotificationtokenprovider) model previously registered to the SDK. This method can be used by the SDK or developers to check and make sure if this model is properly set.
     *
     * @return if previously set, the instance of the [INotificationTokeProvider](/documentation/sdk/reference/android/models/inotificationtokenprovider) known by the NotificationService or null value.
     */
    @Nullable
    public INotificationTokenProvider getNotificationTokenProvider() {
        return NotificationTokenHolderFactory.provider;
    }

    /**
     * Changes the provider (registered into the SDK) of push notifications. It is set to the default value when the uxkit-firebase plugin is used.
     *
     * @param provider the provider or a null to replace by none
     * @return the current instance.
     */
    @NonNull
    public NotificationService setNotificationTokenProvider(@Nullable INotificationTokenProvider provider) {
        NotificationTokenHolderFactory.provider = provider;
        return this;
    }

    /**
     * Changes the mode of notification management into the [enforcedNotificationMode](/documentation/sdk/reference/android/models/enforcednotificationmode).
     *
     * @param enforcedNotificationMode the new mode
     * @return the current instance.
     */
    @NonNull
    public NotificationService setEnforcedNotificationMode(@NonNull EnforcedNotificationMode enforcedNotificationMode) {
        NotificationCenterFactory.instance.setEnforcedNotificationMode(enforcedNotificationMode);
        return this;
    }

    /**
     * Registers the [IIncomingInvitationListener](/documentation/sdk/reference/android/models/iincominginvitationlistener) model dedicated to the [NotificationMode](/documentation/sdk/reference/android/models/notificationmode). The registered [IIncomingInvitationListener](/documentation/sdk/reference/android/models/iincominginvitationlistener) replaces the current model if the [NotificationMode](/documentation/sdk/reference/android/models/notificationmode) has already one attached listener.
     *
     * @param notificationMode   the valid NotificationMode that will receive the the provider
     * @param invitationProvider the instance to register
     * @return the current instance.
     */
    @NonNull
    public NotificationService register(@NonNull NotificationMode notificationMode, @Nullable IIncomingInvitationListener invitationProvider) {
        NotificationCenterFactory.instance.register(notificationMode, invitationProvider);
        return this;
    }

    /**
     * Registers the [VersionFilter](/documentation/sdk/reference/android/models/versionfilter) model for the specified [NotificationMode](/documentation/sdk/reference/android/models/notificationmode).
     *
     * @param notificationMode the valid NotificationMode that will receive the possible filter
     * @param filter           the filter to apply when needed
     * @return the current instance.
     */
    @NonNull
    public NotificationService register(@NonNull NotificationMode notificationMode, @NonNull VersionFilter filter) {
        NotificationCenterFactory.instance.register(notificationMode, filter);
        return this;
    }

    /**
     * Method called by the SDK when a push notification concerning an invitation is received by the WebSocket or a plugin.
     * <p>
     * It is possible to use this method when the SDK works on a system that is not Firebase compatible, or developers need to use a different library. If external push notifications are managed elsewhere, then developers are only responsible for transforming the Bundle or data to a valid [InvitationBundle](/documentation/sdk/reference/android/models/invitationbundle) model.
     *
     * @param context          a valid context to use.
     * @param invitationBundle a non null InvitationBundle instance
     */
    public void onInvitationReceived(@NonNull Context context,
                                     @NonNull InvitationBundle invitationBundle) {
        NotificationCenterFactory.instance.onInvitationReceived(context,
                invitationBundle.asMap(),
                Build.MANUFACTURER,
                Build.VERSION.SDK_INT);
    }

    /**
     * Method called by the SDK when a push notification concerning a cancelled invitation is received by the WebSocket or a plugin.
     *
     * It is possible to use this method when the SDK works on a system that is not Firebase compatible, or developers need to use a different library. If external push notifications are managed elsewhere, then developers are only responsible for transforming the Bundle or data to a valid [InvitationBundle](/documentation/sdk/reference/android/models/invitationbundle) model.
     *
     * @param context          a valid context to use.
     * @param invitationBundle a non null InvitationBundle instance
     * @return the corresponding conference id to be used on higher levels
     */
    @NonNull
    public String onInvitationCanceledReceived(@NonNull Context context,
                                               @NonNull InvitationBundle invitationBundle) {
        return NotificationCenterFactory.instance.onInvitationCanceledReceived(context,
                invitationBundle.asMap(),
                Build.MANUFACTURER,
                Build.VERSION.SDK_INT);
    }

    /**
     * Invites a list of users for a specific conference.
     *
     * @param conferenceId the conference ID of the conference
     * @param participantOptions    a non null list of information about the participants to invite into the specified conference
     * @return a non null promise to resolve.
     */
    @NonNull
    public Promise<List<Participant>> invite(final String conferenceId, final List<ParticipantInfo> participantOptions) {
        return VoxeetSdk.conference().invite(conferenceId, participantOptions);
    }

}
