package com.voxeet.sdk.services;

import android.content.Context;
import android.os.Build;
import android.support.annotation.MainThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.crypto.AbstractNativeMediaCryptoCallback;
import com.voxeet.audio2.devices.MediaDevice;
import com.voxeet.audio2.devices.description.DeviceType;
import com.voxeet.promise.Promise;
import com.voxeet.promise.solve.ThenPromise;
import com.voxeet.sdk.VoxeetSdk;
import com.voxeet.sdk.events.error.CameraSwitchErrorEvent;
import com.voxeet.sdk.events.sdk.CameraSwitchSuccessEvent;
import com.voxeet.sdk.media.MediaSDK;
import com.voxeet.sdk.media.camera.CameraContext;
import com.voxeet.sdk.media.camera.CameraEnumeratorLollipopWrapper;
import com.voxeet.sdk.media.camera.CameraEnumeratorPreLollipopWrapper;
import com.voxeet.sdk.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.services.media.VideoSinkHolder;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.Opt;
import com.voxeet.sdk.utils.Validate;
import com.voxeet.sdk.views.ViewFactory;

import org.webrtc.CameraVideoCapturer;
import org.webrtc.EglBase;
import org.webrtc.VideoSink;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * The MediaDeviceService allows the application to manage the devices that are used during the conference.
 * <p>
 * **Typical application workflow:**
 * <p>
 * **1.** The application calls the [getEglContext](/documentation/sdk/reference/android/mediadevice#geteglcontext) method to get the WebRTC EglBaseContext that enables data rendering.
 * <p>
 * **2.** The application sets the MediaCryptoCallback through the [setCryptoCallback](/documentation/sdk/reference/android/mediadevice#setcryptocallback) method to internally register the component for encrypting and decrypting WebRTC communication packets.
 * <p>
 * **3.** Then, the application is able to attach the stream to the specific receiver using the [attachMediaStream](/documentation/sdk/reference/android/mediadevice#attachmediastream) method. It can detach the stream by calling the [unAttachMediaStream](/documentation/sdk/reference/android/mediadevice#unattachmediastream) method.
 * <p>
 * **4.** The application can also call the [isAudio3DEnabled](/documentation/sdk/reference/android/mediadevice#isaudio3denabled) method to check the state of 3D audio management. It can turn it on and off through the [setAudio3DEnabled](/documentation/sdk/reference/android/mediadevice#setaudio3denabled) method.
 * <p>
 * **5.** The application can also check information about used cameras through the [getCameraContext](/documentation/sdk/reference/android/mediadevice#getcameracontext) method. The [switchCamera](/documentation/sdk/reference/android/mediadevice#switchcamera) method enables the app to change the current camera to another one.
 */
@Annotate
public class MediaDeviceService extends AbstractVoxeetService {

    private CameraContext enumerator;
    private AbstractNativeMediaCryptoCallback mediaCryptoCallback;
    private MediaSDK media;
    private boolean isAudio3DEnabled;

    private CopyOnWriteArrayList<VideoSinkHolder> videoSinkHolders;

    @NoDocumentation
    public MediaDeviceService(@NonNull SdkEnvironmentHolder instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        videoSinkHolders = new CopyOnWriteArrayList<>();

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            enumerator = new CameraEnumeratorLollipopWrapper(instance.voxeetSdk.getApplicationContext());
        } else {
            enumerator = new CameraEnumeratorPreLollipopWrapper(instance.voxeetSdk.getApplicationContext());
        }

        enumerator.setDefaultCameraFront(true);

        isAudio3DEnabled = false;

        ViewFactory.set(new ViewFactory.MediaProvider() {
            @Override
            public EglBase.Context getEglContext() {
                return MediaDeviceService.this.getEglContext();
            }

            @Override
            public boolean hasMedia() {
                return MediaDeviceService.this.hasMedia();
            }

            @Override
            public boolean attachMediaStream(MediaStream stream, VideoSink videoSink) {
                return MediaDeviceService.this.attachMediaStream(stream, videoSink);
            }

            @Override
            public void unAttachMediaStream(VideoSink videoSink) {
                MediaDeviceService.this.unAttachMediaStream(videoSink);
            }
        });
    }

    /**
     * Checks the availability of Media instances. Media are only valid while a conference lasts.
     *
     * @return the availability indicator.
     */
    @NoDocumentation
    public boolean hasMedia() {
        return null != media;
    }

    /**
     * Gets the current instance of the Media or a null value if it does not exist in a Conference (or it is not possible to create one).
     *
     * @return the instance.
     */
    @NoDocumentation
    @Nullable
    public MediaSDK getMedia() {
        return media;
    }

    /**
     * Gets the WebRTC `EglBaseContext`.
     * <p>
     * It is used by SDK elements but it is also publically accessible.
     *
     * @return the current `EglBaseContext` or a null value if media are not created.
     */
    @Nullable
    public EglBase.Context getEglContext() {
        return Opt.of(media).then(MediaEngine::getEglBase).then(EglBase::getEglBaseContext).orNull();
    }

    /**
     * Sets the `MediaCryptoCallback` to use it before joining the conference.
     * <p>
     * Note: This method is not released automatically, it must be unset and set again before conferences.
     *
     * @param callback The callback. The null value unsets the callback.
     */
    public void setCryptoCallback(@Nullable AbstractNativeMediaCryptoCallback callback) {
        this.mediaCryptoCallback = callback;
    }

    /**
     * Attaches a stream to the given receiver.
     * <p>
     * Note: If a stream is already attached to the receiver, this method will detach it.
     *
     * @param stream    The valid media stream to attach.
     * @param videoSink The videosink that will receive updates.
     * @return
     */
    @MainThread
    public boolean attachMediaStream(@NonNull MediaStream stream, @NonNull VideoSink videoSink) {
        MediaEngine media = this.media;
        if (null == media) return false;

        VideoSinkHolder holder = getOrCreate(videoSink);
        if (null == holder) return false;

        unAttachMediaStream(videoSink);
        holder.attach(media, stream);

        return true;
    }

    /**
     * Detaches the stream from the given videoSink.
     *
     * @param videoSink The receiver instance.
     * @return the release indicator.
     */
    @MainThread
    public boolean unAttachMediaStream(@NonNull VideoSink videoSink) {
        VideoSinkHolder holder = getOrCreate(videoSink);
        if (null == holder) return false;

        holder.unattach(media);
        return true;
    }

    /**
     * Checks the state of the 3D audio management.
     *
     * @return the 3D management indicator.
     */
    public boolean isAudio3DEnabled() {
        return isAudio3DEnabled;
    }

    /**
     * Turns on and off the 3D Audio management. If conferences or media are not initialized, the state will not change for the next calls.
     * The SDK keeps the value, it must be recalled with the new value to be updated.
     *
     * @param enable The new state for 3D audio processing.
     * @return true, a false value means that media are not initialized.
     */
    public boolean setAudio3DEnabled(boolean enable) {
        isAudio3DEnabled = enable;
        if (null != media) return media.setAudio3D(enable);
        return false;
    }

    /**
     * Switches the current camera to another available camera that is connected to the device. On Android, the second camera exists by default but on other devices, there may be none, one, two, or even more cameras.
     * <p>
     * It updates the camera provider's information with the new camera type: front or back.
     *
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Boolean> switchCamera() {
        return new Promise<>((resolve, reject) -> {
            Validate.notNull(getMedia(), "media");

            getMedia().switchCamera(new CameraVideoCapturer.CameraSwitchHandler() {
                @Override
                public void onCameraSwitchDone(boolean isFrontCamera) {
                    getCameraContext().setDefaultCameraFront(isFrontCamera);

                    getEventBus().post(new CameraSwitchSuccessEvent(isFrontCamera));
                    resolve.call(true);
                }

                @Override
                public void onCameraSwitchError(String errorDescription) {
                    getEventBus().post(new CameraSwitchErrorEvent(errorDescription));
                    resolve.call(false);
                }
            });
        });
    }

    /**
     * Retrieves the instance of the camera information.
     *
     * @return the `CameraContext`
     */
    @NonNull
    public CameraContext getCameraContext() {
        return enumerator;
    }

    @MainThread
    @Nullable
    private VideoSinkHolder getOrCreate(@Nullable VideoSink videoSink) {
        if (null == videoSink) return null;
        for (VideoSinkHolder holder : videoSinkHolders) {
            if (holder.equals(videoSink)) return holder;
        }

        VideoSinkHolder holder = new VideoSinkHolder(videoSink);
        videoSinkHolders.add(holder);
        return holder;
    }

    @NoDocumentation
    void releaseMedia() {
        for (VideoSinkHolder holder : videoSinkHolders) {
            if (holder.hasVideoSink()) {
                holder.clear();
            }
        }
        videoSinkHolders.clear();

        if (null != media) {
            VoxeetSdk.audio().enumerateDevices().then((ThenPromise<List<MediaDevice>, Boolean>) devices -> {
                MediaDevice mediaDevice = null;
                for (MediaDevice device :
                        Opt.of(devices).or(new ArrayList<>())) {
                    if (DeviceType.NORMAL_MEDIA.equals(device.deviceType())) {
                        mediaDevice = device;
                    }
                }
                if (null != mediaDevice) return VoxeetSdk.audio().connect(mediaDevice);
                return Promise.resolve(true);
            }).then(result -> {

            }).error(Throwable::printStackTrace);

            media.stop();
        }

        media = null;
    }

    @NoDocumentation
    void createMedia(Context context, String participantId, MediaEngine.StreamListener mediaStreamListener,
                     CameraVideoCapturer.CameraEventsHandler cameraEventsHandler,
                     boolean videoOn, boolean useMic) throws MediaEngineException {
        media = new MediaSDK(context, participantId, mediaStreamListener, cameraEventsHandler, videoOn, useMic, mediaCryptoCallback);
        media.setAudio3D(isAudio3DEnabled);
    }

}
