package com.voxeet.sdk.services;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.MediaStreamType;
import com.voxeet.android.media.SdpCandidate;
import com.voxeet.promise.Promise;
import com.voxeet.promise.solve.ErrorPromise;
import com.voxeet.promise.solve.ThenPromise;
import com.voxeet.promise.solve.ThenVoid;
import com.voxeet.sdk.VoxeetSdk;
import com.voxeet.sdk.authent.models.DeviceType;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.error.ParticipantAddedErrorEvent;
import com.voxeet.sdk.events.error.PermissionRefusedEvent;
import com.voxeet.sdk.events.promises.NotInConferenceException;
import com.voxeet.sdk.events.promises.ParticipantAddedErrorEventException;
import com.voxeet.sdk.events.promises.ServerErrorException;
import com.voxeet.sdk.events.restapi.ConferenceStatusResult;
import com.voxeet.sdk.events.restapi.ResumeConference;
import com.voxeet.sdk.events.sdk.ConferenceHistoryResult;
import com.voxeet.sdk.events.sdk.ConferenceParticipantQualityUpdatedEvent;
import com.voxeet.sdk.events.sdk.ConferenceParticipantsInvitedResult;
import com.voxeet.sdk.events.sdk.ConferenceStatusUpdatedEvent;
import com.voxeet.sdk.events.sdk.IncomingCallEvent;
import com.voxeet.sdk.events.sdk.SocketStateChangeEvent;
import com.voxeet.sdk.events.v2.ParticipantAddedEvent;
import com.voxeet.sdk.events.v2.ParticipantUpdatedEvent;
import com.voxeet.sdk.events.v2.StreamAddedEvent;
import com.voxeet.sdk.events.v2.StreamRemovedEvent;
import com.voxeet.sdk.events.v2.StreamUpdatedEvent;
import com.voxeet.sdk.events.websocket.RenegociationEndedEvent;
import com.voxeet.sdk.exceptions.ExceptionManager;
import com.voxeet.sdk.json.ConferenceDestroyedPush;
import com.voxeet.sdk.json.ConferenceEnded;
import com.voxeet.sdk.json.InvitationReceivedEvent;
import com.voxeet.sdk.json.JoinParameters;
import com.voxeet.sdk.json.OfferCreated;
import com.voxeet.sdk.json.OwnConferenceCreated;
import com.voxeet.sdk.json.OwnConferenceParticipantSwitch;
import com.voxeet.sdk.json.ParticipantAdded;
import com.voxeet.sdk.json.ParticipantInfo;
import com.voxeet.sdk.json.ParticipantUpdated;
import com.voxeet.sdk.json.QualityUpdated;
import com.voxeet.sdk.json.internal.MetadataHolder;
import com.voxeet.sdk.json.internal.ParamsHolder;
import com.voxeet.sdk.media.MediaSDK;
import com.voxeet.sdk.media.audio.AudioRoute;
import com.voxeet.sdk.media.audio.ParticipantPosition;
import com.voxeet.sdk.media.camera.CameraContext;
import com.voxeet.sdk.media.peer.SdpDescription;
import com.voxeet.sdk.media.peer.SdpMessage;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.models.v1.CandidatesPush;
import com.voxeet.sdk.models.v1.ConferenceParticipantStatus;
import com.voxeet.sdk.models.v1.ConferenceType;
import com.voxeet.sdk.models.v1.ConferenceUser;
import com.voxeet.sdk.models.v1.CreateConferenceResult;
import com.voxeet.sdk.models.v1.Invitation;
import com.voxeet.sdk.models.v1.OfferCandidate;
import com.voxeet.sdk.models.v1.OfferDescription;
import com.voxeet.sdk.models.v1.RestParticipant;
import com.voxeet.sdk.models.v1.SdkParticipant;
import com.voxeet.sdk.models.v2.ServerErrorOrigin;
import com.voxeet.sdk.network.endpoints.IRestApiConferenceAccess;
import com.voxeet.sdk.network.endpoints.IRestApiMedia;
import com.voxeet.sdk.network.endpoints.IRestApiOutConference;
import com.voxeet.sdk.network.endpoints.IRestApiSimulcast;
import com.voxeet.sdk.network.endpoints.IRestApiVideo;
import com.voxeet.sdk.preferences.VoxeetPreferences;
import com.voxeet.sdk.push.center.invitation.InvitationBundle;
import com.voxeet.sdk.push.center.management.Constants;
import com.voxeet.sdk.push.center.management.PushConferenceDestroyed;
import com.voxeet.sdk.services.authenticate.WebSocketState;
import com.voxeet.sdk.services.builders.ConferenceCreateOptions;
import com.voxeet.sdk.services.builders.ConferenceJoinInformation;
import com.voxeet.sdk.services.conference.ConferenceConfigurations;
import com.voxeet.sdk.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.services.conference.information.ConferenceInformationHolder;
import com.voxeet.sdk.services.conference.information.ConferenceParticipantType;
import com.voxeet.sdk.services.conference.information.ConferenceStatus;
import com.voxeet.sdk.services.conference.information.LocalConferenceType;
import com.voxeet.sdk.services.conference.promises.CreateConferencePromiseable;
import com.voxeet.sdk.services.conference.promises.DeclinePromise;
import com.voxeet.sdk.services.conference.promises.GetConferenceHistoryPromiseable;
import com.voxeet.sdk.services.conference.promises.GetConferenceStatus;
import com.voxeet.sdk.services.conference.promises.GetConferenceStatusPromiseable;
import com.voxeet.sdk.services.conference.promises.InvitePromise;
import com.voxeet.sdk.services.conference.promises.JoinPromise;
import com.voxeet.sdk.services.conference.promises.LeavePromise;
import com.voxeet.sdk.services.conference.promises.ReplayPromise;
import com.voxeet.sdk.services.conference.promises.SimulcastPromiseable;
import com.voxeet.sdk.services.conference.promises.StartVideoPromise;
import com.voxeet.sdk.services.conference.promises.StopVideoPromise;
import com.voxeet.sdk.services.conference.promises.SubscribeConferenceEventPromiseable;
import com.voxeet.sdk.services.conference.promises.SubscribeForCallStartPromiseable;
import com.voxeet.sdk.services.conference.promises.UnsubscribeConferenceEventPromiseable;
import com.voxeet.sdk.services.conference.promises.UnsubscribeForCallStartPromiseable;
import com.voxeet.sdk.services.media.VideoState;
import com.voxeet.sdk.services.simulcast.ParticipantQuality;
import com.voxeet.sdk.utils.AndroidManifest;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.ConferenceUtils;
import com.voxeet.sdk.utils.HttpHelper;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.Opt;
import com.voxeet.sdk.utils.ParticipantUtils;
import com.voxeet.sdk.utils.PeerInformation;
import com.voxeet.sdk.utils.Validate;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.CameraVideoCapturer;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.locks.ReentrantLock;

import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Response;

/**
 * The ConferenceService allows the application to manage the conference life cycle and interact with the conference.
 * <p>
 * **Typical application APIs workflow:**
 * <p>
 * **1.** The application calls the [create](/documentation/sdk/reference/android/conference#create) method to create a conference.
 * <p>
 * **2.** The application can choose to either [join](/documentation/sdk/reference/android/conference#join), [replay](/documentation/sdk/reference/android/conference#replay), [listen](/documentation/sdk/reference/android/conference#listen), or [broadcast](/documentation/sdk/reference/android/conference#broadcast) the conference. Participants that do not wish to participate in the conference can [decline](/documentation/sdk/reference/android/conference#decline) the invitation.
 * <p>
 * **3.** Then, the application can start the video stream for the participant by using the [startVideo](/documentation/sdk/reference/android/conference#startvideo) method or stop the stream by calling the [stopVideo](/documentation/sdk/reference/android/conference#stopvideo).
 * <p>
 * **4.** The application can check the mute state using the [isMuted](/documentation/sdk/reference/android/conference#ismuted) method or [isParticipantSpeaking](/documentation/sdk/reference/android/conference#isparticipantspeaking). It can also [mute](/documentation/sdk/reference/android/conference#mute) participants.
 * <p>
 * **5.** The application can check information about the conference and interact with it by using the following methods: [getConferenceType](/documentation/sdk/reference/android/conference#getconferencetype), [getConference](/documentation/sdk/reference/android/conference#getconference), [getConferenceStatus](/documentation/sdk/reference/android/conference#getconferencestatus), [getCurrentConference](/documentation/sdk/reference/android/conference#getcurrentconference), and [getConferenceInformation](/documentation/sdk/reference/android/conference#getconferenceinformation).
 * <p>
 * **6.** The application can check information about conference participants by using the following methods: [findParticipantById](/documentation/sdk/reference/android/conference#findparticipantbyid), [isInConference](/documentation/sdk/reference/android/conference#isinconference), [getLevel](/documentation/sdk/reference/android/conference#getlevel), [getInvitedParticipants](/documentation/sdk/reference/android/conference#getinvitedparticipants), and [hasParticipants](/documentation/sdk/reference/android/conference#hasparticipants).
 * <p>
 * **The application can interact with the service through these events:**
 * <p>
 * - The application receives the list of invited participants though the [ConferenceParticipantsInvitedResult](/documentation/sdk/reference/android/conference#conferenceparticipantsinvitedresult) event.
 * <p>
 * - The [ConferenceDestroyedPush](/documentation/sdk/reference/android/conference#conferencedestroyedpush) event informs the application that the conference is destroyed. Even though it can be notified about historical conference details through the [ConferenceHistoryResult](/documentation/sdk/reference/android/conference#conferencehistoryresult) event.
 * <p>
 * - The [ConferenceTimeoutNoParticipantsJoinedEvent](/documentation/sdk/reference/android/conference#conferencetimeoutparticipantjoinedevent) informs that during the maximum awaiting time nobody joined the current participant at a conference.
 * <p>
 * - The [IncomingCallEvent](/documentation/sdk/reference/android/conference#incomingcallevent) informs the application that the incoming call is received.
 * <p>
 * - The [VideoStateEvent](/documentation/sdk/reference/android/conference#videostateevent) informs the application about the local video state.
 * <p>
 * - The application is notified about changes in participants' statuses by the [ParticipantAddedEvent](/documentation/sdk/reference/android/conference#participantaddedevent) and the [ParticipantUpdatedEvent](/documentation/sdk/reference/android/conference#getinvitedparticipants).
 * <p>
 * - The application is notified about changes in stream statuses by the [StreamAddedEvent](/documentation/sdk/reference/android/conference#streamaddedevent), [StreamUpdatedEvent](/documentation/sdk/reference/android/conference#streamupdatedevent), and the [StreamRemovedEvent](/documentation/sdk/reference/android/conference#streamremovedevent).
 * <p>
 * <p>
 * _Warning: Always implement the error for each called promise before you start resolving it._
 * You can also use the execute method but in case of exceptions, you will trigger errors.
 */
@Annotate
public class ConferenceService extends AbstractVoxeetService<IRestApiConferenceAccess> {

    private final static String TAG = ConferenceService.class.getSimpleName();
    private final IRestApiConferenceAccess conferenceAccess;
    private final IRestApiSimulcast simulcastAccess;
    private final IRestApiMedia mediaAccess;
    private final IRestApiOutConference outConferenceAccess;
    private final IRestApiVideo videoAccess;

    private EventBus mEventBus;
    private boolean mInConference = false;
    private ReentrantLock joinLock = new ReentrantLock();

    private String mConferenceId = null;
    private ConferenceInformationHolder mConferenceInformationHolder = new ConferenceInformationHolder();

    public final ConferenceConfigurations ConferenceConfigurations = new ConferenceConfigurations();

    private ExecutorService executorService = Executors.newCachedThreadPool();

    private Context mContext;

    @Nullable
    private TimeoutRunnable timeoutRunnable = null;

    private MediaEngine.StreamListener mediaStreamListener = new MediaEngine.StreamListener() {
        @Override
        public void onStreamAdded(@NonNull final String peer, @NonNull final MediaStream stream) {
            PeerInformation peerInformation = new PeerInformation(peer, MediaStreamType.Camera);
            onStreamAddedFromInformation(peerInformation, stream);
        }

        @Override
        public void onStreamUpdated(@NonNull final String peer, @NonNull final MediaStream stream) {
            PeerInformation peerInformation = new PeerInformation(peer, MediaStreamType.Camera);
            onStreamUpdatedFromInformation(peerInformation, stream);
        }

        @Override
        public void onStreamRemoved(@NonNull final String peer) {
            PeerInformation peerInformation = new PeerInformation(peer, MediaStreamType.Camera);
            onStreamRemovedFromInformation(peerInformation);
        }

        @Override
        public void onScreenStreamAdded(@NonNull final String peer, @NonNull final MediaStream stream) {
            PeerInformation peerInformation = new PeerInformation(peer, MediaStreamType.ScreenShare);
            onStreamAddedFromInformation(peerInformation, stream);
        }

        @Override
        public void onScreenStreamRemoved(@NonNull String peer) {
            PeerInformation peerInformation = new PeerInformation(peer, MediaStreamType.ScreenShare);
            onStreamRemovedFromInformation(peerInformation);
        }

        private void onStreamAddedFromInformation(@NonNull final PeerInformation peerInformation, @NonNull MediaStream stream) {
            String peer = peerInformation.peerId;
            Log.d("DUMP_INFORMATION", "onStreamAdded: stream for peer " + peer + " " + stream.getType());
            postOnMainThread(() -> {
                ConferenceInformation conferenceInformation = getCurrentConference();

                Participant participant = findParticipantById(peer);
                Log.d("DUMP_INFORMATION", "New stream for participant " + peer + " participant := " + participant);

                if (null != conferenceInformation && null == participant) {
                    Log.d(TAG, "run: WARNING obtained stream for participant which did not existed " + peer);
                    participant = new Participant(peer, null);
                    conferenceInformation.getConference().updateParticipant(participant);
                }

                if (null != conferenceInformation) {
                    if (!peer.equalsIgnoreCase(VoxeetSdk.session().getParticipantId()) && ConferenceConfigurations.TelecomWaitingForParticipantTimeout != -1) {
                        removeTimeoutCallbacks();
                    }

                    //insert or update -> but in the context, add
                    participant.streamsHandler().insertOrUpdate(stream);

                    //update the current conference state specificly from the participants point of view
                    updateConferenceFromParticipants();
                    mEventBus.post(new StreamAddedEvent(conferenceInformation.getConference(), participant, stream));
                } else {
                    Log.d(TAG, "run: unknown participant in stream added");
                }
            });
        }

        private void onStreamUpdatedFromInformation(@NonNull final PeerInformation peerInformation, @NonNull final MediaStream stream) {
            String peer = peerInformation.peerId;
            Log.d("DUMP_INFORMATION", "onStreamUpdated: screen for peer " + peer);
            postOnMainThread(() -> {
                ConferenceInformation conferenceInformation = getCurrentConference();
                Participant participant = findParticipantById(peer);

                if (null != conferenceInformation && null == participant) {
                    Log.d("DUMP_INFORMATION", "run: WARNING obtained stream for participant which did not existed " + peer);
                    participant = new Participant(peer, null);
                    conferenceInformation.getConference().updateParticipant(participant);
                }
                Log.d(TAG, "Updated stream for participant " + participant);

                if (null != conferenceInformation) {
                    //insert or update -> but in the context, add
                    participant.streamsHandler().insertOrUpdate(stream);

                    //update the current conference state specificly from the participants point of view
                    updateConferenceFromParticipants();
                    mEventBus.post(new StreamUpdatedEvent(conferenceInformation.getConference(), participant, stream));
                } else {
                    Log.d(TAG, "run: unknown participant in stream updated");
                }
            });
        }

        void onStreamRemovedFromInformation(@NonNull final PeerInformation peerInformation) {
            String peer = peerInformation.peerId;
            Log.d("DUMP_INFORMATION", "onStreamRemoved: OnStreamRemoved");
            postOnMainThread(() -> removeStreamForParticipant(peer, peerInformation.calculatedMediaStreamType));
        }

        @Override
        public void onShutdown() {

        }

        @Override
        public void onIceCandidateDiscovered(String peer, SdpCandidate[] candidates) {

            if (null == getConferenceId()) return;

            Call<ResponseBody> call = mediaAccess.candidates(getConferenceId(),
                    peer,
                    new CandidatesPush(candidates));

            HttpHelper.enqueue(call, (response, object, exception) -> {
                if (null != exception) {
                    exception.printStackTrace();
                } else {
                    Log.d(TAG, "onResponse: Candidates sent ");
                }
            });
        }
    };

    private void removeStreamForParticipant(@NonNull String peer, MediaStreamType mediaStreamType) {
        ConferenceInformation conferenceInformation = getCurrentConference();
        Participant participant = findParticipantById(peer);

        if (null != conferenceInformation && null == participant) {
            Log.d(TAG, "run: WARNING obtained stream for participant which did not existed " + peer);
            participant = new Participant(peer, null);
            conferenceInformation.getConference().updateParticipant(participant);
        }
        Log.d(TAG, "Remove stream " + mediaStreamType + " for participant " + participant);

        if (null != conferenceInformation) {
            //insert or update -> but in the context, add
            MediaStream stream = participant.streamsHandler().getFirst(mediaStreamType);
            participant.streamsHandler().remove(mediaStreamType);

            //update the current conference state specificly from the participants point of view
            updateConferenceFromParticipants();

            mEventBus.post(new StreamRemovedEvent(conferenceInformation.getConference(), participant, stream));
        } else {
            Log.d(TAG, "run: unknown participant in stream updated" + Arrays.toString(getParticipants().toArray()));
        }
    }

    private CameraVideoCapturer.CameraEventsHandler cameraEventsHandler = new CameraVideoCapturer.CameraEventsHandler() {
        @Override
        public void onCameraError(String s) {
            Log.d(TAG, "onCameraError: error...");

            stopVideo()
                    .then((ThenVoid<Boolean>) (result) -> Log.d(TAG, "onCall: stopped video after camera issue " + result))
                    .error(Throwable::printStackTrace);
        }

        @Override
        public void onCameraDisconnected() {
            Log.d(TAG, "onCameraDisconnected");
        }

        @Override
        public void onCameraFreezed(String s) {
            Log.d(TAG, "onCameraFreezed: " + s);
        }

        @Override
        public void onCameraOpening(String s) {
            Log.d(TAG, "onCameraOpening: " + s);
        }

        @Override
        public void onFirstFrameAvailable() {
            Log.d(TAG, "onFirstFrameAvailable");
        }

        @Override
        public void onCameraClosed() {
            Log.d(TAG, "onCameraClosed");
        }
    };

    @NoDocumentation
    public ConferenceService(@NonNull SdkEnvironmentHolder instance, long timeout) {
        super(instance, IRestApiConferenceAccess.class);

        conferenceAccess = instance.retrofit.create(IRestApiConferenceAccess.class);
        mediaAccess = instance.retrofit.create(IRestApiMedia.class);
        outConferenceAccess = instance.retrofit.create(IRestApiOutConference.class);
        videoAccess = instance.retrofit.create(IRestApiVideo.class);
        simulcastAccess = instance.retrofit.create(IRestApiSimulcast.class);

        ConferenceConfigurations.TelecomWaitingForParticipantTimeout = timeout;
        mEventBus = EventBus.getDefault();
        mContext = instance.voxeetSdk.getApplicationContext();

        /**
         * Can be used using the metadata in your app manifest :
         * <meta-data android:name="voxeet_automatic_telecom_conference_alias_prefix"
         * android:value="someprefix" />
         */
        ConferenceConfigurations.automaticTelecomModePrefix = AndroidManifest.readMetadata(mContext, "voxeet_automatic_telecom_conference_alias_prefix", null);


        registerEventBus();
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Mutes and unmutes the current participant.
     *
     * @param mute the new unmute/mute state
     * @return informs if the mute state was changed.
     */
    public boolean mute(boolean mute) {
        MediaEngine media = Opt.of(VoxeetSdk.mediaDevice()).then(MediaDeviceService::getMedia).orNull();

        if (null != media) {
            if (!mute && media.isMuted()) {
                if (!Validate.hasMicrophonePermissions(context)) {
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                    return false;
                } else {
                    media.unMute();
                }
            } else if (mute) {
                media.mute();
            }
        }
        return true;
    }

    /**
     * Gets the current muted state.
     *
     * @return the muted state. The false value indicates that the participant is not in the conference or the conference is muted. True value informs that the conference is valid and muted.
     */
    public boolean isMuted() {
        return Opt.of(VoxeetSdk.mediaDevice().getMedia()).then(MediaEngine::isMuted).or(true);
    }

    @NoDocumentation
    @Deprecated
    public boolean isParticipantMuted(String participantId) {
        //TODO muted management
        //RestParticipant participant = findParticipantById(participantId);
        //return participant != null && participant.isMuted();

        return false;
    }

    /**
     * Informs about the current active speaker.
     *
     * @return the ID of a current active speaker. A null value informs that there is no active speaker at the moment.
     */
    @NoDocumentation
    @Nullable
    public String currentSpeaker() {
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();
        Conference conference = getConference();
        if (null == media || conference == null) {
            return VoxeetSdk.session().getParticipantId();
        } else {
            String currentSpeaker = null;
            for (Participant participant : conference.getParticipants()) {
                if (participant.getId() != null
                        && !participant.getId().equals(VoxeetPreferences.id())
                        && participant.isLocallyActive()) {
                    double peerVuMeter = media.getPeerVuMeter(participant.getId());
                    if (Double.isNaN(peerVuMeter)) peerVuMeter = 0;

                    if (currentSpeaker == null || (peerVuMeter > 0.001 && media.getPeerVuMeter(currentSpeaker) < peerVuMeter))
                        currentSpeaker = participant.getId();
                }
            }
            return currentSpeaker;
        }
    }

    /**
     * Informs the current participant speaking state.
     *
     * @param participant the participant to check.
     * @return the boolean indicating if the participant is speaking.
     */
    public boolean isParticipantSpeaking(@NonNull Participant participant) {
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();
        Conference conference = getConference();
        if (null == media || conference == null) {
            return false;
        } else {
            return getLevel(participant) > 0.001;
        }
    }

    @NonNull
    private String currentParticipantOrEmpty() {
        return Opt.of(VoxeetSdk.session()).then(SessionService::getParticipantId).or("");
    }

    /**
     * Informs about the instance of the desired participant.
     *
     * @param participantId the participant id to check upon
     * @return the instance of the participant. The null value informs that the conference or the participant does not exist in the current time session.
     */
    @Nullable
    public Participant findParticipantById(final String participantId) {
        return Opt.of(getConference()).then(c -> c.findParticipantById(participantId)).orNull();
    }

    /**
     * Informs about the current conference's id
     *
     * @return the conference id or null if none is currently joined
     */
    @NoDocumentation
    @Nullable
    public String getConferenceId() {
        return mConferenceId;
    }

    @NoDocumentation
    @Deprecated
    public boolean muteParticipant(@NonNull String participantId, boolean shouldMute) {
        return false;
    }

    private boolean isVideoOn() {
        ConferenceInformation information = getCurrentConference();
        return null != information && information.isOwnVideoStarted();
    }

    @NoDocumentation
    @Deprecated
    public boolean setParticipantPosition(final String participantId, final double angle, final double distance) {
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();
        if (null != media) {
            //TODO possible improvement here in case of recovery
            ConferenceInformation conference = getCurrentConference();
            if (null != conference) {
                ParticipantPosition position = conference.getPosition(participantId);
                position.angle = angle;
                position.distance = distance;
                conference.setPosition(participantId, position);
            }
            media.changePeerPosition(participantId, angle, distance);
            return true;
        }
        return false;
    }

    /**
     * Gets the current conference type.
     *
     * @return the ConferenceType. the ConferenceType.NONE is a default value.
     */
    @NonNull
    public LocalConferenceType getConferenceType() {
        return Opt.of(getCurrentConference()).then(ConferenceInformation::getConferenceType).or(LocalConferenceType.NONE);
    }

    /**
     * Gets the current `Conference` object.
     *
     * @return the nullable object.
     */
    @Nullable
    public Conference getConference() {
        ConferenceInformation information = getCurrentConference();
        if (null == information) return null;

        return information.getConference();
    }

    /**
     * Gets the current state of presence at the conference. It can be used together with the `ConferenceStatus`.
     *
     * @return true if the conference currently exists or if the connection attempt to the conference is pending.
     */
    public boolean isInConference() {
        return mInConference;
    }

    @NoDocumentation
    void setIsInConference(boolean status) {
        mInConference = status;
    }

    @NoDocumentation
    public double getLevel(@Nullable String participantId) {
        Validate.runningOnUiThread();
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();
        if (null == media || null == participantId) return 0;

        double peerVuMeter = media.getPeerVuMeter(participantId);
        if (Double.isNaN(peerVuMeter)) peerVuMeter = 0;
        return peerVuMeter;

    }

    /**
     * Gets the VU Meter level for the given `Participant`.
     *
     * @param participant the participant, the value can be safely set to an incorrect one
     * @return the value between 0. and 1.
     */
    public double getLevel(@Nullable Participant participant) {
        if (null != participant && participant.getId() != null
                && !participant.getId().equals(VoxeetPreferences.id())
                && participant.isLocallyActive()) {
            return getLevel(participant.getId());
        }
        return 0;
    }

    @NoDocumentation
    @NonNull
    public List<Participant> getParticipants() {
        return Opt.of(getConference()).then(Conference::getParticipants).or(new CopyOnWriteArrayList<>());
    }

    @NoDocumentation
    @NonNull
    public List<Participant> getLastInvitationParticipants() {
        return Opt.of(getCurrentConference()).then(ConferenceInformation::getLastInvitationReceived).or(new ArrayList<>());
    }

    @NoDocumentation
    public boolean isLive() {
        return null != getConference();
    }

    /**
     * Enables to cancel the timeout.
     */
    @NonNull
    public ConferenceService cancelTimeout() {
        removeTimeoutCallbacks();

        return this;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Declines an invitation for the specific conference, based on the conference ID.
     *
     * @param conferenceId the conference id to decline
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Boolean> decline(final String conferenceId) {
        return new DeclinePromise(this,
                VoxeetSdk.mediaDevice(),
                conferenceAccess,
                getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId).createPromise();
    }

    @NoDocumentation
    void onParticipantCanceledScreenShare() {
        //call before any specific code so stopScreenCapturer() should do nothing /!\
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();
        if (null != media) {
            media.stopScreenCapturer();
        }
        ConferenceInformation information = getCurrentConference();
        if (null != information) {
            information.setScreenShareOn(false);
        }
    }

    /**
     * Starts the video stream for the participant, which can be either the remote participant or the local participant that belongs to the current session.
     * <p>
     * For remote participants, this method informs the Voxeet service to forward the remote participant's video to the application, which will result in [streamAdded](/documentation/sdk/reference/android/conference#streamadded) event be emitted when successful. If the remote participant has not started sending video yet then the returned promise will contain an error.
     * <p>
     * For the local participant, this method triggers the application to start sending video to the Voxeet service. The application may also opt to start sending video while it joins the conference.
     * <p>
     * The reject value informs about the `PromisePermissionRefusedEventException` or the `MediaException`.
     * If the application does not have permission, it emits the `PermissionRefusedEvent`.
     * <p>
     * For remote participants, this method informs the Voxeet service to forward the remote participant's video to the application, which will result in [streamAdded](/documentation/sdk/reference/android/conference#streamadded) event be emitted when successful. If the remote participant has not started sending video yet then the returned promise will contain an error.
     * <p>
     * For the local participant, this method triggers the application to start sending video to the Voxeet service. The application may also opt to start sending video while it joins the conference.
     * <p>
     * The reject value informs about the `PromisePermissionRefusedEventException` or the `MediaException`.
     * If the application does not have permission, it emits the `PermissionRefusedEvent`.
     */
    @NonNull
    public Promise<Boolean> startVideo() {
        CameraContext provider = VoxeetSdk.mediaDevice().getCameraContext();
        return startVideo(provider.isDefaultFrontFacing());
    }

    /**
     * Start the video
     * <p>
     * Possible rejection causes :
     * - **MediaEngineException**
     * - **NotInConferenceException**
     * - _Throwable_
     * <p>
     * If the app does not have the permission, will fire a PermissionRefusedEvent
     */
    @NoDocumentation
    @NonNull
    public Promise<Boolean> startVideo(boolean isDefaultFrontFacing) {
        return new StartVideoPromise(this,
                VoxeetSdk.mediaDevice(),
                videoAccess,
                getCurrentConference(),
                getEventBus(),
                isDefaultFrontFacing,
                mConferenceInformationHolder).createPromise();
    }

    /**
     * Stops the local video stream. This method only works when the current participant is at the conference.
     * Any call to this method that is outside of the conference, throws an exception in the catch block of the promise.
     *
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Boolean> stopVideo() {
        return new StopVideoPromise(this,
                VoxeetSdk.mediaDevice(),
                videoAccess,
                getCurrentConference(),
                getEventBus(),
                mConferenceInformationHolder).createPromise();
    }

    /**
     * Create a video answer and post it
     * <p>
     * resolve -> true -> everything went fine
     * resolve -> false -> on error occured
     * reject -> media exception
     */
    @NoDocumentation
    Promise<Boolean> createVideoAnswer(@NonNull final String participantId, @Nullable final OfferDescription offerDescription, final List<OfferCandidate> offerCandidates) {
        return new Promise<>(solver -> {
            MediaSDK media = VoxeetSdk.mediaDevice().getMedia();
            String sdp = "";
            String type = "";
            if (null != offerDescription) {
                sdp = offerDescription.sdp;
                type = offerDescription.type;
            }

            if (null == media) {
                Promise.reject(solver, new MediaEngineException("media is null"));
                return;
            }

            if (null == offerDescription) {
                stopVideo().execute();
                Log.d(TAG, "onCall: unable to start video, are you a listener?");
                solver.resolve(false);
                return;
            }


            SdpDescription description = new SdpDescription(type, sdp);

            List<SdpCandidate> candidates = new ArrayList<>();
            if (offerCandidates != null) {
                for (OfferCandidate candidate : offerCandidates) {
                    candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
                }
            }

            //TODO use the promise.resolve()
            createAnswerForPeerThreaded(media, participantId, description.ssrc, description, candidates, participantId.equals(VoxeetSdk.session().getParticipantId()))
                    .then((ThenPromise<SdpMessage, Integer>) message -> answer(participantId, message))
                    .then((ThenVoid<Integer>) (result) -> solver.resolve(true))
                    .error(error -> {
                        if (error instanceof ParticipantAddedErrorEventException)
                            mEventBus.post(((ParticipantAddedErrorEventException) error).event);
                        else
                            error.printStackTrace();

                        //resolve false value : something happened
                        solver.resolve(false);
                    });
        });
    }

    /**
     * Creates the conference with the corresponding conference alias.
     *
     * @param conferenceAlias create a given conference then join it
     * @return the promise to resolve.
     */
    public Promise<CreateConferenceResult> create(@Nullable final String conferenceAlias) {
        return create(new ConferenceCreateOptions.Builder()
                .setConferenceAlias(conferenceAlias).build()
        );
    }

    /**
     * Create a Conference with the corresponding conference alias and its parameters
     * <p>
     * Possible rejection causes :
     * - **ServerErrorException**
     * - _Throwable_
     *
     * @deprecated use {@link #create(ConferenceCreateOptions)} instead.
     */
    @Deprecated
    @NoDocumentation
    public Promise<CreateConferenceResult> create(@Nullable String conferenceAlias,
                                                  @Nullable MetadataHolder metadata,
                                                  @Nullable ParamsHolder paramsholder) {
        return create(new ConferenceCreateOptions.Builder()
                .setConferenceAlias(conferenceAlias)
                .setMetadataHolder(metadata)
                .setParamsHolder(paramsholder)
                .build()
        );
    }

    /**
     * Creates the conference based on information from the built `ConferenceCreateOptions`.
     *
     * @param conferenceCreateOptions the information holder where id, params and metadata can be passed into
     * @return the conference creation information holder.
     */
    @NonNull
    public Promise<CreateConferenceResult> create(@NonNull ConferenceCreateOptions conferenceCreateOptions) {
        return new CreateConferencePromiseable(this,
                VoxeetSdk.mediaDevice(),
                conferenceAccess,
                getCurrentConference(),
                getEventBus(),
                conferenceCreateOptions.getConferenceAlias(),
                conferenceCreateOptions.getMetadataHolder(),
                conferenceCreateOptions.getParamsHolder()
        ).createPromise();
    }

    /**
     * Joins the conference in the listener mode. When the listen method is used, the application connects to the conference in receiving the only mode and is not able to transmit video or audio.
     * <p>
     * Note that the conferenceId needs to be a Voxeet conferenceId,
     * <p>
     * Possible rejection causes :
     * - **ServerErrorException**
     * - **InConferenceException**
     * - **MediaEngineException**
     * - **ParticipantAddedErrorEventException**
     * - _Throwable_
     *
     * @param conferenceId id of the conference to join as a listener
     * @return the promise to resolve.
     */
    public Promise<Conference> listen(@NonNull String conferenceId) {
        return join(new ConferenceJoinInformation.Builder(conferenceId)
                .setConferenceParticipantType(ConferenceParticipantType.LISTENER)
                .build()
        );
    }

    /**
     * Joins the conference in the broadcaster mode which allows the transmission of audio and video.
     * <p>
     * Note that the conferenceId needs to be a Voxeet conferenceId,
     * <p>
     * Possible rejection causes :
     * - **ServerErrorException**
     * - **InConferenceException**
     * - **MediaEngineException**
     * - **ParticipantAddedErrorEventException**
     * - _Throwable_
     *
     * @param conferenceId id of the conference to join as a broadcaster
     * @return the promise to resolve.
     */
    public Promise<Conference> broadcast(@NonNull String conferenceId) {
        return join(new ConferenceJoinInformation.Builder(conferenceId)
                .setConferenceParticipantType(ConferenceParticipantType.BROADCASTER)
                .build()
        );
    }

    /**
     * Joins the conference with the conference ID.
     * <p>
     * The possible exception in the rejection part can be :
     * - **ServerErrorException**
     * - **InConferenceException**
     * - **MediaEngineException**
     * - **ParticipantAddedErrorEventException**
     * - _Throwable_
     *
     * @param conferenceId the existing conference to join
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<Conference> join(@NonNull final String conferenceId) {
        return join(new ConferenceJoinInformation.Builder(conferenceId)
                .setConferenceParticipantType(ConferenceParticipantType.NORMAL)
                .build()
        );
    }

    /**
     * Replays the previously recorded conference.
     * <p>
     * Possible rejection causes :
     * - **ServerErrorException**
     * - **InConferenceException**
     * - **MediaEngineException**
     * - **ParticipantAddedErrorEventException**
     * - _Throwable_
     *
     * @param conferenceJoinInformation the holder of the information to join
     * @return the promise to resolve.
     */
    private Promise<Conference> join(@NonNull ConferenceJoinInformation conferenceJoinInformation) {
        String id = Opt.of(conferenceJoinInformation).then(ConferenceJoinInformation::getConferenceId).or("");
        return new JoinPromise(this,
                VoxeetSdk.audio(),
                VoxeetSdk.mediaDevice(),
                conferenceAccess,
                getConferenceInformation(id),
                getEventBus(),
                getVoxeetSDK(),
                id,
                conferenceJoinInformation.getConferenceParticipantType()).createPromise();
    }

    /**
     * resole -> true
     * reject -> network error or on created error
     *
     * @return a promise with a boolean indicating the demo conference is (or not) successfully joined
     */
    @NoDocumentation
    public Promise<Conference> demo() {
        return new Promise<>(solver -> {
            HttpHelper.promise(conferenceAccess.demo(), ServerErrorOrigin.UNDEFINED)
                    .then((ThenPromise<HttpHelper.HttpAnswer<CreateConferenceResult>, Conference>) answer -> {
                        CreateConferenceResult object = answer.object;
                        return listen(Opt.of(object).then(s -> s.conferenceId).or(""));
                    })
                    .then((ThenVoid<Conference>) solver::resolve)
                    .error(error -> Promise.reject(solver, Opt.of(error).or(new IllegalStateException())));

        });
    }

    /**
     * Invite a list of participants given a specific conference id
     *
     * @param conferenceId       the conference id of the conference where to invite the participants
     * @param participantOptions a non null list of information about the participants to invite into the specified conference
     * @return a non null promise to resolve.
     */
    @NoDocumentation
    @NonNull
    public Promise<List<Participant>> invite(final String conferenceId, final List<ParticipantInfo> participantOptions) {
        return new Promise<>(solver -> {
            //Warning = getConferenceParticipants is returning a new non retained if not in a conference
            List<Participant> participants = getParticipants();
            List<String> strings = new ArrayList<>();

            if (null != participantOptions) {
                for (ParticipantInfo info : participantOptions) {
                    if (null != info && info.getExternalId() != null) {
                        String participantInfoId = Opt.of(info).then(ParticipantInfo::getExternalId).orNull();
                        for (Participant participant : participants) {
                            String arrayId = Opt.of(participant).then(Participant::getInfo).then(ParticipantInfo::getExternalId).orNull();
                            if (null != arrayId && arrayId.equalsIgnoreCase(participantInfoId)) {
                                participant.updateIfNeeded(info.getName(), info.getAvatarUrl());
                            }
                        }
                        strings.add(info.getExternalId());
                    }
                }
            }

            solver.resolve(new InvitePromise(ConferenceService.this,
                    VoxeetSdk.mediaDevice(),
                    conferenceAccess,
                    getCurrentConference(),
                    getEventBus(),
                    conferenceId,
                    strings).createPromise());
        });
    }

    /**
     * Replays the previously recorded conference.
     *
     * @param conferenceId the mConference id
     * @param offset       the offset to start with
     * @return the promise to resolve that indicates the result of the request.
     */
    @NonNull
    public Promise<Conference> replay(final String conferenceId, final long offset) {
        return new ReplayPromise(this,
                VoxeetSdk.mediaDevice(),
                conferenceAccess,
                getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId,
                offset).createPromise();
    }

    /**
     * Get the current mConference status
     * send events but also managed by the promise
     *
     * @param conferenceId the conference id to get the information from
     * @return the promise to resolve containing the `ConferenceStatusResult`.
     */
    @NonNull
    public Promise<ConferenceStatusResult> getConferenceStatus(final String conferenceId) {
        return new GetConferenceStatusPromiseable(this,
                VoxeetSdk.mediaDevice(),
                outConferenceAccess,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> event
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return the promise to resolve.
     */
    @NoDocumentation
    public Promise<ConferenceHistoryResult> conferenceHistory(final String conferenceId) {
        return new GetConferenceHistoryPromiseable(this,
                VoxeetSdk.mediaDevice(),
                outConferenceAccess,
                getCurrentConference(),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return the promise to resolve.
     */
    @NoDocumentation
    public Promise<Boolean> subscribe(@NonNull final String conferenceId) {
        return new SubscribeConferenceEventPromiseable(this,
                VoxeetSdk.mediaDevice(),
                outConferenceAccess,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @return a valid promise
     */
    @NoDocumentation
    public Promise<Boolean> unSubscribe(@NonNull final String conferenceId) {
        return new UnsubscribeConferenceEventPromiseable(this,
                VoxeetSdk.mediaDevice(),
                outConferenceAccess,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId
     * @return the promise to resolve. indicating the result of the request
     */
    @NoDocumentation
    public Promise<Boolean> subscribeForCall(final String conferenceId) {
        return new SubscribeForCallStartPromiseable(this,
                VoxeetSdk.mediaDevice(),
                outConferenceAccess,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    @NoDocumentation
    public Promise<Boolean> unSubscribeFromCall(final String conferenceId) {
        return new UnsubscribeForCallStartPromiseable(this,
                VoxeetSdk.mediaDevice(),
                outConferenceAccess,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(PushConferenceDestroyed event) {
        EventBus.getDefault().post(new ConferenceDestroyedPush(event.conferenceId));
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(@NonNull OwnConferenceParticipantSwitch switchEvent) {
        Log.d(TAG, "onEvent: OwnConferenceParticipantSwitch : type ? " + switchEvent.getType());
        if (isLive()) {
            leave()
                    .then((ThenVoid<Boolean>) (result) -> Log.d(TAG, "onCall: on leave for switch participant"))
                    .error((ErrorPromise) error -> Log.d(TAG, "onError: on leave for switch participant"));
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(@NonNull SocketStateChangeEvent event) {
        if (WebSocketState.CONNECTED.equals(event.state)) {
            String conferenceId = getConferenceId();
            ConferenceInformation information = getCurrentConference();

            if (null != conferenceId && null != information && ConferenceStatus.JOINED.equals(information.getConferenceState())) {
                if (ConferenceConfigurations.isICERestartEnabled) {
                    Log.d(TAG, "onEvent: SocketConnectEvent Joined <3");
                    final Call<ResponseBody> participant = mediaAccess.iceRestart(conferenceId);
                    HttpHelper.enqueue(participant, (response, object, exception) -> {
                        if (null != exception) {
                            Log.d(TAG, "onEvent: SocketConnectEvent Joined error </3");
                            HttpException.dumpErrorResponse(response);
                        } else {
                            Log.d(TAG, "onEvent: SocketConnectEvent Joined responded <3");
                        }
                    });
                } else {
                    Log.d(TAG, "onEvent: socket state opened while in conference but no isICERestartEnabled() = true. A reconnect may be longer");
                }
            } else {
                ConferenceStatus state = ConferenceStatus.DEFAULT;
                if (null != information) {
                    state = information.getConferenceState();
                }
                Log.d(TAG, "onEvent: SocketConnectEvent not rejoined </3 " + state + " " + conferenceId);
            }
        }
    }

    @NoDocumentation
    void setCurrentConferenceIfNotInPreviousConference(@NonNull ConferenceInformation conference) {
        String conferenceId = conference.getConference().getId();

        boolean invalidId = null == conferenceId;
        boolean inConference = isInConference();
        boolean sameConference = null == mConferenceId || mConferenceId.equals(conferenceId);
        boolean canOverride = !inConference || sameConference;

        if (!invalidId && canOverride) {
            mConferenceId = conferenceId;
        }
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> ServerErrorException
     * <p>
     * Warning : if multiple attempt to join the same conference are made, it is not managed...
     * May be fix in a future version but currently not in the flow
     *
     * @param conference the conference to join
     * @return a promise to catch
     */
    @NonNull
    @NoDocumentation
    Promise<Conference> joinConferenceInternalPackage(@NonNull final ConferenceInformation conference) {
        return new Promise<>(solver -> {
            final String conferenceId = conference.getConference().getId();

            boolean inConference = isInConference();
            boolean sameConference = null == mConferenceId || mConferenceId.equals(conferenceId);

            //remove the timeout, a new one should be started when participant will be invited
            removeTimeoutCallbacks();

            Log.d(TAG, "joining " + conferenceId);

            setIsInConference(true);
            mConferenceId = conferenceId;

            conference.setConferenceState(ConferenceStatus.JOINING);

            AudioService service = VoxeetSdk.audio();
            service.enableAec(true);
            service.enableNoiseSuppressor(true);

            boolean isSuccess = initMedia(conference.isListener());
            Conference correspondingConference = conference.getConference();

            if (!isSuccess) {
                Log.d(TAG, "onCall: InitMedia failed... new state = left");
                try {
                    setIsInConference(false);
                    conference.setConferenceState(ConferenceStatus.LEFT);
                    closeMedia();
                    ConferenceStatusUpdatedEvent error = new ConferenceStatusUpdatedEvent(correspondingConference, correspondingConference.getState());
                    mEventBus.post(error);
                    throw new MediaEngineException("onCall: InitMedia failed... new state = left");
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                }
                return;
            }

            mEventBus.post(new ConferenceStatusUpdatedEvent(correspondingConference, correspondingConference.getState()));

            final Call<ResumeConference> participant = conferenceAccess.join(conferenceId, new JoinParameters(DeviceType.ANDROID, conference.isListener()));

            HttpHelper.enqueue(participant, (response, object, exception) -> {
                if (null != object) {
                    createOrSetConferenceWithParams(object.getConferenceId(),
                            object.getConferenceAlias());
                    initMedia(conference.isListener());


                    String automaticTelecomModePrefix = ConferenceConfigurations.automaticTelecomModePrefix;

                    boolean hasAutomaticPrefix = null != automaticTelecomModePrefix && !TextUtils.isEmpty(automaticTelecomModePrefix);
                    Log.d(TAG, "onSuccess: trying to join conference, will an attempt to be telecom be made ? " + hasAutomaticPrefix);
                    ConferenceInformation information = getCurrentConference();
                    if (null != information) {

                        if (hasAutomaticPrefix && !TextUtils.isEmpty(object.getConferenceAlias())
                                && object.getConferenceAlias().startsWith(automaticTelecomModePrefix)) {
                            Log.d(TAG, "onSuccess: the conference is now in telecom mode");
                            information.setTelecomMode(true);
                        } else {
                            Log.d(TAG, "onSuccess: the prefix does not match - if the telecom mode is required, it must be done programmatically at this point");
                        }

                        information.setConferenceState(ConferenceStatus.JOINED);
                    }

                    solver.resolve(onConferenceResumedInternal(object));
                } else {
                    setIsInConference(false);

                    Conference correspondingConference1 = conference.getConference();
                    //ConferenceInformation information = getCurrentConference();
                    conference.setConferenceState(ConferenceStatus.ERROR);
                    closeMedia();

                    //send error -> left
                    ConferenceStatusUpdatedEvent event = new ConferenceStatusUpdatedEvent(correspondingConference1, correspondingConference1.getState());
                    getEventBus().post(event);

                    if (exception instanceof HttpException) {
                        Promise.reject(new ServerErrorException(((HttpException) exception).error, ServerErrorOrigin.JOIN));
                    } else {
                        solver.reject(exception);
                    }
                }
            });
        });
    }

    @NonNull
    public Promise<Boolean> simulcast(@NonNull List<ParticipantQuality> requested) {
        return new SimulcastPromiseable(this,
                VoxeetSdk.mediaDevice(),
                simulcastAccess,
                getCurrentConference(),
                getEventBus(),
                requested).createPromise();
    }

    @NonNull
    private Promise<Conference> onConferenceResumedInternal(@NonNull ResumeConference response) {
        mConferenceId = response.getConferenceId();

        ConferenceInformation information = getCurrentConference();

        //information should not be null in this case
        if (null == information) {
            return Promise.reject(new NotInConferenceException());
        }

        Conference conference = information.getConference();

        conference.setConferenceId(response.getConferenceId());
        conference.setConferenceAlias(response.getConferenceId());

        if (response.getConferenceAlias() != null) {
            conference.setConferenceAlias(response.getConferenceAlias());
        }

        information.participantsToConferenceParticipants(response.getParticipants());

        setIsInConference(true);

        VoxeetSdk.audio().setAudioRoute(AudioRoute.ROUTE_PHONE);

        mEventBus.post(new ConferenceStatusUpdatedEvent(conference, conference.getState()));

        //now loop over every participants to send artifical participant "updated" event
        List<RestParticipant> participants = response.getParticipants();
        int length = null != participants ? participants.size() : -1;

        Log.d(TAG, "onConferenceResumedInternal: resuming with participants ? " + participants + " " + length);
        if (null != participants) {
            for (RestParticipant participant : participants) {
                Log.d(TAG, "onConferenceResumedInternal: call onParticipantAddedOrUpdated " + participant);
                onParticipantAddedOrUpdated(conference.getId(),
                        participant.getParticipantId(),
                        participant.getParticipantInfo(),
                        ConferenceParticipantStatus.fromString(participant.getStatus()),
                        true);
            }
        }

        if (null == response.getCandidates() && null == response.getDescription()) {
            Log.d(TAG, "onConferenceResumedInternal: candidates + description");
            return Promise.resolve(conference);
        }

        Log.d(TAG, "onConferenceResumedInternal: having candidates and description");

        ParticipantInfo info = VoxeetSdk.session().getParticipantInfo();
        String name = null;
        String externalId = null;
        String avatarUrl = null;
        String participantId = "";
        if (null != info) {
            participantId = Opt.of(VoxeetSdk.session()).then(SessionService::getParticipantId).or("");
            name = info.getName();
            externalId = info.getExternalId();
            avatarUrl = info.getAvatarUrl();
        }

        return handleAnswer(response.getConferenceId(), participantId, externalId, name, avatarUrl,
                "ANDROID", true,
                response.getDescription(), response.getCandidates());
    }

    /**
     * Retrieves the list of participants invited to the conference.
     *
     * @param conferenceId the conference id from which retrieving the invited participants
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<ConferenceParticipantsInvitedResult> getInvitedParticipants(@NonNull String conferenceId) {
        return new GetConferenceStatus(this,
                VoxeetSdk.mediaDevice(),
                outConferenceAccess,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * Leaves the current conference.
     */
    @NonNull
    public Promise<Boolean> leave() {
        return new LeavePromise(this,
                VoxeetSdk.mediaDevice(),
                conferenceAccess,
                getCurrentConference(),
                getEventBus())
                .createPromise();
    }


    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Event management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final InvitationReceivedEvent invitation) {

        //get conference id
        String conferenceId = invitation.conferenceId;
        if (null == conferenceId && null != invitation.conference) {
            conferenceId = invitation.conference.getConferenceId();
        }

        Log.d(TAG, "onEvent: current mConferenceId " + mConferenceId + " vs " + invitation.conferenceId + " vs " + conferenceId);

        if (null != mConferenceId && mConferenceId.equals(conferenceId)) {
            Log.d(TAG, "onEvent: receiving invitation for our conference, we prevent this");
            return;
        }

        //we update the information for this conference only
        final ConferenceInformation information = getConferenceInformation(null != conferenceId ? conferenceId : "");

        boolean is_own_participant = invitation.participantId.equals(VoxeetPreferences.id());
        ConferenceType type = ConferenceType.fromId(invitation.conference.getConferenceType());

        if (isLive() && null != getConferenceId() && getConferenceId().equals(conferenceId)) {
            Log.d(TAG, "onEvent: receiving invitation for the same conference, invalidate the call");
            return;
        }

        if (!is_own_participant || ConferenceType.SCHEDULED.equals(type)) {
            InvitationReceivedEvent.ParticipantInviter inviter = invitation.getInviter();
            List<Invitation> invitations = invitation.invitations;

            final Participant[] foundParticipant = {null};

            if (null != inviter && null != inviter.externalId && null != inviter.participantId) {
                Log.d(TAG, "onEvent: Invitation with inviter " + inviter.participantId + " " + inviter.externalId);
                ParticipantInfo info = new ParticipantInfo(inviter.nickName, inviter.externalId, inviter.externalAvatarUrl);

                Participant inviter_participant = new Participant(inviter.participantId, info);

                foundParticipant[0] = inviter_participant;
                information.getLastInvitationReceived().add(inviter_participant);
                information.getParticipantIdsCached().put(inviter.participantId, inviter.externalId);
            } else {
                Log.d(TAG, "onEvent: Invitation with invalid inviter");
            }

            getConferenceStatus(invitation.conference.getConferenceId())
                    .then((result, internal_solver) -> {
                        Log.d(TAG, "onSuccess: " + result);

                        //removed the check for live Event since now it is able to have multiple

                        List<ConferenceUser> participants = Opt.of(result).then(ConferenceStatusResult::getConferenceUsers).or(new ArrayList<>());
                        if (participants.size() > 0) {
                            Log.d(TAG, "onEvent: participants " + participants.size());

                            String foundExternalId = null;

                            List<Participant> merged_list = new ArrayList<>();
                            //add every participant from the mConference
                            for (SdkParticipant participant : Opt.of(result).then(r -> r.participants).or(new ArrayList<>())) {
                                merged_list.add(new Participant(participant));
                            }
                            //add every use from the invitation
                            merged_list.addAll(getLastInvitationParticipants());

                            List<Participant> list = ConferenceUtils.findParticipantsMatching(currentParticipantOrEmpty(), merged_list);

                            for (Participant conferenceParticipant : list) {
                                String participantId = conferenceParticipant.getId();
                                if (null == foundExternalId) {
                                    String externalId = Opt.of(conferenceParticipant).then(Participant::getInfo).then(ParticipantInfo::getExternalId).orNull();
                                    String cachedExternalId = information.getParticipantIdsCached().get(participantId);
                                    Log.d(TAG, "onEvent: " + participantId + " " + externalId + " " + cachedExternalId);

                                    if (null == foundParticipant[0])
                                        foundParticipant[0] = conferenceParticipant;
                                    if (null != participantId && (null == externalId || participantId.equals(externalId))) {
                                        //should be different than null
                                        externalId = cachedExternalId;
                                        if (null != externalId) {
                                            foundExternalId = cachedExternalId;
                                        }
                                    }
                                }
                            }

                            if (foundParticipant[0] != null && null == foundExternalId) {
                                foundExternalId = foundParticipant[0].getId();
                                Log.d(TAG, "externalId is null, setting it to participantId");
                            }

                            Map<String, String> infos = new HashMap<>();
                            infos.put(Constants.INVITER_ID, Opt.of(foundParticipant[0]).then(Participant::getId).or(""));
                            infos.put(Constants.INVITER_NAME, Opt.of(foundParticipant[0]).then(Participant::getInfo).then(ParticipantInfo::getName).or(""));
                            //NOTIF_TYPE ?? useless or usefull ??
                            infos.put(Constants.NOTIF_TYPE, Opt.of(result).then(ConferenceStatusResult::getType).or(""));
                            infos.put(Constants.INVITER_EXTERNAL_ID, Opt.of(foundParticipant[0]).then(Participant::getInfo).then(ParticipantInfo::getExternalId).or(""));
                            infos.put(Constants.INVITER_URL, Opt.of(foundParticipant[0]).then(Participant::getInfo).then(ParticipantInfo::getAvatarUrl).or(""));
                            infos.put(Constants.CONF_ID, Opt.of(result).then(r -> r.conferenceId).or(""));

                            VoxeetSdk.notification().onInvitationReceived(context, new InvitationBundle(infos));

                            mEventBus.post(new IncomingCallEvent(Opt.of(result).then(r -> r.conferenceId).or("")));
                        }
                    })
                    .error(Throwable::printStackTrace);
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(OwnConferenceCreated ownConferenceStartedEvent) {
        if (null != ownConferenceStartedEvent.conferenceInfos) {
            mConferenceId = ownConferenceStartedEvent.conferenceInfos.getConferenceId();
        }

        Conference conference = getConference();

        if (null != conference) {
            conference.setConferenceInfos(ownConferenceStartedEvent.conferenceInfos);
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(RenegociationEndedEvent event) {
        Log.d(TAG, "onEvent: Renegociation " + event.getConferenceId()
                + " " + event.getType() + " " + event.getAnswerReceived()
                + " " + event.getOfferSent());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final OfferCreated offer) {
        Conference conference = getConference();
        String conferenceId = null != conference ? conference.getId() : null;

        if (null != conferenceId && conferenceId.equals(offer.getConferenceId())) {

            Participant participant = findParticipantById(offer.getParticipantId());

            if (null == participant) {
                participant = new Participant(offer.getParticipantId(), new ParticipantInfo(offer.getName(), offer.getExternalId(), offer.getAvatarUrl()));
                Log.d(TAG, "run: WARNING obtained OfferCreated for participant which did not existed " + participant);
                conference.updateParticipant(participant);

                mEventBus.post(new ParticipantAddedEvent(conference, participant));
            }

            handleAnswer(offer.getConferenceId(),
                    offer.getParticipantId(),
                    offer.getExternalId(),
                    offer.getName(),
                    offer.getAvatarUrl(),
                    offer.getDevice(),
                    offer.isMaster(),
                    offer.getDescription(),
                    offer.getCandidates())
                    .then((ThenVoid<Conference>) (conf) -> Log.d(TAG, "onCall: answer called, result is " + conf + " " + offer.getParticipantId() + " " + VoxeetPreferences.id()))
                    .error(error -> {
                        error.printStackTrace();
                        ExceptionManager.sendException(error);
                    });
        } else {
            Log.d(TAG, "onEvent: OfferCreated for another conference");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ParticipantAdded event) {
        onParticipantAddedOrUpdated(event.conferenceId,
                event.participantId, new ParticipantInfo(event.name, event.externalId, event.avatarUrl),
                ConferenceParticipantStatus.fromString(event.status),
                true);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ParticipantUpdated event) {
        onParticipantAddedOrUpdated(event.conferenceId,
                event.participantId, null,
                ConferenceParticipantStatus.valueOf(event.status),
                false);
    }

    private void onParticipantAddedOrUpdated(@NonNull String conferenceId,
                                             @NonNull String participantId,
                                             @Nullable ParticipantInfo participantInfo,
                                             @NonNull ConferenceParticipantStatus status,
                                             boolean added) {
        String id = VoxeetSdk.session().getParticipantId();
        if (null == id) {
            return;
        }

        boolean known_participant = false;
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();

        ConferenceInformation information = getConferenceInformation(conferenceId);
        Conference conference = information.getConference();
        Participant participant = conference.findParticipantById(participantId);

        if (null != participant) {
            known_participant = true;
            if (added && null != participantInfo) {
                participant.setParticipantInfo(participantInfo);
            }
            participant.updateStatus(status);
        } else {
            participant = new Participant(participantId, participantInfo);

            conference.updateParticipant(participant);
        }

        //update the current conference state specificly from the participants point of view
        updateConferenceFromParticipants();

        switch (status) {
            case CONNECTING:
                if (!id.equals(participant.getId())) {
                    Log.d(TAG, "Cancelling timeout timer from participant connecting");
                    removeTimeoutCallbacks();
                }
                break;
            case LEFT:
            case DECLINE:
                if (null != media) {
                    participant.streamsHandler().removeAllStreams();
                    Log.d(TAG, "In mConference participant with id: " + participant + " status updated to " + status);
                    media.removePeer(participantId);
                }
                break;
            default:
                Log.d(TAG, "status not managed updated to " + status);
        }

        //check if the timeout runnable was not triggered
        boolean timeout_triggered = null != timeoutRunnable && timeoutRunnable.isTriggered();

        if (!timeout_triggered) {
            switch (status) {
                case DECLINE:
                    if (!participantId.equals(VoxeetSdk.session().getParticipantId())) {
                        Log.d(TAG, "Conference participant with id: " + participantId + " declined the call");

                        checkForTelecomEvent(conferenceId, participantId);
                        mEventBus.post(new ParticipantUpdatedEvent(conference, participant));
                        break;
                    }
                case LEFT:
                    checkForTelecomEvent(conference.getId(), participant.getId());
                default:
                    updateConferenceFromParticipants();
                    //if decline but not managed -> fire
                    //else -> fire
                    if (known_participant) {
                        mEventBus.post(new ParticipantUpdatedEvent(conference, participant));
                    } else {
                        mEventBus.post(new ParticipantAddedEvent(conference, participant));
                    }
            }
        }
    }

    private void checkForTelecomEvent(String conferenceId, String participantId) {
        ConferenceInformation currentConference = getCurrentConference();
        //checking if the current conference is in telecom mode
        Log.d(TAG, "checkForTelecomEvent: having currentConference ? " + currentConference);
        if (null != currentConference && currentConference.isTelecomMode()) {
            Conference conference = currentConference.getConference();
            //check if the conference matches
            if (null != conferenceId && conferenceId.equals(conference.getId())) {
                String id = VoxeetPreferences.id();
                if (!TextUtils.isEmpty(participantId) && null != id && !id.equals(participantId)) {
                    leave().then((ThenVoid<Boolean>) (result) -> Log.d(TAG, "onCall: conference left since declined/left event"))
                            .error(error -> Log.e(TAG, "onError: conference left since declined/left event", error));
                }
            }
        } else {
            Log.d(TAG, "checkForTelecomEvent: not a telecom mode");
        }
    }


    @Subscribe(threadMode = ThreadMode.MAIN, priority = 999)
    public void onEvent(final ConferenceDestroyedPush event) {
        onEndOrDestroyed(event.conferenceId, ConferenceStatus.DESTROYED);
    }

    @Subscribe(threadMode = ThreadMode.MAIN, priority = 999)
    public void onEvent(final ConferenceEnded event) {
        onEndOrDestroyed(event.conferenceId, ConferenceStatus.ENDED);
    }

    private void onEndOrDestroyed(@Nullable String eventConfId, ConferenceStatus newStatus) {
        String conferenceId = mConferenceId;
        ConferenceInformation information = getCurrentConference();
        if (null != information && !TextUtils.isEmpty(information.getConference().getId())) {
            conferenceId = information.getConference().getId();
        }

        if (null != information && !TextUtils.isEmpty(conferenceId) && conferenceId.equals(eventConfId)) {
            information.setConferenceState(newStatus);
            closeMedia();
            mEventBus.post(new ConferenceStatusUpdatedEvent(information.getConference(), newStatus));
        } else {
            Log.d(TAG, "onEvent: another conference has ended");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final QualityUpdated event) {
        for (ConferenceUser conferenceParticipant : event.participants) {
            Participant participant = findParticipantById(conferenceParticipant.getUserId());
            if (participant != null) {
                participant.setQuality(conferenceParticipant.getQuality());

                mEventBus.post(new ConferenceParticipantQualityUpdatedEvent(participant));
            }
        }
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Protected management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    @NoDocumentation
    @NonNull
    protected Participant createParticipant(String participantId, String device, ParticipantInfo participantInfo) {
        //TODO add device information for the participants ?
        return new Participant(participantId, /*device,*/ participantInfo);
    }

    @NoDocumentation
    protected void setConference(@NonNull Conference conference) {
        mConferenceId = conference.getId();
        ConferenceInformation information = getCurrentConference();
        if (null != information) information.setConference(conference);
    }

    @NoDocumentation
    protected void setConferenceAlias(String alias) {
        Conference conference = getConference();
        if (null != conference) conference.setConferenceAlias(alias);
        //null ? throw error B)
    }

    /**
     * Return true if the current implementation is SDK
     * false otherwise
     *
     * @return the state of this implementation
     */
    @NoDocumentation
    protected boolean isSDK() {
        return true;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    @NoDocumentation
    protected Promise<Integer> answer(final String peer, final SdpMessage message) {
        return new Promise<>(solver -> {
            if (null == mConferenceId) {
                Promise.reject(solver, new NotInConferenceException());
                return;
            }

            if (null == message) {
                Promise.reject(new ParticipantAddedErrorEventException(new ParticipantAddedErrorEvent("Invalid message")));
                return;
            }

            Log.d("SDKMEDIA", "answer: peer := " + peer + " message := " + message);

            final Call<ResponseBody> call = mediaAccess.answerConference(mConferenceId, peer, message.getDescription());
            HttpHelper.enqueue(call, (response, object, exception) -> {
                int code = Opt.of(response).then(Response::code).or(0);

                if (null != exception) {
                    HttpException.dumpErrorResponse(response);
                    Promise.reject(solver, new ParticipantAddedErrorEventException(new ParticipantAddedErrorEvent(handleError(exception))));
                } else if (200 != code) {
                    Promise.reject(solver, new ParticipantAddedErrorEventException(new ParticipantAddedErrorEvent(code + "")));
                } else {
                    solver.resolve(code);
                }
            });
        });
    }


    /**
     * resolve -> true everything went fine
     * resolve -> false an error occured
     * reject -> media exception
     *
     * @return a promise with the result of the call
     */
    @NoDocumentation
    private Promise<Conference> handleAnswer(@NonNull final String conferenceId,
                                             @NonNull final String participantId,
                                             @Nullable final String externalId,
                                             @Nullable final String name,
                                             @Nullable final String avatarUrl,
                                             @NonNull final String device,
                                             final boolean isMaster,
                                             @NonNull final OfferDescription offerDescription,
                                             @NonNull final List<OfferCandidate> offerCandidates) {
        return new Promise<>(solver -> {
            MediaSDK media = VoxeetSdk.mediaDevice().getMedia();

            if (null == media) {
                Promise.reject(solver, new MediaEngineException("handleAnswer media is null"));
                return;
            }

            ConferenceInformation conferenceInformation = getConferenceInformation(conferenceId);
            final Conference conference = conferenceInformation.getConference();

            if (!conferenceId.equals(mConferenceId)) {
                Log.d(TAG, "onCall: CONFERENCE IS NOT THE SAME ! ANSWER SHALL BE DISCARDED");
            }

            SdpDescription description = new SdpDescription(offerDescription.type, offerDescription.sdp);

            List<SdpCandidate> candidates = new ArrayList<>();
            for (OfferCandidate candidate : offerCandidates) {
                Log.d(TAG, "onCall: sdpcandidate " + candidate.getMid() + " " + candidate.getmLine());
                candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
            }

            //String participantId = offer.getParticipantId();//isSDK() ? offer.getExternalId() : offer.getParticipantId();
            Log.d(TAG, "handleAnswer: offer := " + participantId + " " + externalId);
            Log.d("SDKMEDIA", "handleAnswer: " + participantId + " " + externalId + " " + description.ssrc);


            Conference conference1 = getConference();

            //TODO in the callback, check for the conferenceId being the same !
            createAnswerForPeerThreaded(media, participantId, description.ssrc, description, candidates, isMaster)
                    .then((ThenPromise<SdpMessage, Integer>) (message) -> {
                        if (null == mConferenceId) {
                            Log.d(TAG, "onMessage: INVALID CONFERENCE ID WHEN OFFER IS RECEIVED");
                            mConferenceId = conferenceId;
                        }

                        Participant participant = null;

                        for (Participant in_conf : conference1.getParticipants()) {
                            if (in_conf.getId() != null && in_conf.getId().equals(participantId)) {
                                participant = in_conf;
                            }
                        }

                        ParticipantInfo infos = ParticipantUtils.createParticipantInfo(participantId, name, externalId, avatarUrl);

                        if (participant == null) {
                            participant = new Participant(participantId, /*device,*/ infos);
                            conference1.updateParticipant(participant);
                        } else {
                            participant.setParticipantInfo(infos);
                        }


                        setParticipantPosition(participantId, 0, 0.5);

                        return answer(participantId, message);
                    })
                    .then((result) -> {
                        Log.d(TAG, "onSuccess: " + result);

                        solver.resolve(conference1);
                    })
                    .error(error -> {
                        if (error instanceof ParticipantAddedErrorEventException)
                            mEventBus.post(((ParticipantAddedErrorEventException) error).event);
                        else
                            error.printStackTrace();

                        solver.reject(error);
                    });
        });
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    @NoDocumentation
    protected Participant updateConferenceParticipants(String participantId, ConferenceParticipantStatus status) {
        Participant participant = findParticipantById(participantId);
        Conference conference = getConference();

        if (null != participant && null != conference) {
            //TODO check for owner information ?
            //TODO set quality ?
            participant.updateStatus(status);

            return participant;
        }
        return null;
    }

    /**
     * @return false if permission refused -> cancel ! true otherwise
     */
    @NoDocumentation
    protected boolean initMedia(boolean listenerMode) {
        Validate.notNull(mContext, "mContext");
        Validate.notNull(VoxeetPreferences.id(), "participant id");
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();

        if (null == media) {
            try {
                if (!listenerMode && !Validate.hasMicrophonePermissions(context)) {
                    Log.d(TAG, "the app does not seem to have mic permission, disabling mic");
                    mute(true);
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                    return false;
                }


                AudioService audioService = VoxeetSdk.audio();
                MediaDeviceService mediaDeviceService = VoxeetSdk.mediaDevice();

                audioService.enable();
                mediaDeviceService.createMedia(context, VoxeetPreferences.id(), mediaStreamListener, cameraEventsHandler,
                        ConferenceConfigurations.isDefaultVideo || isVideoOn(), !listenerMode && Validate.hasMicrophonePermissions(context));
                media = mediaDeviceService.getMedia();

                //audioService.setSpeakerMode(ConferenceConfigurations.isDefaultOnSpeaker);
                audioService.unsetMediaRoute().requestAudioFocus();
                audioService.checkDevicesToConnectOnConference(ConferenceConfigurations.isDefaultOnSpeaker);

                if (null != media) {
                    if (ConferenceConfigurations.isDefaultMute) {
                        media.mute();
                    } else {
                        media.unMute();
                    }
                }
            } catch (IllegalStateException e) {
                e.printStackTrace();
                return false;
            } catch (MediaEngineException e) {
                e.printStackTrace();
                return false;
            }
        }

        return true;
    }

    @NoDocumentation
    protected void closeMedia() {
        MediaSDK media = VoxeetSdk.mediaDevice().getMedia();
        if (null != media) {
            Conference conference = getConference();
            ConferenceInformation information = getCurrentConference();

            if (null != information) {
                mConferenceInformationHolder.setVideoState(information, VideoState.STOPPED);
                information.setRequestedOwnVideo(false);
                information.setScreenShareOn(false);
                information.setConferenceState(ConferenceStatus.LEFT);
            }

            try {
                for (Participant participant : Opt.of(conference).then(Conference::getParticipants).or(new CopyOnWriteArrayList<>()))
                    if (participant != null && participant.getId() != null)
                        media.removePeer(participant.getId());
            } catch (Exception ex) {
                Log.e(TAG, "Error", ex);
            }

            VoxeetSdk.audio().abandonAudioFocusRequest().setMediaRoute().disable();

            VoxeetSdk.mediaDevice().releaseMedia();

            media.unsetStreamListener();
        }

        setIsInConference(false);
        mConferenceId = null;
    }

    /**
     * Remove any previous timeout callback
     * <p>
     * To be used when it is mandatory to remove such behaviour
     * for instance, when joining a new mConference
     * <p>
     * It happens since we do not want to manage every edge case :
     * - decline
     * - network error
     * - etc...
     */
    @NoDocumentation
    protected void removeTimeoutCallbacks() {
        if (timeoutRunnable != null) {
            timeoutRunnable.setCanceled(true);
            handler.removeCallbacks(timeoutRunnable);
        }
    }

    @NoDocumentation
    protected void sendTimeoutCallbacks() {
        if (ConferenceConfigurations.TelecomWaitingForParticipantTimeout != -1) {

            timeoutRunnable = new TimeoutRunnable(this,
                    mEventBus,
                    ConferenceConfigurations.TelecomWaitingForParticipantTimeout);

            handler.postDelayed(timeoutRunnable, ConferenceConfigurations.TelecomWaitingForParticipantTimeout);
        }
    }

    @NonNull
    @NoDocumentation
    protected ConferenceInformation createOrSetConferenceWithParams(@NonNull String conferenceId, @Nullable String conferenceAlias) {
        Log.d(TAG, "createOrSetConferenceWithParams: set conference id := " + conferenceId);
        ConferenceInformation information = getConferenceInformation(conferenceId);

        Conference conference = information.getConference();

        conference.setConferenceId(conferenceId);
        if (null != conferenceAlias || null == conference.getAlias()) {
            conference.setConferenceAlias(conferenceAlias);
        }

        //the information is always valid
        return information;
    }

    /**
     * Gets the current conference information holder. It contains the instance of the class holding the conference and other utility components.
     *
     * @return the current conference that is on air or null.
     */
    @Nullable
    public ConferenceInformation getCurrentConference() {
        if (null == mConferenceId) return null;
        return getConferenceInformation(mConferenceId);
    }

    /**
     * Informs about that at least one participant is actively connected.
     *
     * @return indicates if at least one participant is actively connected to the conference.
     */
    public boolean hasParticipants() {
        Conference conference = getConference();

        if (null == conference) return false;

        if (conference.hasLocalStreams(false)) return true;
        return conference.hasAny(ConferenceParticipantStatus.ON_AIR, false);
    }

    @NonNull
    @NoDocumentation
    public Context getContext() {
        return mContext;
    }

    /**
     * Accesses the `ConferenceInformation` wrapper given the provided conference ID.
     *
     * @param conferenceId a non null conference id.
     * @return the `ConferenceInformation`.
     */
    @NonNull
    public ConferenceInformation getConferenceInformation(@NonNull String conferenceId) {
        return mConferenceInformationHolder.getInformation(conferenceId);
    }

    @NoDocumentation
    void joinLock() {
        try {
            joinLock.lock();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @NoDocumentation
    void joinUnlock() {
        try {
            if (joinLock.isLocked())
                joinLock.unlock();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void updateConferenceFromParticipants() {
        ConferenceInformation information = getCurrentConference();
        if (Opt.isNonNull(information)) {
            ConferenceStatus state = Opt.of(information).then(ConferenceInformation::getConferenceState).or(ConferenceStatus.DEFAULT);
            switch (state) {
                case JOINED:
                    if (hasParticipants())
                        information.setConferenceState(ConferenceStatus.FIRST_PARTICIPANT);
                    break;
                case FIRST_PARTICIPANT:
                    if (!hasParticipants())
                        information.setConferenceState(ConferenceStatus.NO_MORE_PARTICIPANT);
                    break;
                case NO_MORE_PARTICIPANT:
                    if (hasParticipants())
                        information.setConferenceState(ConferenceStatus.FIRST_PARTICIPANT);
                    break;
                default:
                    //nothing
            }
        }
    }

    @NoDocumentation
    void clearConferencesInformation() {
        this.mConferenceInformationHolder.clearConferencesInformation();
    }

    @NoDocumentation
    public void moveConference(@NonNull String conferenceId, @NonNull ConferenceInformation conferenceInformation) {
        Log.d(TAG, "moveConference: change " + conferenceInformation.getConference().getId() + " to " + conferenceId);
        mConferenceInformationHolder.moveConference(conferenceId, conferenceInformation);
    }

    /**
     * Submit a Create Answer For Peer into the new Threaded Pool
     *
     * @param media         the media instance to use
     * @param participantId the participantId to create the answer for
     * @param ssrc          the ssrc
     * @param offer         the actual offer
     * @param candidates    the candidates
     * @param master        is this peer a master one
     * @return the Promise to resolve to get the actual Sdpmessage from
     */
    private Promise<SdpMessage> createAnswerForPeerThreaded(@NonNull MediaSDK media,
                                                            @NonNull String participantId,
                                                            long ssrc,
                                                            @NonNull SdpDescription offer,
                                                            @NonNull List<SdpCandidate> candidates,
                                                            boolean master) {
        return new Promise<>(solver -> executorService.submit(() -> {
            if (!Opt.isNonNull(media)) { //TODO remove this, after testing tho
                Promise.reject(solver, new MediaEngineException("MediaEngine is null in createAnswerForPeerThreaded"));
                return;
            }

            try {
                media.createAnswerForPeer(participantId,
                        ssrc,
                        offer,
                        candidates,
                        master,
                        solver::resolve);
            } catch (MediaEngineException e) {
                e.printStackTrace();
                solver.reject(e);
            }
        }));
    }
}