package com.voxeet.sdk.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

import com.voxeet.promise.Promise;
import com.voxeet.sdk.media.MediaSDK;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.models.v1.ConferenceParticipantStatus;
import com.voxeet.sdk.models.v1.OfferCandidate;
import com.voxeet.sdk.models.v1.OfferDescription;
import com.voxeet.sdk.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.utils.Opt;

import org.greenrobot.eventbus.EventBus;

import java.util.List;

import retrofit2.Response;

public abstract class AbstractPromiseable<PROMISE_RETURN_TYPE, API_REST_CALL> {
    private final API_REST_CALL provider;
    private final ConferenceInformation information;
    private final EventBus eventBus;
    @NonNull
    private MediaDeviceService mediaDeviceService;
    @NonNull
    private ConferenceService parent;

    protected AbstractPromiseable(@NonNull ConferenceService parent,
                                  @NonNull MediaDeviceService mediaDeviceService,
                                  @NonNull API_REST_CALL provider,
                                  @Nullable ConferenceInformation information,
                                  @NonNull EventBus eventBus) {

        this.parent = parent;
        this.mediaDeviceService = mediaDeviceService;
        this.provider = provider;
        this.information = information;
        this.eventBus = eventBus;
    }

    @NonNull
    protected ConferenceService getParent() {
        return parent;
    }

    @NonNull
    protected MediaDeviceService getMediaService() {
        return mediaDeviceService;
    }

    @NonNull
    protected API_REST_CALL getApiRestCall() {
        return provider;
    }

    /**
     * Get the behold ConferenceInformation
     * <p>
     * in case of null, check for a possible current information afterward, in case of promise
     * while creating and joining but not in the conference
     *
     * @return an instance of ConferenceInformation or null
     */
    @Nullable
    protected ConferenceInformation getInformation() {
        if (null != information)
            return information;
        return parent.getCurrentConference();
    }

    @NonNull
    protected EventBus getEventBus() {
        return eventBus;
    }

    protected boolean isInConference() {
        return getParent().isInConference();
    }

    protected boolean hasMedia() {
        return getMediaService().hasMedia();
    }

    @Nullable
    protected MediaSDK getMedia() {
        return getMediaService().getMedia();
    }

    @NonNull
    public abstract Promise<PROMISE_RETURN_TYPE> createPromise();

    protected void log(@NonNull String log) {
        if (!TextUtils.isEmpty(log)) {
            Log.d(getClass().getSimpleName(), log);
        }
    }

    protected void setIsInConference(boolean status) {
        getParent().setIsInConference(status);
    }

    protected Promise<Boolean> createVideoAnswer(final String participantId,
                                                 final OfferDescription offerDescription,
                                                 final List<OfferCandidate> offerCandidates) {
        return getParent().createVideoAnswer(participantId,
                offerDescription,
                offerCandidates);
    }

    protected boolean isSDK() {
        return getParent().isSDK();
    }

    protected Participant updateConferenceParticipants(String participantId, ConferenceParticipantStatus status) {
        return getParent().updateConferenceParticipants(participantId, status);
    }

    protected void closeMedia() {
        getParent().closeMedia();
    }

    protected void removeTimeoutCallbacks() {
        getParent().removeTimeoutCallbacks();
    }

    protected void sendTimeoutCallbacks() {
        getParent().sendTimeoutCallbacks();
    }

    @NonNull
    protected ConferenceInformation createOrSetConferenceWithParams(@NonNull String conferenceId, @Nullable String conferenceAlias) {
        return getParent().createOrSetConferenceWithParams(conferenceId, conferenceAlias);
    }

    @NonNull
    protected ConferenceInformation getConferenceInformation(@NonNull String conferenceId) {
        return getParent().getConferenceInformation(conferenceId);
    }

    protected void moveConference(@NonNull String conferenceId, @NonNull ConferenceInformation conferenceInformation) {
        getParent().moveConference(conferenceId, conferenceInformation);
    }

    protected void joinLock() {
        getParent().joinLock();
    }

    protected void joinUnlock() {
        getParent().joinUnlock();
    }

    @NonNull
    protected Promise<Conference> joinConference(@NonNull ConferenceInformation conference) {
        return getParent().joinConferenceInternalPackage(conference);
    }

    protected void setCurrentConferenceIfNotInPreviousConference(@NonNull ConferenceInformation conference) {
        getParent().setCurrentConferenceIfNotInPreviousConference(conference);
    }

    protected boolean is200(@Nullable Response response) {
        return 200 == Opt.of(response).then(Response::code).or(0);
    }
}
