package com.voxeet.sdk.network.websocket;

import android.os.Handler;
import android.os.HandlerThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

import com.voxeet.sdk.authent.user_agent.UserAgentHelper;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.factories.EventsFactory;
import com.voxeet.sdk.services.VoxeetHttp;
import com.voxeet.sdk.services.authenticate.WebSocketState;

import java.util.ArrayList;
import java.util.List;

import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;

public class WebSocketProxy implements Runnable {
    private static final long DELAY_PING = 30 * 1000;//secondes - delay between various pings
    private static HandlerThread HANDLER_THREAD_SOCKETS = null;
    private final static Handler HANDLER = new Handler();
    private static final int MAXIMUM_RETRY = 60;
    private static final long SOCKET_RETRY_MAX = 2000; //10s
    private static final long SOCKET_RETRY_AFTER = 500; //0.5s
    private static final String TAG = WebSocketProxy.class.getSimpleName();
    private String mVersionName;

    private WebSocketState mState;
    private Handler mHandler;
    private String mSocketUrl;
    private int mCount = 0;
    private boolean shouldRetry = true;
    private WebSocketListener mAdapter;
    private List<SocketListener> mListeners;
    private long mSocketCurrentRetryDelay;
    private WebSocket mWebSocket;
    private ConnectListener mConnectListener;
    private boolean isCanceled = false;
    private VoxeetHttp mProvider;
    private int seq = 0;
    private boolean pinging = false;
    private Runnable pingRunnable = new Runnable() {
        @Override
        public void run() {
            if (pinging) {

                String json = EventsFactory.getJson(new KeepAlive(seq++));
                if (null == json) {
                    json = "{}";
                }
                Log.d(TAG, "run: sending ping : " + json);
                mWebSocket.send(json);

                if (pinging) {
                    mHandler.postDelayed(pingRunnable, DELAY_PING);
                }
            }
        }
    };

    public WebSocketProxy(@NonNull String socketUrl, String versionName) {
        mVersionName = versionName;
        mState = WebSocketState.CLOSED;
        mSocketUrl = socketUrl;

        if (null == HANDLER_THREAD_SOCKETS) {
            HANDLER_THREAD_SOCKETS = new HandlerThread("HANDLER_THREAD_SOCKETS");
            HANDLER_THREAD_SOCKETS.start();
        }

        mHandler = new Handler(HANDLER_THREAD_SOCKETS.getLooper());

        mListeners = new ArrayList<>();
        mSocketCurrentRetryDelay = SOCKET_RETRY_AFTER;
        mCount = 0;

        mAdapter = new WebSocketListener() {

            @Override
            public void onOpen(@NonNull WebSocket websocket, @NonNull Response response) {
                Log.d(TAG, "onOpen: websocket:=" + websocket + " response:=" + response);
                if (isCanceled) {
                    checkCancel();
                    return;
                }
                startPing();

                mWebSocket = websocket;

                mState = WebSocketState.CONNECTED;

                //reset values to let the app reconnect on error
                mSocketCurrentRetryDelay = SOCKET_RETRY_AFTER;
                mCount = 0;
                shouldRetry = true;

                try {
                    attemptConnectListenerConnected(websocket);
                } catch (Exception e) {
                    e.printStackTrace();
                }

                onStateChanged(websocket, mState);
                for (SocketListener listener : mListeners) {
                    try {
                        listener.onConnect(websocket);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

            @Override
            public void onMessage(@NonNull WebSocket websocket, @NonNull String message) {
                Log.d(TAG, "onMessage: websocket:=" + websocket + " message:=" + message);
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                for (SocketListener listener : mListeners) {
                    try {
                        listener.onTextMessage(message);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

            @Override
            public void onFailure(@NonNull WebSocket websocket, @NonNull Throwable cause, Response response) {
                Log.d(TAG, "onFailure: websocket:=" + websocket + " cause:=" + cause + " isCanceled" + isCanceled);
                if (isCanceled) {
                    checkCancel();
                    return;
                }
                stopPing();
                cause.printStackTrace();

                Log.d(TAG, "onClosed: cause:=" + cause);
                sendInitialFailed(response);
                checkRetry(websocket, cause.getMessage());
            }

            @Override
            public void onClosing(@NonNull WebSocket webSocket, int code, @NonNull String reason) {
                if (isCanceled) {
                    checkCancel();
                    return;
                }
                stopPing();
                mState = WebSocketState.CLOSING;
                //WebSocketProxy.this.onClosing();
            }

            @Override
            public void onClosed(@NonNull WebSocket webSocket, int code, @NonNull String reason) {
                Log.d(TAG, "onClosed: websocket:=" + webSocket + " code:=" + code + " reason" + reason);

                if (isCanceled) {
                    checkCancel();
                    return;
                }
                stopPing();

                Log.d(TAG, "onClosed: code:=" + code + " reason:=" + reason);
                //sendInitialFailed(null);
                checkRetry(webSocket, reason);
            }

            private void sendInitialFailed(@Nullable Response response) {
                //mHasBeenConnectedOrSentInitialConnectFail = true;
                HttpException error = HttpException.throwResponse(response);
                for (SocketListener listener : mListeners) {
                    try {
                        listener.onError(error);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

            private void checkRetry(@NonNull WebSocket websocket, @Nullable String reason) {
                Log.d(TAG, "onClosed: reason := " + reason + " shouldRetry:=" + shouldRetry);

                if (shouldRetry) {
                    if (mCount < MAXIMUM_RETRY) {

                        mState = WebSocketState.CONNECTING;
                        mCount++;
                        mSocketCurrentRetryDelay = Math.min(mSocketCurrentRetryDelay * 2, SOCKET_RETRY_MAX);
                        HANDLER.postDelayed(WebSocketProxy.this, mSocketCurrentRetryDelay);
                    } else {
                        mState = WebSocketState.CLOSED;
                        attemptConnectListenerDisconnected();
                        WebSocketProxy.this.onDisconnected();
                    }
                } else {
                    mState = WebSocketState.CLOSED;
                    onStateChanged(websocket, mState);
                    WebSocketProxy.this.onDisconnected();
                }
            }


            void onStateChanged(WebSocket websocket, WebSocketState newState) {
                if (isCanceled) {
                    checkCancel();
                    return;
                }

                for (SocketListener listener : mListeners) {
                    try {
                        listener.onStateChanged(newState);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        };
    }

    private void attemptConnectListenerDisconnected() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        if (mConnectListener != null) {
            mConnectListener.onConnectError(new HttpException((Response) null));
            mConnectListener = null;
        }
    }

    private void attemptConnectListenerConnected(WebSocket webSocket) {
        if (isCanceled) {
            checkCancel();
            return;
        }

        if (mConnectListener != null) {
            mConnectListener.onConnect(webSocket);
            mConnectListener = null;
        }
    }

    @Nullable
    private Request setWebSocketRequest() {
        if (isCanceled) {
            checkCancel();
            return null;
        }

        String socketUrl = mSocketUrl;

        String jwtToken = mProvider.getJwtToken();
        if (null != jwtToken) {
            if (!socketUrl.contains("?")) socketUrl += "?Token=" + jwtToken;
            else socketUrl += "Token=" + jwtToken;

            Log.d(TAG, "setWebSocketRequest: socketUrl := " + mSocketUrl + " has Token :=" + (!TextUtils.isEmpty(jwtToken)));

            return UserAgentHelper.setUserAgent(new Request.Builder(), mVersionName)
                    .url(socketUrl)
                    .build();
        }
        return null;
    }

    public void connect(@NonNull VoxeetHttp provider, @Nullable ConnectListener connectListener) {
        if (isCanceled) {
            checkCancel();
            return;
        }

        mConnectListener = connectListener;
        mProvider = provider;

        Request request = setWebSocketRequest();

        if (null != request) {
            mState = WebSocketState.CONNECTING;

            mWebSocket = provider.getClient().newWebSocket(request, mAdapter);
        }
    }

    private void onDisconnected() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        for (SocketListener listener : mListeners) {
            listener.onDisconnected();
        }
    }

    public void addListener(@NonNull SocketListener listener) {
        if (!mListeners.contains(listener)) {
            mListeners.add(listener);
        }
    }

    public void removeListener(@NonNull SocketListener listener) {
        mListeners.remove(listener);
    }

    @Override
    public void run() {
        if (isCanceled) {
            checkCancel();
            return;
        }

        connect(mProvider, mConnectListener);
    }

    public boolean isOpen() {
        return WebSocketState.CONNECTED.equals(mState);//mWebSocket != null && mWebSocket.isOpen();
    }

    public void disconnect() {
        stopPing();
        mState = WebSocketState.CLOSED;
        //prevent reconnection
        mCount = MAXIMUM_RETRY;
        shouldRetry = false;

        //disconnect
        if (null != mWebSocket) {
            mWebSocket.close(1000, "Closing socket from client");
            //mWebSocket.disconnect(WebSocketCloseCode.NORMAL, null, CLOSE_VOXEET_AFTER_MILLISECS);
        }

        for (SocketListener listener : mListeners) {
            listener.onClose();
        }
    }

    public void removeListeners() {
        mListeners.clear();
        mAdapter = null;
    }

    public WebSocketState getState() {
        return mState;
    }

    public WebSocket getWebSocket() {
        return mWebSocket;
    }

    private void checkCancel() {
        Log.d(TAG, "checkCancel: isCanceled:=" + isCanceled + " isOpen:=" + isOpen());
        if (isCanceled && isOpen()) {
            disconnect();
        }
    }

    public void cancel() {
        removeListeners();
        isCanceled = true;
    }

    private void startPing() {
        if (!pinging) {
            Log.d(TAG, "startPing");
            pinging = true;

            mHandler.postDelayed(pingRunnable, DELAY_PING);
        }
    }

    private void stopPing() {
        if (pinging) {
            Log.d(TAG, "stopPing");
            pinging = false;
            mHandler.removeCallbacks(pingRunnable);
        }
    }
}
