package com.voxeet.sdk.models.v1;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.voxeet.sdk.json.ParticipantInfo;
import com.voxeet.sdk.utils.ConferenceUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@JsonIgnoreProperties(ignoreUnknown = true)
public class Conference {
    @JsonIgnore
    private String TAG = Conference.class.getSimpleName();

    @JsonIgnore
    private boolean isOffline;

    @JsonIgnore
    private boolean isDemo = false;

    @JsonIgnore
    private boolean hasBeenCreated = false;

    @JsonIgnore
    private boolean isGuestAccess;

    @JsonIgnore
    private boolean isRecorded = false;

    @JsonIgnore
    private Date startRecordTimestamp;

    @JsonIgnore
    private String recordingUser;

    @JsonIgnore
    private RecordingStatus recordingStatus = RecordingStatus.NOT_RECORDING;

    @JsonProperty("conference")
    private ConferenceInfos conferenceInfos;

    private String conferenceId;

    private String conferenceAlias;

    private String conferenceType;

    private String securityToken;

    private String meetingId;

    private String location;

    private String title;

    private String description;

    private UserProfile ownerProfile;

    private String pstnPinCode;

    private List<ConferenceUser> conferenceUsers = new ArrayList<>();

    @JsonProperty("nodes")
    private List<PSTNItem> pstnItems;

    public Conference() {

    }

    @JsonProperty("e164FallbackPhoneNumber")
    private String fallBackNumber;

    public String getFallBackNumber() {
        return fallBackNumber;
    }

    public void setFallBackNumber(String fallBackNumber) {
        this.fallBackNumber = fallBackNumber;
    }

    public ConferenceInfos getConferenceInfos() {
        return conferenceInfos;
    }

    public void setConferenceInfos(ConferenceInfos conferenceInfos) {
        this.conferenceInfos = conferenceInfos;
    }

    public List<PSTNItem> getPstnItems() {
        return pstnItems;
    }

    public void setPstnItems(List<PSTNItem> pstnItems) {
        this.pstnItems = pstnItems;
    }

    public boolean hasBeenCreated() {
        return hasBeenCreated;
    }

    public void setHasBeenCreated(boolean hasBeenCreated) {
        this.hasBeenCreated = hasBeenCreated;
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public String getMeetingId() {
        return meetingId;
    }

    public void setMeetingId(String meetingId) {
        this.meetingId = meetingId;
    }

    public String getLocation() {
        return location;
    }

    public void setLocation(String location) {
        this.location = location;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public UserProfile getOwnerProfile() {
        return ownerProfile;
    }

    public void setOwnerProfile(UserProfile ownerProfile) {
        this.ownerProfile = ownerProfile;
    }

    public String getPstnPinCode() {
        return pstnPinCode;
    }

    public void setPstnPinCode(String pstnPinCode) {
        this.pstnPinCode = pstnPinCode;
    }

    public String getConferenceId() {
        return conferenceId;
    }

    public void setConferenceId(String conferenceId) {
        this.conferenceId = conferenceId;
    }

    public String getConferenceType() {
        return conferenceType;
    }

    public void setConferenceType(String conferenceType) {
        this.conferenceType = conferenceType;
    }

    public String getSecurityToken() {
        return securityToken;
    }

    @NonNull
    public List<ConferenceUser> getConferenceUsers() {
        if (null != conferenceUsers) {
            return conferenceUsers;
        }
        return new ArrayList<>();
    }

    public int getConferenceRoomSize() {
        if (conferenceUsers != null)
            return conferenceUsers.size();
        return 0;
    }

    public void setSecurityToken(String securityToken) {
        this.securityToken = securityToken;
    }

    public void setConferenceUsers(List<ConferenceUser> conferenceUsers) {
        this.conferenceUsers = conferenceUsers;
    }

    public boolean isOffline() {
        return isOffline;
    }

    public void setOffline(boolean offline) {
        isOffline = offline;
    }

    public boolean isDemo() {
        return isDemo;
    }

    public void setDemo(boolean demo) {
        isDemo = demo;
    }

    public boolean isGuestAccess() {
        return isGuestAccess;
    }

    public void setGuestAccess(boolean guestAccess) {
        isGuestAccess = guestAccess;
    }

    public boolean isRecorded() {
        return isRecorded;
    }

    public void setRecorded(boolean recorded) {
        isRecorded = recorded;
    }

    public Date getStartRecordTimestamp() {
        return startRecordTimestamp;
    }

    public void setStartRecordTimestamp(Date startRecordTimestamp) {
        this.startRecordTimestamp = startRecordTimestamp;
    }

    public String getRecordingUser() {
        return recordingUser;
    }

    public ConferenceUser getUserById(final String userId) {
        if (conferenceUsers != null) {
            return ConferenceUtils.findConferenceUserById(userId, conferenceUsers);
        }
        return null;
    }

    public void setRecordingUser(String recordingUser) {
        this.recordingUser = recordingUser;
    }

    public boolean isConferenceEmpty() {
        if (conferenceUsers == null)
            return true;

        for (ConferenceUser user : conferenceUsers) {
            if (user.getConferenceStatus() == ConferenceParticipantStatus.ON_AIR)
                return false;
        }
        return true;
    }

    public RecordingStatus getRecordingStatus() {
        return recordingStatus;
    }

    public boolean isUsersOnAir() {
        if (conferenceUsers == null)
            return true;

        for (ConferenceUser user : conferenceUsers) {
            if (user.getConferenceStatus() == ConferenceParticipantStatus.ON_AIR)
                return true;
        }
        return false;
    }

    public void setRecordingStatus(RecordingStatus recordingStatus) {
        this.recordingStatus = recordingStatus;
    }

    public void setConferenceAlias(String conferenceAlias) {
        this.conferenceAlias = conferenceAlias;
    }

    public String getConferenceAlias() {
        return this.conferenceAlias;
    }


    /**
     * Transform any given participant to conference participant
     * <p>
     * Get the ConferenceUser or null
     *
     * @param participant the participant to map
     * @return true if the participant has not been added
     */
    public boolean updateParticipantToConferenceUsers(@NonNull RestParticipant participant) {
        ConferenceUser user = findUserById(participant.getParticipantId());
        boolean found = null != user;

        if (null == user) {
            user = new ConferenceUser(participant.getParticipantId(),
                    participant.getDeviceType(),
                    new ParticipantInfo(participant.getName(),
                            participant.getExternalId(),
                            participant.getAvatarUrl()));
            getConferenceUsers().add(user);
            found = false;
        } else {
            user.updateIfNeeded(participant.getName(),
                    participant.getAvatarUrl());
        }
        //TODO check for any participant modification > name, avatarUrl

        ConferenceParticipantStatus status = ConferenceParticipantStatus.fromString(participant.getStatus());
        user.setConferenceStatus(status);

        return !found;
    }

    @Nullable
    public ConferenceUser findUserById(@NonNull final String userId) {
        return ConferenceUtils.findConferenceUserById(userId, getConferenceUsers());
    }

}
