package com.voxeet.sdk.models;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.sdk.VoxeetSdk;
import com.voxeet.sdk.json.ParticipantInfo;
import com.voxeet.sdk.models.v1.ConferenceInfos;
import com.voxeet.sdk.models.v1.ConferenceParticipantStatus;
import com.voxeet.sdk.models.v1.ConferenceUser;
import com.voxeet.sdk.models.v1.RecordingStatus;
import com.voxeet.sdk.models.v1.RestParticipant;
import com.voxeet.sdk.services.SessionService;
import com.voxeet.sdk.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.services.conference.information.ConferenceStatus;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import java.util.Date;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.ReentrantLock;

/**
 * The Conference model allows the application to get information about the conference, such as:
 * - [conference ID](/documentation/sdk/reference/android/models/conference#getid)
 * - [conference alias](/documentation/sdk/reference/android/models/conference#getalias)
 * - reference to the conference [participants](/documentation/sdk/reference/android/models/conference#getparticipants)
 * - [raw information](/documentation/sdk/reference/android/models/conference#getconferenceinfos) about the conference
 * - [information and the recording state](/documentation/sdk/reference/android/models/conference#getrecordinginformation) of the current conference
 * - the current [conference state](/documentation/sdk/reference/android/models/conference#getstate)
 *
 * It also checks if :
 * - the conference participants are in a specific [state](/documentation/sdk/reference/android/models/conference#hasany)
 * - the Conference instance has the [locally-based MediaStream](/documentation/sdk/reference/android/models/conference#haslocalstreams)
 * - there are [mixer participants](/documentation/sdk/reference/android/models/conference#hasmixer) at the conference.
 *
 * It also [looks for specific users](/documentation/sdk/reference/android/models/conference#finduserbyexternalid) at the conference.
 */
@Annotate
public class Conference {

    private ReentrantLock lock = new ReentrantLock();

    private static final String TAG = "Conference";
    @NonNull
    private String id;

    @Nullable
    private String alias;

    @NonNull
    private CopyOnWriteArrayList<Participant> participants;

    @Nullable
    private Participant mixer;

    @Nullable
    private ConferenceInfos conferenceInfos;

    @Nullable
    private RecordingInformation recordingInformation;

    @NonNull
    private ConferenceInformation conferenceInformation;

    private Conference() {

    }

    @NoDocumentation
    public Conference(@NonNull ConferenceInformation conferenceInformation) {
        participants = new CopyOnWriteArrayList<>();
        id = "";
        this.conferenceInformation = conferenceInformation;
    }

    @NoDocumentation
    public Conference(com.voxeet.sdk.models.v1.Conference fromConference, @NonNull ConferenceInformation conferenceInformation) {
        this();

        updateParticipants(fromConference.getConferenceUsers());
        id = fromConference.getConferenceId();
        alias = fromConference.getConferenceAlias();
        this.conferenceInformation = conferenceInformation;
    }

    /**
     * Gets the conference ID.
     *
     * @return the conference ID.
     */
    @NonNull
    public String getId() {
        return id;
    }

    /**
     * Gets the conference alias that is set by a developer.
     *
     * @return the conference alias.
     */
    @Nullable
    public String getAlias() {
        return alias;
    }

    /**
     * Gets the reference to the participants in the conference.
     *
     * @return the direct reference to the array of participants.
     */
    @NonNull
    public CopyOnWriteArrayList<Participant> getParticipants() {
        return participants;
    }

    /**
     * Checks if in the current conference there are participants in one of the following statuses:
     * - on-air participants
     * - left participants
     * - invited
     *
     * @param status The participant's status.
     * @param local  Information that the search includes only local participants.
     * @return at least one participant with the corresponding state.
     */
    public boolean hasAny(@NonNull ConferenceParticipantStatus status, boolean local) {
        String id = VoxeetSdk.session().getParticipantId();
        if (null == id) id = "";

        for (Participant participant : participants) {
            if (null != participant && status.equals(participant.getStatus()) && (local || !id.equals(participant.getId())))
                return true;
        }
        return false;
    }

    /**
     * Checks if the Conference instance has locally-based MediaStream.
     *
     * @param includeLocalParticipant Information that the search includes only local participants.
     * @return the indicator of the local streams that is in 'connecting' or 'on air' state.
     */
    public boolean hasLocalStreams(boolean includeLocalParticipant) {
        String id = VoxeetSdk.session().getParticipantId();
        if (null == id) id = "";

        for (Participant participant : participants) {
            if (null != participant) {
                boolean properStatus = ConferenceParticipantStatus.ON_AIR.equals(participant.getStatus());
                properStatus |= ConferenceParticipantStatus.CONNECTING.equals(participant.getStatus());
                if (properStatus && participant.streams().size() > 0 && (includeLocalParticipant || !id.equals(participant.getId())))
                    return true;
            }
        }
        return false;
    }

    /**
     * Update a given participant in the conference
     * to be used by the SDK
     *
     * @param participant a raw participant to add or update
     * @return the current reference to the conference
     */
    @NoDocumentation
    @NonNull
    public Conference updateParticipants(ConferenceUser participant) {
        lock();
        Participant internalParticipant = getInternalParticipantInConference(participant);
        if (null == internalParticipant) {
            log("updateParticipants: create RestParticipant from ConferenceUser := " + new Participant(participant));
            addParticipant(new Participant(participant));
        } else {
            internalParticipant.updateStatus(participant.getConferenceStatus());
        }
        unlock();
        return this;
    }

    /**
     * Update a given participant in the conference
     * to be used by the SDK
     *
     * @param participant a raw participant to add or update
     * @return the current reference to the conference
     */
    @NoDocumentation
    @NonNull
    public Conference updateParticipant(@NonNull RestParticipant participant) {
        Participant internalParticipant = getInternalParticipantInConference(participant);
        lock();
        if (null == internalParticipant) {
            String externalId = participant.getExternalId();
            Participant existing = findParticipantByExternalId(externalId);
            if (null != existing) {
                log("updateParticipant: existing participant, updating hi·er");
                existing.updateIfNeeded(participant.getName(), participant.getAvatarUrl());
            } else {
                log("updateParticipant: new participant, adding hi·er, ");
                addParticipant(new Participant(participant));
            }
        } else {
            internalParticipant.updateStatus(ConferenceParticipantStatus.fromString(participant.getStatus()));
        }
        unlock();
        return this;
    }

    /**
     * Update a given participant in the conference
     * to be used by the SDK
     *
     * @param participant a raw participant to add or update
     * @return the current reference to the conference
     */
    @NoDocumentation
    @NonNull
    public Conference updateParticipant(Participant participant) {
        lock();
        Participant internalParticipant = getInternalParticipantInConference(participant);
        if (null == internalParticipant) {
            log("updateParticipants: create RestParticipant from RestParticipant := " + participant);
            addParticipant(participant);
        } else {
            internalParticipant.updateStatus(participant.getStatus());
        }
        unlock();
        return this;
    }

    /**
     * Update a list of participants in the conference
     * to be used by the SDK
     *
     * @param users a list of raw participant to add or update
     * @return the current reference to the conference
     */
    @NoDocumentation
    @NonNull
    public Conference updateParticipants(List<ConferenceUser> users) {
        lock();
        try {
            for (ConferenceUser user : users) updateParticipants(user);
        } catch (Exception e) {
            e.printStackTrace();
        }
        unlock();
        return this;
    }

    /**
     * Update a given participant in the conference
     * to be used by the SDK
     *
     * @param participants a list of raw participant to add or update
     * @return the current reference to the conference
     */
    @NoDocumentation
    @NonNull
    public Conference updateRestParticipants(List<RestParticipant> participants) {
        try {
            for (RestParticipant participant : participants) updateParticipant(participant);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return this;
    }

    @Nullable
    private Participant getInternalParticipantInConference(ConferenceUser participant) {
        try {
            for (Participant u : participants) if (u.equals(participant)) return u;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    @Nullable
    private Participant getInternalParticipantInConference(Participant participant) {
        try {
            for (Participant u : participants) if (u.equals(participant)) return u;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    @Nullable
    private Participant getInternalParticipantInConference(RestParticipant participant) {
        try {
            for (Participant u : participants) if (u.equals(participant)) return u;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Set the current conference id
     * to be used by the SDK
     *
     * @param id
     * @return the current reference to the conference
     */
    @NoDocumentation
    public Conference setConferenceId(@NonNull String id) {
        this.id = id;
        return this;
    }

    @NoDocumentation
    public Conference setConferenceAlias(String alias) {
        this.alias = alias;
        return this;
    }

    /**
     * Checks if the current Conference has a Mixer participant currently connected. It helps to filter out such participants into the UI or add a label.
     * It returns a true value when a Mixer participant exists in the conference.
     *
     * @return the presence of an on-air mixer participant at the conference.
     */
    public boolean hasMixer() {
        return null != mixer && ConferenceParticipantStatus.ON_AIR.equals(mixer.getStatus());
    }

    private boolean isMixer(@NonNull Participant participant) {
        ParticipantInfo participantInfo = participant.getInfo();
        String name = null != participantInfo ? participantInfo.getName() : null;

        return null != name && "Mixer".equalsIgnoreCase(name.trim());
    }

    /**
     * Filters participants to find the one with the desired Voxeet `userId`. A null value informs that such a participant does not exist in a conference.
     *
     * @param participantId The valid ID of the conference participant.
     * @return the instance of the participant or a null value.
     */
    @Nullable
    public Participant findParticipantById(@Nullable String participantId) {
        if (null == participantId) return null;
        for (Participant participant : participants) {
            if (participantId.equals(participant.getId())) return participant;
        }

        SessionService sessionService = VoxeetSdk.session();
        if (null == sessionService) return null;

        String localParticipantId = sessionService.getParticipantId();
        if (null != localParticipantId && localParticipantId.equals(participantId)) {
            Log.d(TAG, "findParticipantById: participant not found but local one, creating a TEMP ONE ONLY");
            return sessionService.getParticipant();
        }
        return null;
    }

    /**
     * Filters participants to find the one with a desired external ID `externalId`. A null value informs that such participant does not exist in a conference.
     *
     * @param externalId The valid external ID of the conference participant.
     * @return the instance of the participant or a null value.
     */
    @Nullable
    public Participant findParticipantByExternalId(@Nullable String externalId) {
        if (null == externalId) return null;
        for (Participant participant : participants) {
            ParticipantInfo participantInfo = participant.getInfo();
            if (null != participantInfo && externalId.equals(participantInfo.getExternalId())) return participant;
        }
        return null;
    }

    @NoDocumentation
    public void setConferenceInfos(@NonNull ConferenceInfos conferenceInfos) {
        this.conferenceInfos = conferenceInfos;
    }

    /**
     * Gets a raw information about the conference.
     *
     * @return the information or a null value if it does not exist.
     */
    @Nullable
    public ConferenceInfos getConferenceInfos() {
        return conferenceInfos;
    }

    /**
     * Set the current recording information state
     * to be used internally by the SDK
     *
     * @param recordingInformation the new information
     */
    @NoDocumentation
    public void setRecordingInformation(@Nullable RecordingInformation recordingInformation) {
        this.recordingInformation = recordingInformation;
    }

    /**
     * Gets information and a state about the current conference recording.
     *
     * @return the reference to the recording information.
     */
    @Nullable
    public RecordingInformation getRecordingInformation() {
        return recordingInformation;
    }

    /**
     * Add a given participant to the list of participants
     * <p>
     * Warning : this does not manage the case where multiple mixers could be added - bug?
     * or even mix with how developers would create participants... mixing participant with the mixer
     *
     * @param participant a newly instance to add
     */
    private void addParticipant(@NonNull Participant participant) {
        dumpParticipants();
        if (isMixer(participant)) mixer = participant;
        else participants.add(participant);
    }

    private void dumpParticipants() {
        StringBuilder u = new StringBuilder();
        for (Participant participant : participants) u.append(participant.getId()).append(",");
        log(u.toString());
    }

    /**
     * Informs about the current conference state.
     *
     * @return the valid ConferenceState for a manipulation.
     */
    @NonNull
    public ConferenceStatus getState() {
        return conferenceInformation.getConferenceState();
    }

    public static class RecordingInformation {

        private Date startRecordTimestamp;
        private RecordingStatus recordingStatus;
        private String recordingParticipant;

        public void setStartRecordTimestamp(Date startRecordTimestamp) {
            this.startRecordTimestamp = startRecordTimestamp;
        }

        public Date getStartRecordTimestamp() {
            return startRecordTimestamp;
        }

        public void setRecordingStatus(RecordingStatus recordingStatus) {
            this.recordingStatus = recordingStatus;
        }

        public RecordingStatus getRecordingStatus() {
            return recordingStatus;
        }

        public void setRecordingParticipant(String recordingParticipant) {
            this.recordingParticipant = recordingParticipant;
        }

        public String getRecordingParticipant() {
            return recordingParticipant;
        }
    }

    private void log(@NonNull String string) {
        Log.d(TAG, "log: " + string);
    }

    private void lock() {
        try {
            lock.lock();
        } catch (Exception e) {

        }
    }

    private void unlock() {
        try {
            if (lock.isLocked()) lock.unlock();
        } catch (Exception e) {

        }
    }
}
