package com.voxeet.sdk.services;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.audio2.devices.MediaDevice;
import com.voxeet.audio2.devices.description.ConnectionState;
import com.voxeet.audio2.devices.description.DeviceType;
import com.voxeet.promise.Promise;
import com.voxeet.promise.solve.ThenVoid;
import com.voxeet.sdk.VoxeetSdk;
import com.voxeet.sdk.events.sdk.ConferenceStatusUpdatedEvent;
import com.voxeet.sdk.json.ConferenceDestroyedPush;
import com.voxeet.sdk.json.ConferenceEnded;
import com.voxeet.sdk.media.audio.AudioRoute;
import com.voxeet.sdk.media.audio.SoundManager;
import com.voxeet.sdk.media.sensors.ConferenceLock;
import com.voxeet.sdk.media.sensors.ProximitySensor;
import com.voxeet.sdk.media.sensors.ScreenSensor;
import com.voxeet.sdk.services.conference.information.ConferenceStatus;
import com.voxeet.sdk.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.AudioType;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.Opt;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.voiceengine.WebRtcAudioUtils;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;

/**
 * The AudioService allows an application to get an audio connection and manage it.
 * <p>
 * **In this service the application:**
 * <p>
 * - Calls the [getSoundManager](/documentation/sdk/reference/android/audio#getsoundmanager) method to retrieve the static [SoundManager](/documentation/sdk/reference/android/models/soundmanager). Then, it initializes it through the [preInitSounds](/documentation/sdk/reference/android/audio#preinitsounds) method.
 * <p>
 * - Calls the [currentRoute](/documentation/sdk/reference/android/audio#currentroute) or [getAvailableRoutes](/documentation/sdk/reference/android/audio#getavailableroutes) method to get the current or available routes. It can also change audio routes by calling the [setAudioRoute](/documentation/sdk/reference/android/audio#setaudioroute). When the audio route is changed, the application is notified about it via the [AudioRouteChangeEvent](/documentation/sdk/reference/android/audio#audioroutechangeevent). The [checkOutputRoute](/documentation/sdk/reference/android/audio#checkoutputroute) method enables checking route side effects for internal management.
 * <p>
 * - Can ask for a focus on currently available or valid audio routes through the [requestAudioFocus](/documentation/sdk/reference/android/audio#requestaudiofocus) method and removes this focus by calling the [abandonAudioFocusRequest](/documentation/sdk/reference/android/audio#abandonaudiofocusrequest).
 * <p>
 * - Can update the state of audio route sensors through the [updateSensors](/documentation/sdk/reference/android/audio#updatesensors) method and releases the sensors by using the [releaseSensors](/documentation/sdk/reference/android/audio#releasesensors) method.
 * <p>
 * - Can call the [enableMedia](/documentation/sdk/reference/android/audio#enablemedia) to enable the media mode, it calls the [setMediaRoute](/documentation/sdk/reference/android/audio#setmediaroute) to set this mode, and the [unsetMediaRoute](/documentation/sdk/reference/android/audio#unsetmediaroute) to unset it.
 * <p>
 * - Can call the [enable](/documentation/sdk/reference/android/audio#enable) method to start audio service. It can also [unable](/documentation/sdk/reference/android/audio#unable) it and [stop](/documentation/sdk/reference/android/audio#stop) it.
 * <p>
 * - Can call the [playSoundType](/documentation/sdk/reference/android/audio#playsoundtype) method to play audio and calls the [stopSoundType](/documentation/sdk/reference/android/audio#stopsoundtype) to stop it.
 * <p>
 * - Can call the [setSound](/documentation/sdk/reference/android/audio#setsound) method to attach assets.
 * <p>
 * - Can call the [resetDefaultSoundType](/documentation/sdk/reference/android/audio#resetdefaultsoundtype) method to reset the system audio for media or a system.
 * <p>
 * - Can activate or deactivate the AEC mode through the [enableAec](/documentation/sdk/reference/android/audio#enableaec) method. It can also activate or deactivate the NS mode through the [enableNoiseSuppressor](/documentation/sdk/reference/android/audio#enablenoisesuppressor).
 * <p>
 * - Can acquire or release the system locks of the sensors through the [acquireLocks](/documentation/sdk/reference/android/audio#acquirelocks) and [releaseLocks](/documentation/sdk/reference/android/audio#releaselocks) methods.
 * <p>
 * - Can call the [isSpeakerOn](/documentation/sdk/reference/android/audio#isspeakeron) method to check the speaker mode and can set the speaker mode by calling the [setSpeakerMode](/documentation/sdk/reference/android/audio#setspeakermode).
 * <p>
 * - Can get the current Bluetooth and headset connections through the [isBluetoothHeadsetConnected](/documentation/sdk/reference/android/audio#isbluetoothheadsetconnected) and [isWiredHeadsetOn](/documentation/sdk/reference/android/audio#iswiredheadseton) methods.
 */
@Annotate
public class AudioService extends AbstractVoxeetService {

    private static SoundManager sSoundManager;
    private static final String TAG = AudioService.class.getSimpleName();
    private List<ConferenceLock> locks = new ArrayList<>();
    private Boolean speakerMode = null;

    /**
     * Instantiate a new Audio Service
     *
     * @param instance the voxeet parent instance
     */
    @NoDocumentation
    public AudioService(@NonNull SdkEnvironmentHolder instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        AudioService.preInitSounds(instance.voxeetSdk.getApplicationContext());

        sSoundManager.registerUpdateDevices(this::tryConnectFirst);
        registerEventBus();

        locks.add(new ProximitySensor(instance.voxeetSdk.getApplicationContext()));
        locks.add(new ScreenSensor(instance.voxeetSdk.getApplicationContext()));
    }

    /**
     * Retrieves the static [SoundManager](/documentation/sdk/reference/android/models/soundmanager).
     *
     * @return the instance of the SoundManager or a null value if it is not initialized.
     */
    @Nullable
    public static SoundManager getSoundManager() {
        return sSoundManager;
    }

    /**
     * Initializes the [SoundManager](/documentation/sdk/reference/android/models/soundmanager) in cases when the SDK is not initialized. For example, when Cordova or React Native is used.
     *
     * @param applicationContext a valid instance
     * @return the boolean indicating if the initialization has just happened or was initialized before using this method.
     */
    public static boolean preInitSounds(@NonNull Context applicationContext) {
        if (null == sSoundManager) {
            sSoundManager = new SoundManager(applicationContext);
            return true;
        }

        return false;
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceStatusUpdatedEvent event) {
        if (ConferenceStatus.JOINING.equals(event.state)) {
            enableAec(true);
            enableNoiseSuppressor(true);

            updateSensors(currentRoute());
            sSoundManager.onConferencePreJoinedEvent();
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceEnded event) {
        sSoundManager.onConferenceDestroyedPush();
        releaseSensors();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceDestroyedPush event) {
        sSoundManager.onConferenceDestroyedPush();
        releaseSensors();
    }

    /**
     * Gets the available routes.
     *
     * @return the list of all possible routes.
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        return sSoundManager.getAvailableRoutes();
    }

    /**
     * Gets the current route used by the [SoundManager](/documentation/sdk/reference/android/models/soundmanager) instance.
     *
     * @return the currently selected route.
     */
    @NonNull
    public AudioRoute currentRoute() {
        return sSoundManager.currentRoute();
    }


    public boolean registerUpdateDevices(SoundManager.Call<List<MediaDevice>> callback) {
        return sSoundManager.registerUpdateDevices(callback);
    }

    public void unregisterUpdateDevices(SoundManager.Call<List<MediaDevice>> callback) {
        sSoundManager.unregisterUpdateDevices(callback);
    }

    public boolean isNotFiltered(@NonNull MediaDevice device) {
        return sSoundManager.isNotFiltered(device);
    }

    @NonNull
    public Promise<List<MediaDevice>> enumerateDevices() {
        return sSoundManager.enumerateDevices();
    }

    @NonNull
    public Promise<Boolean> connect(@NonNull MediaDevice device) {
        return sSoundManager.connect(device);
    }

    @NonNull
    public Promise<Boolean> disconnect(@NonNull MediaDevice device) {
        return sSoundManager.disconnect(device);
    }

    /**
     * Changes the audio route and updates various states of sensors.
     *
     * @param route the new route to use
     * @return the boolean indicating if a switch was made.
     */
    @Deprecated
    public boolean setAudioRoute(@NonNull AudioRoute route) {
        boolean new_switch = sSoundManager.setAudioRoute(route);
        updateSensors(currentRoute());

        return new_switch;
    }

    /**
     * Checks route side effects for internal management.
     *
     * @return the current instance.
     */
    @NonNull
    public AudioService checkOutputRoute() {
        sSoundManager.checkOutputRoute();
        return this;
    }

    /**
     * Changes the speaker mode.
     *
     * @param isSpeaker activate or deactivate the speakers.
     */
    @Deprecated
    public void setSpeakerMode(boolean isSpeaker) {
        speakerMode = isSpeaker;
        sSoundManager.setSpeakerMode(isSpeaker);
        updateSensors(currentRoute());
    }

    /**
     * Gets the current Bluetooth device connection.
     *
     * @return the boolean indicating if the device is currently connected or not.
     */
    @Deprecated
    public boolean isBluetoothHeadsetConnected() {
        return sSoundManager.isBluetoothHeadsetConnected();
    }

    /**
     * Gets the current wired headset connection state.
     *
     * @return the boolean indicating if wired headsets are connected to the device.
     */
    @Deprecated
    public boolean isWiredHeadsetOn() {
        return sSoundManager.isWiredHeadsetOn();
    }

    /**
     * Checks the speaker mode.
     *
     * @return the boolean indicating if the speakers are activated.
     */
    @Deprecated
    public boolean isSpeakerOn() {
        return sSoundManager.isSpeakerOn();
    }

    /**
     * Acquires the system locks of the sensors.
     *
     * @return the current instance.
     */
    @NonNull
    public AudioService acquireLocks() {
        sSoundManager.acquireLocks();
        return this;
    }

    /**
     * Releases the system locks of the sensors
     *
     * @return the current instance.
     */
    @NonNull
    public AudioService releaseLocks() {
        sSoundManager.releaseLocks();
        return this;
    }

    /**
     * Resets the system audio for media or the system.
     *
     * @return the current instance.
     */
    @Deprecated
    @NonNull
    public AudioService resetDefaultSoundType() {
        sSoundManager.resetDefaultSoundType();
        return this;
    }

    /**
     * Removes the focus on the currently available or valid audio route.
     *
     * @return the current instance.
     */
    @NonNull
    public AudioService abandonAudioFocusRequest() {
        sSoundManager.abandonAudioFocusRequest();
        return this;
    }

    /**
     * Asks for a focus on the current available or valid audio route.
     *
     * @return the current instance.
     */
    @NonNull
    public AudioService requestAudioFocus() {
        sSoundManager.requestAudioFocus();
        return this;
    }

    /**
     * Attaches the specific asset from the sound type name into the phone mode.
     *
     * @param type      the Sound Type to attach to
     * @param assetName the asset's name
     * @return information about the successful attach.
     */
    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return sSoundManager.setSound(type, assetName);
    }

    /**
     * Attaches the specific asset from a sound type name into the specific sound mode pool.
     *
     * @param type      the Sound Type to attach to
     * @param assetName the asset's name
     * @param soundMode the sound mode which will reflect the pool mode to link
     * @return information about the successful attach.
     */
    public boolean setSound(@NonNull AudioType type, @NonNull String assetName, int soundMode) {
        return sSoundManager.setSound(type, assetName, soundMode);
    }

    /**
     * Plays the specific type of audio in the phone mode.
     *
     * @param type the AudioType to play
     * @return the current instance.
     */
    @NonNull
    public AudioService playSoundType(@NonNull AudioType type) {
        sSoundManager.playSoundType(type);
        return this;
    }

    /**
     * Plays the given audio type.
     *
     * @param type      the AudioType to play
     * @param soundMode the sound mode to use
     * @return the current instance.
     */
    @NonNull
    public AudioService playSoundType(@NonNull AudioType type, int soundMode) {
        sSoundManager.playSoundType(type, soundMode);
        return this;
    }

    /**
     * Stops the given type of audio.
     *
     * @param audioType the requested type
     * @return the current instance.
     */
    @NonNull
    public AudioService stopSoundType(@NonNull AudioType audioType) {
        sSoundManager.stopSoundType(audioType);
        return this;
    }

    /**
     * Stops the given type and mode of audio.
     *
     * @param audioType the requested type
     * @param soundMode the mode it was working on
     * @return the current instance.
     */
    @NonNull
    public AudioService stopSoundType(@NonNull AudioType audioType, int soundMode) {
        sSoundManager.stopSoundType(audioType, soundMode);
        return this;
    }

    /**
     * Stops the service but does not free the holds or resources.
     *
     * @return the current instance.
     */
    @NonNull
    public AudioService stop() {
        sSoundManager.stop();
        return this;
    }

    /**
     * Enables the service and lets it manage sounds and mode modifications.
     *
     * @return the current instance.
     */
    @NonNull
    public AudioService enable() {
        sSoundManager.enable();
        return this;
    }

    /**
     * Enables the media mode in the manager to accept implementation features.
     *
     * @return the current instance.
     */
    @NonNull
    @Deprecated
    public AudioService enableMedia() {
        sSoundManager.enableMedia();
        return this;
    }

    /**
     * Unsets the media mode without restoring the phone mode. It does not hold any specific internal features.
     *
     * @return the current instance.
     */
    @Deprecated
    @NonNull
    public AudioService unsetMediaRoute() {
        sSoundManager.unsetMediaRoute();
        return this;
    }

    /**
     * Changes the Audio output to the Media mode (instead of the Phone mode).
     *
     * @return the current instance.
     */
    @Deprecated
    @NonNull
    public AudioService setMediaRoute() {
        sSoundManager.setMediaRoute();
        return this;
    }

    /**
     * Disables the AudioService and stops elements inside it.
     *
     * @return the current instance.
     */
    @NonNull
    public AudioService disable() {
        sSoundManager.disable();
        stop();
        return this;
    }

    /**
     * Updates state of sensors given in the specified audio route.
     *
     * @param route a valid route to update the sensors with
     */
    public void updateSensors(@NonNull AudioRoute route) {
        for (ConferenceLock lock : locks) {
            if (!route.useProximitySensor() && lock.isProximity()) {
                lock.release();
            } else {
                lock.acquire();
            }
        }
    }

    /**
     * Releases the sensors.
     */
    public void releaseSensors() {
        Log.d("ProximitySensor", "releaseSensors: ");
        for (ConferenceLock lock : locks) {
            lock.release();
        }
    }

    /**
     * Activates or deactivates the AEC mode.
     *
     * @param enable the new state to use
     * @return success or a failure.
     */
    public boolean enableAec(boolean enable) {
        WebRtcAudioUtils.setWebRtcBasedAcousticEchoCanceler(enable);
        return true;
    }

    /**
     * Activates or deactivates the NS mode.
     *
     * @param enable the new state to use
     * @return success or a failure.
     */
    public boolean enableNoiseSuppressor(boolean enable) {
        WebRtcAudioUtils.setWebRtcBasedNoiseSuppressor(enable);
        return true;
    }

    public void checkDevicesToConnectOnConference(boolean speaker) {
        if (VoxeetSdk.conference().isLive()) {
            speakerMode = speaker;
            enumerateDevices().then((ThenVoid<List<MediaDevice>>) m -> this.tryConnectFirst(m, speaker))
                    .error(Throwable::printStackTrace);
        }
    }

    private void tryConnectFirst(List<MediaDevice> mediaDevices) {
        tryConnectFirst(mediaDevices, null);
    }

    private void tryConnectFirst(List<MediaDevice> mediaDevices, Boolean speaker) {
        boolean hasConnecting = false;
        //get the first device that is currently connected
        MediaDevice internal = null;
        MediaDevice current = null;
        for (MediaDevice device : mediaDevices) {
            if (DeviceType.INTERNAL_SPEAKER.equals(device.deviceType())) internal = device;
            switch (device.connectionState()) {
                case CONNECTING:
                    hasConnecting = true;
                    break;
                case CONNECTED:
                    if (!DeviceType.NORMAL_MEDIA.equals(device.deviceType())) {
                        current = device;
                    }
            }
        }

        if (!VoxeetSdk.conference().isLive()) {
            Log.d(TAG, "update of devices without a conference... returning...");

            if (null != current && !ConnectionState.DISCONNECTING.equals(current.connectionState()) && ConnectionState.CONNECTED.equals(current.connectionState())) {
                disconnect(current).then(c -> {
                }).error(Throwable::printStackTrace);
            }
            return;
        }

        com.voxeet.audio.utils.Log.d(TAG, "speakerMode ? " + speakerMode + " speaker ? " + speaker);

        if (null != current && ConnectionState.DISCONNECTED.equals(current.platformConnectionState())) {
            current = null;
        }
        com.voxeet.audio.utils.Log.d(TAG, "current ? " + current + " hasConnecting ? " + hasConnecting);

        //if there are no connecting devices
        if (!hasConnecting) {
            DeviceType type = Opt.of(current).then(MediaDevice::deviceType).or(DeviceType.INTERNAL_SPEAKER);
            List<MediaDevice> bluetooths = ofType(mediaDevices, DeviceType.BLUETOOTH);
            List<MediaDevice> wireds = ofType(mediaDevices, DeviceType.WIRED_HEADSET);
            List<MediaDevice> external = ofType(mediaDevices, DeviceType.EXTERNAL_SPEAKER);
            List<MediaDevice> not_speaker = ofType(mediaDevices, DeviceType.INTERNAL_SPEAKER);

            Promise<Boolean> connect = null;
            if (!DeviceType.BLUETOOTH.equals(type) && bluetooths.size() > 0) {
                com.voxeet.audio.utils.Log.d(TAG, "has bluetooth to connect to " + bluetooths);
                connect = connect(bluetooths.get(0));
            } else if (!DeviceType.WIRED_HEADSET.equals(type) && wireds.size() > 0) {
                connect = connect(wireds.get(0));
            }
            if (null != speaker) {
                speakerMode = speaker;
            }
            if (null == connect && null != speakerMode) {
                com.voxeet.audio.utils.Log.d(TAG, "connecting to speaker ? ");
                if (null != current) {
                    com.voxeet.audio.utils.Log.d(TAG, "have a current device");
                } else if (speakerMode && !DeviceType.EXTERNAL_SPEAKER.equals(type) && external.size() > 0) {
                    connect = connect(external.get(0));
                } else if (not_speaker.size() > 0) {
                    connect = connect(not_speaker.get(0));
                }
            }

            com.voxeet.audio.utils.Log.d(TAG, "connecting to a device ... " + connect + " " + speakerMode);

            if (null != connect) {
                connect.then(result -> {
                    com.voxeet.audio.utils.Log.d("AudioService", "finished connecting automatically to the device... " + currentRoute());
                    updateSensors(currentRoute());
                }).error(Throwable::printStackTrace);
            } else {
                com.voxeet.audio.utils.Log.d("AudioService", "update simply the route " + currentRoute());
                updateSensors(currentRoute());
            }
        } else {
            com.voxeet.audio.utils.Log.d(TAG, "has connecting in progress");
        }
    }

    private List<MediaDevice> ofType(List<MediaDevice> mediaDevices, DeviceType type) {
        List<MediaDevice> list = new ArrayList<>();
        for (MediaDevice device : mediaDevices) {
            if (type.equals(device.deviceType()) && isNotFiltered(device) && ConnectionState.CONNECTED.equals(device.platformConnectionState()))
                list.add(device);
        }
        return list;
    }

    public Promise<MediaDevice> currentMediaDevice() {
        return sSoundManager.current();
    }
}
