package com.voxeet.sdk.models.v2;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.MediaStreamType;
import com.voxeet.sdk.models.Participant;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import java.util.List;

/**
 * The ParticipantMediaStreamHandler model manages media streams for the specific conference participants. It [checks](/documentation/sdk/reference/android/models/participantmediastreamhandler#has) their type, [gets first streams](/documentation/sdk/reference/android/models/participantmediastreamhandler#getfirst), [inserts or updates](/documentation/sdk/reference/android/models/participantmediastreamhandler#insertorupdate) them, and removes the [specific stream](/documentation/sdk/reference/android/models/participantmediastreamhandler#remove) or [all streams](/documentation/sdk/reference/android/models/participantmediastreamhandler#removeallstreams).
 *
 * It assumes that the RestParticipant.streams method resolves references for RestParticipant streams.
 */
@Annotate
public class ParticipantMediaStreamHandler {
    private Participant participant;

    @NoDocumentation
    public ParticipantMediaStreamHandler(@NonNull Participant participant) {
        this.participant = participant;
    }

    /**
     * Checks if a specific MediaStream Type exists in the current participant lifecycle.
     * @param type The type of MediaStream.
     * @return the search result.
     */
    public boolean has(@NonNull MediaStreamType type) {
        for (MediaStream stream : participant.streams()) {
            if (type.equals(stream.getType())) return true;
        }
        return false;
    }

    /**
     * Gets the first `MediaStream` corresponding to the given type.
     * @param type The type of MediaStream.
     * @return the MediaStream instance or a null value.
     */
    @Nullable
    public MediaStream getFirst(@NonNull MediaStreamType type) {
        for (MediaStream stream : participant.streams()) {
            if (type.equals(stream.getType())) return stream;
        }
        return null;
    }

    /**
     * Inserts or updates a stream and manages this process.
     * @param mediaStream The MediaStream.
     * @return the participant's instance.
     */
    @NonNull
    public Participant insertOrUpdate(@NonNull MediaStream mediaStream) {
        List<MediaStream> streams = participant.streams();
        boolean found = false;
        for (MediaStream stream : streams) {
            if (stream.equals(mediaStream)) {
                found = true;
                break;
            }
        }

        if (!found) streams.add(mediaStream);
        else {
            streams.remove(mediaStream);
            streams.add(mediaStream);
        }
        return participant;
    }

    /**
     * Cleans the given Participant's streams.
     * @return the Participant instance
     */
    @NonNull
    public Participant removeAllStreams() {
        List<MediaStream> streams = participant.streams();
        streams.clear();
        return participant;
    }

    /**
     * Removes the given `mediaStream` from the Participant list.
     * @param mediaStream The MediaStream to remove.
     * @return the result of the erasure in the list.
     */
    @NonNull
    public boolean remove(@NonNull MediaStream mediaStream) {
        List<MediaStream> streams = participant.streams();
        return streams.remove(mediaStream);
    }

    /**
     * Removes the given `mediaStream` type from the Participant list.
     * @param type The type of MediaStream to erase from the streams.
     * @return if a given stream type was removed.
     */
    @NonNull
    public boolean remove(@NonNull MediaStreamType type) {
        List<MediaStream> streams = participant.streams();
        MediaStream toRemove = null;
        for (MediaStream stream : streams) {
            if (type.equals(stream.getType())) {
                toRemove = stream;
                break;
            }
        }

        if (null != toRemove) {
            streams.remove(toRemove);
            return true;
        }

        return false;
    }
}
