package com.voxeet.sdk.models.v1;

import android.support.annotation.Nullable;

import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

/**
 * The ConferenceParticipantStatus model describes possible statuses for SDK participants. It includes the [raw status](/documentation/sdk/reference/android/models/conferenceparticipantstatus#value) but also the [enum](/documentation/sdk/reference/android/models/conferenceparticipantstatus#fromId) and [enum string](/documentation/sdk/reference/android/models/conferenceparticipantstatus#fromString) representation of this status.
 *
 *  **The possible statuses (values):**
 * - IN_PROGRESS
 * - ON_AIR
 * - LATER
 * - DECLINE
 * - LEFT
 * - MISSED
 * - RESERVED
 * - CONNECTING
 * - INACTIVE
 * - WARNING
 * - ERROR
 */
@Annotate
public enum ConferenceParticipantStatus {
    IN_PROGRESS(0),
    ON_AIR(1),
    LATER(2),
    DECLINE(3),
    LEFT(4),
    MISSED(5),
    RESERVED(6),
    CONNECTING(7),
    INACTIVE(8),
    WARNING(9),
    ERROR(10);

    private final int value;

    @NoDocumentation
    ConferenceParticipantStatus(int value) {
        this.value = value;
    }

    /**
     * possibly invalid string to transform into a proper enum value
     * @return the value.
     */
    @NoDocumentation
    public int value() {
        return value;
    }

    /**
     * Represents ordinal numbers of the enum from any possible integer. It returns the proper corresponding enum value.
     *
     * @param value Any number, preferrably a valid ordinal.
     * @return an enum representation. The default value is IN_PROGRESS.
     */
    @NoDocumentation
    public static ConferenceParticipantStatus fromId(int value) {
        switch (value) {
            case 1:
                return ON_AIR;
            case 2:
                return LATER;
            case 3:
                return DECLINE;
            case 4:
                return LEFT;
            case 5:
                return MISSED;
            case 6:
                return RESERVED;
            case 7:
                return CONNECTING;
            case 8:
                return INACTIVE;
            case 9:
                return WARNING;
            case 10:
                return ERROR;
            default:
                return IN_PROGRESS;
        }
    }

    /**
     * Transform a possible integer value representing the enum ordinal to a possible english representation of the status. To be used for testing purposes.
     * @param value the value with defaulting to "Pending invite"
     * @return the English representation of the ordinal
     */
    @NoDocumentation
    public static String toString(int value) {
        switch (value) {
            case 1:
                return "On Air";
            case 2:
                return "Later";
            case 3:
                return "Declined";
            case 4:
                return "Left";
            case 5:
                return "Missed";
            case 6:
                return "Reserved";
            case 7:
                return "Connecting";
            case 8:
                return "Inactive";
            default:
                return "Pending invite";
        }
    }

    /**
     * Transforms strings.
     * @param status The possibly invalid string that can be transformed into a proper enum value.
     * @return an enum string representation. The default value is always IN_PROGRESS.
     */
    @NoDocumentation
    public static ConferenceParticipantStatus fromString(@Nullable String status) {
        if (null == status) return ConferenceParticipantStatus.ERROR;
        switch (status) {
            case "ON_AIR":
                return ON_AIR;
            case "LATER":
                return LATER;
            case "DECLINE":
                return DECLINE;
            case "LEFT":
                return LEFT;
            case "MISSED":
                return MISSED;
            case "RESERVED":
                return RESERVED;
            case "CONNECTING":
                return CONNECTING;
            case "INACTIVE":
                return INACTIVE;
            case "WARNING":
                return WARNING;
            case "ERROR":
                return ERROR;
            default:
                return IN_PROGRESS;
        }
    }
}
