package com.voxeet.sdk.media.audio;

import android.content.Context;
import android.media.Ringtone;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.audio.AudioRoute;
import com.voxeet.audio.AudioStackManager;
import com.voxeet.audio.listeners.IAudioRouteListener;
import com.voxeet.audio.utils.Constants;
import com.voxeet.sdk.events.sdk.AudioRouteChangeEvent;
import com.voxeet.sdk.media.MediaPowerManager;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.AudioType;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.SoundPool;
import com.voxeet.sdk.utils.Validate;

import org.greenrobot.eventbus.EventBus;

import java.util.HashMap;
import java.util.List;

@Annotate
public class SoundManager implements IAudioRouteListener {

    private AudioStackManager mAudioManager;
    private MediaPowerManager mMediaPowerManager;
    private Context mContext;
    private HashMap<Integer, SoundPool> _sound_pools;
    private HashMap<AudioType, String> mSounds;

    private boolean mEnable;

    private SoundManager() {
        disable();
    }

    @NoDocumentation
    public SoundManager(@NonNull Context context) {
        this();

        _sound_pools = new HashMap<>();

        mAudioManager = new AudioStackManager(context);
        mAudioManager.setMediaRoute();
        mAudioManager.configureVolumeStream(Constants.STREAM_VOICE_CALL,
                Constants.STREAM_MUSIC);
        mAudioManager.registerAudioRouteListener(this);
        mMediaPowerManager = new MediaPowerManager(context,
                currentRoute());

        mContext = context;

        mSounds = new HashMap<>();
        configure();
    }

    /**
     * Get the current available routes
     *
     * @return a non null route
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        return mAudioManager.availableRoutes();
    }

    /**
     * Get the current route calculated given the current internal state of the manager
     * @return the current calculated route
     */
    @NonNull
    public AudioRoute currentRoute() {
        return mAudioManager.outputRoute();
    }

    /**
     * Set the current route, forcing a new calculated one if compatible with the current system state
     * for instance, impossible to set bluetooth if no bluetooth headset is connected
     * but as soon as a bluetooth headset is connected, calling this method with bluetooth will work
     * etc...
     *
     * @param route a valid route to set if possible
     * @return the result of the internal call
     */
    public boolean setAudioRoute(@NonNull AudioRoute route) {
        Validate.runningOnUiThread();
        return mAudioManager.setOutputRoute(route);
    }

    /**
     * Get the Ringtone from the Android system
     * @return a Ringtone instance corresponding to the current system one
     */
    @Nullable
    public Ringtone getSystemRingtone() {
        return mAudioManager.getSystemRingtone();
    }

    /**
     * Check for headset connected to the device
     * @return the connection state
     */
    public boolean isBluetoothHeadsetConnected() {
        return mAudioManager.isBluetoothHeadsetConnected();
    }

    /**
     * Check for wired headset connected to the device
     * @return the connection state
     */
    public boolean isWiredHeadsetOn() {
        return mAudioManager.isWiredHeadsetOn();
    }

    /**
     * Change the output/internal speaker state
     * @param isSpeaker change to external speaker
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager setSpeakerMode(boolean isSpeaker) {
        mAudioManager.setSpeakerMode(isSpeaker);
        return this;
    }

    /**
     * Check for external speaker mode
     * @return the current speaker state
     */
    public boolean isSpeakerOn() {
        return AudioRoute.ROUTE_SPEAKER.equals(mAudioManager.outputRoute());
    }

    /**
     * Acquire locks for the internal manager
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager acquireLocks() {
        mMediaPowerManager.acquire();
        return this;
    }

    /**
     * Release locks for the internal manager
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager releaseLocks() {
        mMediaPowerManager.release();
        return this;
    }

    /**
     * Reset the internal Sound Type to the default one
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager resetDefaultSoundType() {
        mAudioManager.resetDefaultSoundType();
        return this;
    }

    /**
     * Call the internal UiSoundsStreamType accessible from Android
     *
     * For more information, go to the android-sdk-audio library
     * @return the current integer representing the UiSoundsStreamType
     */
    private int getUiSoundsStreamType() {
        return mAudioManager.getDefaultSoundStreamType();
    }

    /**
     * Force a given UI Volume Control Stream calling the system directly
     * @param type the given type
     * @return the current instance to chain calls
     */
    @NonNull
    private SoundManager forceVolumeControlStream(int type) {
        mAudioManager.forceVolumeControlStream(type);
        return this;
    }

    /**
     * Set the current mode to Media (not the call
     *
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager enableMedia() {
        mEnable = true;
        if (null != mAudioManager) {
            mAudioManager.setMediaRoute();
            mAudioManager.enable();
        }
        return this;
    }

    /**
     * Unset the Call mode to set the standard Media/System type
     * To call when not in conferences
     *
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager setMediaRoute() {
        mAudioManager.setMediaRoute();
        return this;
    }

    /**
     * Unset the Media mode to set the Call mode
     * To call when in a conference
     *
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager unsetMediaRoute() {
        mAudioManager.unsetMediaRoute();
        return this;
    }

    /**
     * Make the current app instance release the AudioFocus
     *
     * see android-sdk-audio
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager abandonAudioFocusRequest() {
        mAudioManager.abandonAudioFocus();
        return this;
    }

    /**
     * Make the current app instance acquire the AudioFocus
     *
     * see android-sdk-audio
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager requestAudioFocus() {
        mAudioManager.requestAudioFocus();
        return this;
    }

    /**
     * Calculate the output route and set the internal state accordingly
     *
     * Call this method each time it must be recalculated
     *
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager checkOutputRoute() {
        if(isEnabled()) {
            mAudioManager.checkOutputRoute();
        }
        return this;
    }

    private SoundPool getSoundPool(int soundMode) {
        SoundPool pool = _sound_pools.get(soundMode);
        if (pool == null) {
            pool = new SoundPool(mContext, soundMode, getVolume(soundMode));
            _sound_pools.put(soundMode, pool);
        }
        return pool;
    }

    private void configure() {
        //default disable the audio manager
        mAudioManager.disable();

        mSounds.put(AudioType.RING, "out.mp3");
        mSounds.put(AudioType.HANGUP, "leave.mp3");

        setSound(AudioType.RING, mSounds.get(AudioType.RING));
        setSound(AudioType.HANGUP, mSounds.get(AudioType.HANGUP));
    }

    /**
     * in the Voice Call Mode, set the AudioType to the given asset reference
     * @param type the AudioType to set in Stream Voice Call
     * @param assetName the name of the asset
     *
     * @return the current instance to chain calls
     */
    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return setSound(type, assetName, Constants.STREAM_VOICE_CALL);
    }

    /**
     * For a given soundMode and an AudioType, set the asset reference name
     * @param type the AudioType to set in Stream Voice Call
     * @param assetName the name of the asset
     * @param soundMode the sound mode for the system
     * @return the current instance to chain calls
     */
    public boolean setSound(@NonNull AudioType type, @NonNull String assetName, int soundMode) {
        return getSoundPool(soundMode).release(type).setShortResource(type, assetName);
    }

    /**
     * Play agiven AudioType in Stream Voice Call
     * @param type the AudioType
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager playSoundType(@NonNull AudioType type) {
        return playSoundType(type, Constants.STREAM_VOICE_CALL);
    }

    /**
     * Play a given AudioType in Stream Voice Call for a given sound mode
     * @param type the AudioType
     * @param soundMode the given sound mode
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager playSoundType(@NonNull AudioType type, int soundMode) {
        if (isEnabled()) {
            getSoundPool(soundMode).playShortResource(type, mSounds.get(type));
        }
        return this;
    }

    /**
     * Force a given sound AudioType to be played in Stream Voice Call
     * @param type the AudioType to force play
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager playSoundTypeForce(@NonNull AudioType type) {
        return playSoundType(type, Constants.STREAM_VOICE_CALL);
    }

    /**
     * Force a given sound AudioType to be played in the given sound mode
     * @param type the AudioType to foce play
     * @param soundMode the sound mode
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager playSoundTypeForce(@NonNull AudioType type, int soundMode) {
        getSoundPool(soundMode).playShortResource(type, mSounds.get(type));
        return this;
    }

    /**
     * Stop the sound played with the specified type for the stream voice call
     * @param audioType the AudioType to stop
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager stopSoundType(@NonNull AudioType audioType) {
        return stopSoundType(audioType, Constants.STREAM_VOICE_CALL);
    }

    /**
     * Stop the sound played with the specified type in the given mode
     * @param audioType the AudioType to stop
     * @param soundMode the sound mode in which it was started
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager stopSoundType(@NonNull AudioType audioType, int soundMode) {
        getSoundPool(soundMode).stop(audioType);
        return this;
    }

    /**
     * Stop the sound played in the given mode
     * @param soundMode the sound mode to stop
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager stop(int soundMode) {
        getSoundPool(soundMode).stop();
        return this;
    }

    /**
     * Stop the sound played in Stream Voice Call mode
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager stop() {
        stop(Constants.STREAM_VOICE_CALL);
        return this;
    }

    /**
     * Call this methid when a conference is in joining state
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager onConferencePreJoinedEvent() {
        mAudioManager.checkOutputRoute();
        return this;
    }

    /**
     * Call this method when a conference is destroyed
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager onConferenceDestroyedPush() {
        abandonAudioFocusRequest();
        return this;
    }

    /**
     * Enable this manager as well as enabling the internal one
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager enable() {
        mEnable = true;
        if (null != mAudioManager) {
            mAudioManager.enable();
        }
        return this;
    }

    /**
     * Disable this manager as well disabling the internal one
     * @return the current instance to chain calls
     */
    @NonNull
    public SoundManager disable() {
        mEnable = false;
        if (null != mAudioManager) {
            mAudioManager.disable();
        }
        return this;
    }

    private boolean isEnabled() {
        return mEnable;
    }

    private float getVolume(int soundMode) {
        switch (soundMode) {
            case Constants.STREAM_VOICE_CALL:
                return 0.1f;
            default:
                return 1.f;
        }
    }

    @NoDocumentation
    @Override
    public void onAudioRouteChanged() {
        EventBus.getDefault().post(new AudioRouteChangeEvent());
    }
}
