package com.voxeet.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.media.MediaSDK;
import com.voxeet.sdk.models.User;
import com.voxeet.sdk.models.v1.ConferenceUserStatus;
import com.voxeet.sdk.models.v1.OfferCandidate;
import com.voxeet.sdk.models.v1.OfferDescription;

import org.greenrobot.eventbus.EventBus;

import java.util.List;

import eu.codlab.simplepromise.Promise;

public abstract class AbstractPromiseable<PROMISE_RETURN_TYPE> {
    private final ConferenceSdkObservableProvider provider;
    private final ConferenceInformation information;
    private final EventBus eventBus;
    @NonNull
    private MediaDeviceService mediaDeviceService;
    @NonNull
    private ConferenceService parent;

    protected AbstractPromiseable(@NonNull ConferenceService parent,
                                  @NonNull MediaDeviceService mediaDeviceService,
                                  @NonNull ConferenceSdkObservableProvider provider,
                                  @Nullable ConferenceInformation information,
                                  @NonNull EventBus eventBus) {

        this.parent = parent;
        this.mediaDeviceService = mediaDeviceService;
        this.provider = provider;
        this.information = information;
        this.eventBus = eventBus;
    }

    @NonNull
    protected ConferenceService getParent() {
        return parent;
    }

    @NonNull
    protected MediaDeviceService getMediaService() {
        return mediaDeviceService;
    }

    @NonNull
    protected ConferenceSdkObservableProvider getProvider() {
        return provider;
    }

    /**
     * Get the behold ConferenceInformation
     *
     * in case of null, check for a possible current information afterward, in case of promise
     * while creating and joining but not in the conference
     *
     * @return an instance of ConferenceInformation or null
     */
    @Nullable
    protected ConferenceInformation getInformation() {
        if (null != information)
            return information;
        return parent.getCurrentConferenceInformation();
    }

    @NonNull
    protected EventBus getEventBus() {
        return eventBus;
    }

    protected boolean isInConference() {
        return getParent().isInConference();
    }

    protected boolean hasMedia() {
        return getMediaService().hasMedia();
    }

    @NonNull
    protected MediaSDK getMedia() {
        return getMediaService().getMedia();
    }

    @NonNull
    public abstract Promise<PROMISE_RETURN_TYPE> createPromise();

    protected void log(@NonNull String log) {
        if (!TextUtils.isEmpty(log)) {
            Log.d(getClass().getSimpleName(), log);
        }
    }

    protected void setIsInConference(boolean status) {
        getParent().setIsInConference(status);
    }

    protected Promise<Boolean> createVideoAnswer(final String userId,
                                                 final OfferDescription offerDescription,
                                                 final List<OfferCandidate> offerCandidates) {
        return getParent().createVideoAnswer(userId,
                offerDescription,
                offerCandidates);
    }

    protected boolean isSDK() {
        return getParent().isSDK();
    }

    protected User updateConferenceParticipants(String userId, ConferenceUserStatus status) {
        return getParent().updateConferenceParticipants(userId, status);
    }

    protected void closeMedia() {
        getParent().closeMedia();
    }

    protected void removeTimeoutCallbacks() {
        getParent().removeTimeoutCallbacks();
    }

    protected void sendTimeoutCallbacks() {
        getParent().sendTimeoutCallbacks();
    }

    protected VoxeetSdk getVoxeetSDK() {
        return getParent().getVoxeetSDK();
    }

    @NonNull
    protected ConferenceInformation createOrSetConferenceWithParams(@NonNull String conferenceId, @Nullable String conferenceAlias) {
        return getParent().createOrSetConferenceWithParams(conferenceId, conferenceAlias);
    }

    protected ConferenceInformation getConferenceInformation(@Nullable String conferenceId) {
        return getParent().getConferenceInformation(conferenceId);
    }

    protected void moveConference(@NonNull String conferenceId, @NonNull ConferenceInformation conferenceInformation) {
        getParent().moveConference(conferenceId, conferenceInformation);
    }

    protected void joinLock() {
        getParent().joinLock();
    }

    protected void joinUnlock() {
        getParent().joinUnlock();
    }

    protected Promise<Boolean> joinConference(@NonNull ConferenceInformation conference) {
        return getParent().joinConferenceInternalPackage(conference);
    }

    protected void setCurrentConferenceIfNotInPreviousConference(@NonNull ConferenceInformation conference) {
        getParent().setCurrentConferenceIfNotInPreviousConference(conference);
    }
}
